/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.document.ui.actions;

import java.io.File;
import java.io.FileNotFoundException;

import org.eclipse.swt.widgets.Display;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.document.ui.dialogs.FileDetailDialog;
import org.simantics.document.ui.graphfile.FileDocumentUtil;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * Action for importing files as document.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ImportDocumentWithDetail extends AddDocumentAction {
	

	public ImportDocumentWithDetail(ReadGraph graph, String relationUri) throws DatabaseException {
		super(graph, relationUri);
	}

	@Override
	public Runnable create(Object target) {
		if(!(target instanceof Resource))
			return null;
		final Resource resource = (Resource)target;
		return new Runnable() {
			
			@Override
			public void run() {
				final FileDetailDialog dialog = new FileDetailDialog(Display.getCurrent().getActiveShell(), resource);
				// TODO : is there any way to read file/executable bindings from OS?
				//        if is, use those extensions to filter this list.
				//        note: in windows using "reg query ..." to read bindings form registry would work.
				if (dialog.open() == FileDetailDialog.CANCEL) {
					dialog.getAnnotationConfigurator().dispose();
					return;
				}
				final String filename = dialog.getFileName();
				final String name = dialog.getName();
				if (filename == null) {
					dialog.getAnnotationConfigurator().dispose();
					return;
				}
				Simantics.getSession().asyncRequest(new WriteRequest() {
					@Override
					public void perform(WriteGraph graph) throws DatabaseException {
					    graph.markUndoPoint();
				        
					    Resource newDoc = doDocumentImport(graph, resource, filename, name);
						dialog.getAnnotationConfigurator().apply(graph,newDoc);
					}
				},new Callback<DatabaseException>() {				
					@Override
					public void run(DatabaseException parameter) {
						dialog.getAnnotationConfigurator().dispose();
						if (parameter != null) {
							ExceptionUtils.logAndShowError("Cannot import document.", parameter);
						}
						
					}
				});
			}
		};
	}
	
	public Resource doDocumentImport(WriteGraph graph, Resource target, String filename, String name) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(graph);
        Resource newDoc = FileDocumentUtil.importFileWithName(graph,filename);
        graph.claimLiteral(newDoc, l0.HasName, name);
        linkDocument(graph, target, newDoc);
        return newDoc;
	}
	
	public static Resource importDocumentWithDetailSCL(WriteGraph graph, Resource target, String filename) throws FileNotFoundException, DatabaseException {
	    File file = new File(filename);
	    if (!file.exists())
	        throw new FileNotFoundException("File not found - " + file.getAbsolutePath());
	    ImportDocumentWithDetail document = new ImportDocumentWithDetail(graph, "http://www.simantics.org/Layer0-1.1/ConsistsOf");
	    return document.doDocumentImport(graph, target, filename, file.getName());
	}

}
