package fi.vtt.simantics.procore.internal;

import org.simantics.db.Database;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.impl.graph.DelayedWriteGraph;
import org.simantics.db.impl.graph.WriteGraphImpl;
import org.simantics.db.procore.cluster.ClusterImpl;
import org.simantics.db.procore.cluster.TestCluster;
import org.simantics.db.procore.protocol.Constants;
import org.simantics.db.request.WriteTraits;
import org.simantics.db.service.ClusterUID;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.db.service.XSupport;

public class XSupportImpl implements XSupport {
    final private boolean DEBUG = false;
    final private SessionImplSocket session;

    XSupportImpl(SessionImplSocket session) {
        this.session = session;
    }

    @Override
    public void corruptPageTableAndDie() throws DatabaseException {
        execute("corruptPageTableAndDie");
    }

    @Override
    public void corruptCluster(Resource r) throws DatabaseException {
        ClusterImpl cluster = getCluster(r);
        long clusterId = Constants.NewClusterId;
        if (null == cluster.cc)
            cluster.cc = new ClusterChange(session.clusterStream, cluster);
        session.clusterStream.corruptCluster(cluster.cc, clusterId);
    }

    @Override
    public int corruptClusterTable(long clusterId)
    throws DatabaseException {
        return session.clusterTable.makeProxy(ClusterUID.make(0,666), clusterId).getClusterKey();
    }

    @Override
    public void flushCluster(Resource r) throws ServiceException {
        session.writeSupport.flushCluster(r);
    }

    @Override
    public void breakConnection() throws DatabaseException {
        throw new DatabaseException("XSupport.breakConnection not implemented.");
    }

    @Override
    public void setClusterStreamOff(boolean setOff)
    throws DatabaseException {
        session.clusterTranslator.setStreamOff(setOff);
    }

    @Override
    public int clearMetadataCache()
    throws DatabaseException {
        return session.graphSession.metadataCache.clear();
    }

    @Override
    public <T> void commitAndContinue(WriteOnlyGraph wograph, WriteTraits wtraits) {
    	if(wograph instanceof DelayedWriteGraph) {
    		DelayedWriteGraph dw = (DelayedWriteGraph)wograph;
    		dw.addCommitAndContinue();
    	} else {
    		session.state.commitAndContinue2(wograph.getService(WriteGraphImpl.class), session.clusterStream, wtraits);
    	}
    }

    @Override
    public boolean getImmutable(Resource resource)
    throws DatabaseException {
        if(!resource.isPersistent()) return false;
        ClusterImpl clusterImpl = getCluster(resource);
        return clusterImpl.getImmutable();
    }

    @Override
    public void setImmutable(Resource resource, boolean immutable)
    throws DatabaseException {
        ClusterImpl clusterImpl = getCluster(resource);
        clusterImpl.setImmutable(immutable, session.clusterTranslator);
    }
    private ClusterImpl getCluster(Resource resource)
    throws DatabaseException {
        if (null == resource)
            return null;
        int key = session.getService(SerialisationSupport.class).getTransientId(resource);
        return (ClusterImpl)session.clusterTranslator.getClusterByResourceKey(key);
    }

    @Override
    public void setServiceMode(boolean allow, boolean create) {
        if  (DEBUG) {
            new Throwable("printing stack trace").printStackTrace();
            System.out.println("XSupportImpl.setServiceMode allow=" + allow + " create=" + create + ", old mode=" + session.serviceMode);
        }
        int newServiceMode = (allow ? 1:0) + (create ? 2:0);
        if(newServiceMode != session.serviceMode) {
	        session.serviceMode = newServiceMode; 
	        session.writeSupport.flushCluster();
	        session.clusterSetsSupport.clear();
        }
    }
    @Override
    public Resource convertDelayedResourceToResource(Resource resource) {
        return DelayedWriteGraph.convertDelayedResource(resource);
    }
    @Override
    public String execute(String command)
    throws DatabaseException {
        boolean transaction = true;
        try {
            session.state.startReadTransaction(Integer.MIN_VALUE);
        } catch (Throwable t) {
            Logger.defaultLogError("Trying to muddle on.", t);
            transaction = false;
        }
        try {
            return session.graphSession.execute(command);
        } finally {
            if (transaction)
                session.state.stopReadTransaction();
        }
    }
    @Override
    public void testCluster(Session session)
    throws DatabaseException {
        TestCluster.test(session);
    }
    @Override
    public ClusterUID[] listClusters() throws DatabaseException {
        return session.graphSession.listClusters();
    }
    @Override
    public void deleteCluster(ClusterUID clusterUID) throws DatabaseException {
        ClusterImpl clusterImpl = session.clusterTable.getClusterByClusterUIDOrMakeProxy(clusterUID);
        //clusterImpl.setDeleted(true, session.clusterTranslator);
        session.clusterTranslator.setDeleted(clusterImpl, true);
    }
    @Override
    public void purge() throws DatabaseException {
        if (null == session)
            return;
        if (null == session.graphSession)
            return;
        Database db = session.graphSession.dbSession.getDatabase();
        db.purgeDatabase();
    }

    @Override
    public boolean rolledback() {
        return session.graphSession.rolledback();
    }
}
