/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on 13.10.2005
 * @author Toni Kalajainen 
 */
package org.simantics.utils.ui.dialogs;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * MultiLineInputDialog is a input dialog that can contain multiple input lines.
 * The minimum number of input lines is one. Each line must have message,
 * initial value and validator.
 * 
 */
public class MultiLineInputDialog extends InputDialog {

    private String values[];

    private IInputValidator validators[];

    private String messages[];

    /** Error message text boxes */
    private Text errorMessageTexts[];

    /** Text boxes */
    private Text texts[];

    private int count;
    
    /** Error message of first line */
//    private String firstErrorMessage;

    private static String INDEX_KEY = "index";

    /**
     * MultiLineInputDialog is a input dialog that can contain multiple input
     * lines. The minimum number of input lines is one. Each line must have
     * message, initial value and validator.
     * 
     * @param parentShell
     *            parent shell
     * @param dialogTitle
     *            window title
     * @param dialogMessage
     *            first line message
     * @param initialValue
     *            first line initial value
     * @param validator
     *            first line validator
     * @param moreLines
     *            more lines, each line must have {message, initialvalue,
     *            validator}
     */
    public MultiLineInputDialog(Shell parentShell, String dialogTitle, String dialogMessage, String initialValue, IInputValidator validator, Object... moreLines) {
        super(parentShell, dialogTitle, dialogMessage, initialValue, validator);

        // Parse more lines
        if (moreLines.length % 3 != 0)
            throw new IllegalArgumentException("bad argument count");

        count = moreLines.length / 3;
        values = new String[count];
        messages = new String[count];
        validators = new IInputValidator[count];
        errorMessageTexts = new Text[count];
        texts = new Text[count];
        for (int i = 0; i < count; i++) {
            values[i] = (String) moreLines[i * 3 + 1];
            messages[i] = (String) moreLines[i * 3];
            validators[i] = (IInputValidator) moreLines[i * 3 + 2];
        }
    }

    public String getValue(int index) {
        if (index == 0)
            return this.getValue();
        return values[index - 1];
    }
    
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);

        // Text box modify listener
        ModifyListener textModifyListener = new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                if (getOkButton()==null) return;
                
                int index = (Integer) e.widget.getData(INDEX_KEY);
                Text text = (Text) e.widget;
                IInputValidator validator = validators[index];
                String errorMessage = null;
                if (validator != null)
                    errorMessage = validator.isValid(text.getText());

                errorMessageTexts[index].setText(errorMessage == null ? "" : errorMessage);
                errorMessageTexts[index].getParent().update();

                                
                boolean ok = MultiLineInputDialog.this.getValidator().isValid(MultiLineInputDialog.this.getText().getText())==null;
                for (int i = 0; i < count; i++)
                    ok = ok & (errorMessageTexts[i].getText().equals(""));
                    
                getOkButton().setEnabled(ok);     
            }
        };

        // Add some more lines, text boxes
        for (int i = 0; i < count; i++) {
            String message = messages[i];
            if (message != null) {
                Label label = new Label(composite, SWT.WRAP);
                label.setText(message);
                GridData data = new GridData(GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL | GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_CENTER);
                data.widthHint = convertHorizontalDLUsToPixels(IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH);
                label.setLayoutData(data);
                label.setFont(parent.getFont());
            }
            texts[i] = new Text(composite, SWT.SINGLE | SWT.BORDER);
            texts[i].setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL));
            texts[i].setData(INDEX_KEY, new Integer(i));
            texts[i].addModifyListener(textModifyListener);
            errorMessageTexts[i] = new Text(composite, SWT.READ_ONLY);
            errorMessageTexts[i].setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL));
            errorMessageTexts[i].setBackground(errorMessageTexts[i].getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
            errorMessageTexts[i].setForeground(errorMessageTexts[i].getDisplay().getSystemColor(SWT.COLOR_RED));
        }        
        
        for (int i=0; i<count; i++) {
            texts[i].setText(values[i]);            
        }

        // Fix tab order
        ArrayList<Control> tabs = new ArrayList<Control>();
        tabs.add(getText());
        for (Text t : texts)
            tabs.add(t);
        //tabs.add(getButton(IDialogConstants.CANCEL_ID));
        //tabs.add(getButton(IDialogConstants.OK_ID));
        composite.setTabList(tabs.toArray(new Control[0]));
        
        applyDialogFont(composite);
        return composite;
    }

    @Override
    protected void createButtonsForButtonBar(Composite parent) {
        super.createButtonsForButtonBar(parent);
        boolean ok = getValidator().isValid(getText().getText())==null;
        for (int i = 0; i < count; i++) {
            String txt = texts[i].getText();
            if (txt==null) txt="";
            String error = validators[i].isValid(txt);
            boolean valid = error==null;
            ok = ok & valid;
        }
        getOkButton().setEnabled(ok);
    }
    
    protected boolean allFieldsValid() {
        if (!(getValidator().isValid(getText().getText())==null)) return false;
        
        for (int i = 0; i < count; i++) {
        	
            // Eclipse 3.2R3 seems to initialize code in new order
            // Some hubba here to omit initialization problems
        	if (texts[i]==null) return true;
        	
            String txt = texts[i].getText();
            if (txt==null) txt="";
            boolean valid = validators[i].isValid(txt)==null;
            if (!valid) return false;
        }
        return true;
    }
    
    @Override
    public void setErrorMessage(String errorMessage) {
//        this.firstErrorMessage = errorMessage;
        super.setErrorMessage(errorMessage);
        
        // Eclipse 3.2R3 seems to initialize code in new order
        // Some hubba here to omit initialization problems
        if (getOkButton()==null) return;
        
        getOkButton().setEnabled(allFieldsValid());
    }

    @Override
    protected void buttonPressed(int buttonId) {
        if (buttonId == IDialogConstants.OK_ID) {
            for (int i=0; i<values.length; i++)
                values[i] = texts[i].getText();
        } else {
            for (int i=0; i<values.length; i++)
                values[i] = null;
        }
        super.buttonPressed(buttonId);
    }    
    
}
