package org.simantics.selectionview;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.utils.Functions;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.request.VariableIndexRoot;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.ui.selection.WorkbenchSelectionUtils;
import org.simantics.utils.ui.ISelectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardSelectionProcessor implements SelectionProcessor<Object, ReadGraph> {
    private static final Logger LOGGER = LoggerFactory.getLogger(StandardSelectionProcessor.class);

	private static Resource getIndexRoot(ReadGraph graph, Object selection) throws DatabaseException {
		
		Variable variable = WorkbenchSelectionUtils.getPossibleVariable(graph, selection);
		if (variable != null) {
			return graph.sync(new VariableIndexRoot(variable));
		}
		
		variable = ISelectionUtils.filterSingleSelection(selection, Variable.class);
		if (variable != null) {
			return graph.sync(new VariableIndexRoot(variable));
		}
		
		Resource indexRoot = ISelectionUtils.getSinglePossibleKey(selection, SelectionHints.KEY_MODEL, Resource.class, false);
		if (indexRoot == null) {
			Resource r = ISelectionUtils.getSinglePossibleKey(selection, SelectionHints.KEY_MAIN, Resource.class);
			if (r != null)
			    indexRoot = graph.sync(new PossibleIndexRoot(r));
		}

		return indexRoot != null && graph.isInstanceOf(indexRoot, Layer0.getInstance(graph).IndexRoot) ? indexRoot : null;
	}

	@Override
	public Collection<?> process(Object selection, ReadGraph graph) {
		return processStatic(selection, graph);
	}
	
	public static Collection<?> processStatic(Object selection, ReadGraph graph) {
		
		try {
			
			SelectionViewResources SEL = SelectionViewResources.getInstance(graph);
			Instances contributionFinder = graph.adapt(SEL.TabContribution, Instances.class);
			Instances transformationFinder = graph.adapt(SEL.SelectionTransformation, Instances.class);
			
			ArrayList<TabContribution<?>> contributions = new ArrayList<TabContribution<?>>();
			
			Resource index = getIndexRoot(graph, selection);
			if(index != null) {
			
				for(Resource r : contributionFinder.find(graph, index)) {
					TabContribution<?> contribution = graph.adapt(r, TabContribution.class);
					contributions.add(contribution);
					if(DebugPolicy.DEBUG) System.err.println("-contribution " + graph.getURI(r));
				}

				ArrayList<Resource> transforms = new ArrayList<Resource>();
				transforms.addAll(transformationFinder.find(graph, index));
				if(DebugPolicy.DEBUG) {
					for(Resource r : transforms)
						System.err.println("-transform " + NameUtils.getSafeLabel(graph, r));
				}
				
				HashSet<Object> inputs = new HashSet<Object>();
				HashSet<Object> newInputs = new HashSet<Object>();
				inputs.add(selection);
				boolean changed = true;
				while(changed) {
					for(Resource tr : transforms) {
						for(Object input : inputs) {
							try {
								Object newInput = Functions.exec(graph, tr, graph, input);
								if(newInput != null)
									newInputs.add(newInput);
							} catch (DatabaseException e) {
							}
						}
					}
					changed = inputs.addAll(newInputs);
					newInputs.clear();
				}
				

				if(DebugPolicy.DEBUG) {
					for(Object o : inputs)
						System.err.println("-input " + inputName(graph, o));
				}

				Set<ComparableTabContributor> result = new HashSet<ComparableTabContributor>();
				for(TabContribution c : contributions) {
					for(Object input : inputs) {
						try {
							if(DebugPolicy.DEBUG) {
								System.err.println("contribution: " + c);
								System.err.println("->input: " + input);
								System.err.println("->input name: " + inputName(graph, input));
							}
							if (c.accept(graph, input))
								c.contribute(graph, input, result);
						} catch (Exception e) {
							LOGGER.error("Selection view tab contribution failed.", e);
						}
					}
				}
				
				if(DebugPolicy.DEBUG) {
					System.err.println("result: " + result);
				}
				
				return result;
			
			}
			
		
		} catch (DatabaseException e) {
			
			e.printStackTrace();
			
		}
        
		return Collections.emptyList();
		
	}
	
	private static String inputName(ReadGraph graph, Object o) throws DatabaseException {
		if(o instanceof Resource) {
			return "Resource: " + NameUtils.getURIOrSafeNameInternal(graph, (Resource)o);
		} else if(o instanceof Variable) {
			return "Variable: " + ((Variable)o).getURI(graph) + " " + NameUtils.getSafeLabel(graph, (Resource)((Variable)o).getPossiblePropertyValue(graph, Variables.TYPE));
        } else if(o instanceof SelectionInput) {
            return "SelectionInput: " + o.toString();
		} else {
			return "Object(" + o.getClass().getSimpleName() + "): " + o.toString();
		}
	}

}
