/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.selectionview;

import java.util.concurrent.atomic.AtomicBoolean;

import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.db.common.procedure.adapter.ListenerSupport;
import org.simantics.db.management.ISessionContext;

/**
 * Override {@link #createControls(Composite, ISessionContext)} to create
 * controls for your own tab.
 * 
 * <p>
 * This class implements {@link ListenerSupport} to help in dealing with graph
 * request listeners.
 * 
 * <p>
 * This adapter has the following default implementations for IPropertyTab:
 * <ul>
 * <li>{@link #getSelectionProvider()} - returns null, i.e. does not provide any
 * selection to other UI/Workbench parts.</li>
 * <li>{@link #requestFocus()} - calls {@link Control#setFocus()} for
 * {@link #getControl()} if a control exists</li>
 * <li>{@link #isDisposed()} - returns <code>false</code> by default and
 * <code>true</code> if {@link #setDisposed()} has been called</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 * 
 * @see PropertyTabContributorImpl
 */
public abstract class PropertyTabAdapter implements IPropertyTab2, ListenerSupport {

    protected IWorkbenchSite    site;

    private final AtomicBoolean disposed = new AtomicBoolean();

    public PropertyTabAdapter(IWorkbenchSite site) {
        this.site = site;
    }


    /**
     * Override this implementation and call super.createControl(parent) as the
     * last thing in your property table.
     * 
     * @see org.simantics.selectionview.IPropertyTab#createControl(org.eclipse.swt.widgets.Composite)
     */
    @Override
    public final void createControl(Composite parent, ISessionContext context) {
        createControls(parent, context);
        Control control = getControl();

        if (control == null || control.isDisposed())
            return;

        control.addDisposeListener(new DisposeListener() {
            @Override
            public void widgetDisposed(DisposeEvent e) {
                setDisposed();
            }
        });
    }

    /**
     * Override this implementation to create the tab's controls.
     * 
     * @see #createControl(Composite, ISessionContext)
     */
    public void createControls(Composite parent, ISessionContext context) {
        Control control = getControl();
        if (control == null || control.isDisposed())
            return;

        control.addDisposeListener(new DisposeListener() {
            @Override
            public void widgetDisposed(DisposeEvent e) {
                setDisposed();
            }
        });
    }

    @Override
    public void dispose() {
    }

    /**
     * @return <code>true</code> if tab was marked disposed, <code>false</code>
     *         if it was already marked disposed
     */
    protected boolean setDisposed() {
        return disposed.compareAndSet(false, true);
    }

    @Override
    public boolean isDisposed() {
        return disposed.get();
    }

    @Override
    public ISelectionProvider getSelectionProvider() {
        return null;
    }

    @Override
    public void requestFocus() {
        Control control = getControl();
        if (control == null || control.isDisposed())
            return;

        control.setFocus();
    }

    @Override
    public void exception(Throwable t) {
        ErrorLogger.defaultLogError("PropertyTabAdapter received unexpected exception.", t);
    }

}
