/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.workbench.search;

import org.simantics.NameLabelUtil;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.ServiceLocator;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.SerialisationSupport;

/**
 * @author Tuukka Lehtonen
 * @author Antti Villberg
 */
public class NamedResource {

    private final Resource resource;
    private final String resourceText;
    private final String name;
    private final String uri;

    public static NamedResource of(ReadGraph graph, Resource resource) throws DatabaseException {
        return of(graph, resource, NameLabelUtil.modalName(graph, resource));
    }

    public static NamedResource of(ReadGraph graph, Resource resource, String name) throws DatabaseException {
        return of(graph, resource, name, graph.getPossibleURI(resource));
    }

    public static NamedResource of(ServiceLocator locator, Resource resource, String name, String uri) throws DatabaseException {
        SerialisationSupport ss = locator.getService(SerialisationSupport.class);
        return new NamedResource(resource, "" + ss.getRandomAccessId(resource), name, uri);
    }

    public NamedResource(String resourceText, String name, String uri) {
        this(null, resourceText, name, uri);
    }

    public NamedResource(Resource resource, String resourceText, String name, String uri) {
        this.resource = resource;
        this.resourceText = resourceText;
        this.name = name;
        this.uri = uri;
    }

    public Resource getResourceDB() {
        if (resource == null)
            throw new NullPointerException("null Resource in NamedResource(" + resourceText + ", " + name + ", " + uri + ")");
        return resource;
    }

    public Resource getPossibleResourceDB() {
        return resource;
    }

    public String getResource() {
        return resourceText;
    }

    public String getName() {
        return name;
    }

    public String getUri() {
        return uri;
    }

}
