/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - issue #4384
 *******************************************************************************/
package org.simantics.ui.workbench.e4;

import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.ListenerAdapter;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.TernaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.utils.datastructures.map.Tuple;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.SWTUtils;

/**
 * A helper class for adding input validation and reaction to input invalidation
 * to an E4 (editor) part that uses Simantics database {@link Resource}s as its
 * input.
 * 
 * <p>
 * Only useful with E4ResourceEditorBase extending part implementations.
 * 
 * @author Tuukka Lehtonen
 */
public class E4ResourceEditorSupport {

    private E4ResourceEditorBase editor;

    private ParametrizedRead<IResourceEditorInput, Boolean> inputValidator;

    private InputListener inputListener;

    public E4ResourceEditorSupport(E4ResourceEditorBase editor) {
        this(editor, null);
    }

    public E4ResourceEditorSupport(E4ResourceEditorBase editor, ParametrizedRead<IResourceEditorInput, Boolean> inputValidator) {
        this.editor = editor;
        this.inputValidator = inputValidator;
    }

    public void dispose() {
        deactivateValidation();
        inputValidator = null;
        editor = null;
    }

    protected boolean isDisposed() {
        return editor == null;
    }

    public synchronized void activateValidation() {
        if (isDisposed())
            throw new IllegalStateException(this + " is disposed");
        if (inputListener != null)
            return;

        E4ResourceEditorBase e = editor;
        inputListener = new InputListener();
        Simantics.getSession().asyncRequest(new InputEvaluator(e.getPart().getElementId(), e.getResourceInput(), inputValidator), inputListener);
    }

    public synchronized void deactivateValidation() {
        if (isDisposed())
            throw new IllegalStateException(this + " is disposed");
        if (inputListener == null)
            return;
        inputListener.dispose();
    }

    static enum InputState {
        VALID,
        INVALID,
        NON_EXISTENT;

        public static InputState parse(boolean exists, boolean valid) {
            if (!exists)
                return NON_EXISTENT;
            return valid ? VALID : INVALID;
        }
    }

    static class Evaluation extends Tuple {
        public Evaluation(String editorElementId, InputState state) {
            super(editorElementId, state);
        }

        public String getEditorElementId() {
            return (String) getField(0);
        }

        public InputState getInputState() {
            return (InputState) getField(1);
        }
    }

    public static class InputEvaluator extends TernaryRead<String, IResourceEditorInput, ParametrizedRead<IResourceEditorInput, Boolean>, Evaluation> {

        public InputEvaluator(String editorElementId, IResourceEditorInput input, ParametrizedRead<IResourceEditorInput, Boolean> inputValidator) {
            super(editorElementId, input, inputValidator);
        }

        @Override
        public Evaluation perform(ReadGraph graph) throws DatabaseException {
            //System.out.println(this + ": checking input " + parameter + ", " + parameter2);
            IResourceEditorInput i = parameter2;
            boolean exists = i.exists(graph);
            boolean valid = exists && parameter3 != null
                    ? graph.syncRequest(parameter3.get(i))
                    : exists;
            InputState state = InputState.parse(exists, valid);
            Evaluation eval = new Evaluation(parameter, state);
            //System.out.println(this + ": validation evaluation: " + eval);
            return eval;
        }
    }

    private static class InputListener extends ListenerAdapter<Evaluation> {

        private boolean disposed = false;

        public void dispose() {
            disposed = true;
        }

        @Override
        public void execute(Evaluation evaluation) {
            //System.out.println("InputListener: " + evaluation);
            switch (evaluation.getInputState()) {
                case VALID:
                    break;

                case INVALID:
                case NON_EXISTENT:
                    scheduleEditorClose(evaluation.getEditorElementId());
                    break;
            }
        }

        @Override
        public void exception(Throwable t) {
            ExceptionUtils.logError("E4ResourceEditorSupport.InputListener received an unexpected exception.", t);
        }

        @Override
        public boolean isDisposed() {
            return disposed;
        }
    }

    private static void scheduleEditorClose(String editorElementId) {
        SWTUtils.asyncExec(PlatformUI.getWorkbench().getDisplay(), () -> {
            E4WorkbenchUtils.closeEditor(editorElementId);
        });
    }

}
