/*******************************************************************************
' * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.trend.util;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.simantics.g2d.canvas.impl.CanvasContext;
import org.simantics.history.HistoryManager;
import org.simantics.trend.TrendInitializer;
import org.simantics.trend.configuration.TrendSpec;
import org.simantics.trend.configuration.Viewport;
import org.simantics.trend.impl.MilestoneSpec;
import org.simantics.trend.impl.TrendNode;
import org.simantics.utils.page.MarginUtils;
import org.simantics.utils.page.PageDesc;
import org.simantics.utils.page.PageOrientation;
import org.simantics.utils.threads.AWTThread;
import org.simantics.utils.threads.CurrentThread;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.FontFactory;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.DefaultFontMapper;
import com.lowagie.text.pdf.FontMapper;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfWriter;

public class PrintUtil {

	PageDesc pageDesc;
    FontMapper mapper;
    List<PrintPage> pages = new ArrayList<PrintPage>();
	
    public PrintUtil()
    {
    	pageDesc = new PageDesc("A4", PageOrientation.Landscape, 210, 297).withMargins( MarginUtils.MARGINS_10mm );
        mapper = new DefaultFontMapper();
        FontFactory.registerDirectories();
    }
    
    public void addTrendPage(TrendSpec trendSpec, HistoryManager historian, MilestoneSpec milestones, Viewport viewport)
    {
    	pages.add( TrendPage.addTrendPage(trendSpec, historian, milestones, viewport) );
    }
    
    public void addTrendPage(TrendNode srcTrend) {
    	pages.add( TrendPage.createFrom(srcTrend));
    }
    
	public void setPageDesc(PageDesc pageDesc)
	{
		this.pageDesc = pageDesc;
    }
	
	public void print() throws PrinterException {
		PrinterJob printJob = PrinterJob.getPrinterJob();
		printJob.setPrintable( new Printable() {
			@Override
			public int print(Graphics graphics, PageFormat pageFormat, int pageIndex) throws PrinterException {
				if (graphics instanceof Graphics2D == false) {
					throw new PrinterException("Printer context does not support Graphics2D");
				}
				if (pageIndex < pages.size()) {
					Graphics2D g2d = (Graphics2D) graphics;

					g2d.translate(pageFormat.getImageableX(), pageFormat.getImageableY());			
					
					Rectangle2D rect = new Rectangle2D.Double(0, 0, pageFormat.getImageableWidth(), pageFormat.getImageableHeight());
					//g2d.setClip( rect );
					System.out.println("Printing page "+pageIndex+", pageFormat="+pageFormat.getImageableX()+", "+pageFormat.getImageableY()+", "+pageFormat.getImageableWidth()+", "+pageFormat.getImageableHeight());
					PrintPage trend = pages.get(pageIndex);
					trend.print(g2d, rect);
					
					return PAGE_EXISTS;
				}
				return NO_SUCH_PAGE;
			}} );
		printJob.printDialog();
		printJob.print();
	}	
	
	public void printPdf(File file) throws IOException
	{		
		Document document = null;
		FileOutputStream fos = null;
		Graphics2D g2 = null;
		try {
			// Page area
			Rectangle2D.Float pageArea = new Rectangle2D.Float();
			pageDesc.getPageRectangle( pageArea );

			// Content area in page
			Rectangle2D.Float contentArea = new Rectangle2D.Float();
			pageDesc.getMarginsRectangle( contentArea );
	
			// step 1: creation of a document-object
			Rectangle pageArea2 = new Rectangle( 
					(float) PageDesc.toPoints(pageArea.getMinX()), 
					(float) PageDesc.toPoints(pageArea.getMinY()), 
					(float) PageDesc.toPoints(pageArea.getWidth()), 
					(float) PageDesc.toPoints(pageArea.getHeight()) );
			document = new Document( pageArea2 );
          
			// step 2: creation of the writer
			if ( !file.exists() ) file.createNewFile();
			fos = new FileOutputStream( file, false );
			PdfWriter writer;
			try {
				writer = PdfWriter.getInstance(document, fos);
			} catch (DocumentException e) {
				throw new IOException(e);
			}
            
			// step 3: we open the document
			document.open();

			// step 4:
			for ( PrintPage page : pages ) 
			{
				if (!document.newPage()) break;
				PdfContentByte cb = writer.getDirectContent();
				g2 = cb.createGraphics( pageArea2.getWidth(), pageArea2.getHeight(), mapper);
				
				Rectangle2D contentArea2 = new Rectangle2D.Float( 
						(float) PageDesc.toPoints(contentArea.getMinX()), 
						(float) PageDesc.toPoints(contentArea.getMinY()), 
						(float) PageDesc.toPoints(contentArea.getWidth()), 
						(float) PageDesc.toPoints(contentArea.getHeight()) );
	
				page.print(g2, contentArea2);
			}
			
        } finally {
        	try {
        		if (g2 != null) g2.dispose();
	        	if (document != null) document.close();
	        	if (fos != null) fos.close();
        	} catch (Exception e) {}
        }
	}

	interface PrintPage {
		void print(Graphics2D g2d, Rectangle2D pageArea);
	}
	
	static class TrendPage implements PrintPage
	{
		boolean autoscale;
		TrendNode trend;
		
		public static TrendPage createFrom(TrendNode srcTrend)
		{
	    	TrendPage page = new TrendPage();
	        CanvasContext ctx = TrendInitializer.createDefaultCanvas(CurrentThread.getThreadAccess(), srcTrend.historian, srcTrend.collector, null, srcTrend.getTrendSpec());
	        TrendNode trend = TrendInitializer.getTrendNode( ctx );        
	        trend.setViewport( srcTrend.getViewport() );
	        page.autoscale = false;
	        if (srcTrend.milestones != null) trend.setMilestones( srcTrend.milestones );
	    	trend.printing = true;
	    	page.trend = trend;
	    	return page;
		}
		
	    public static TrendPage addTrendPage(TrendSpec trendSpec, HistoryManager historian, MilestoneSpec milestones, Viewport viewport)
	    {
	    	TrendPage page = new TrendPage();
	        CanvasContext ctx = TrendInitializer.createDefaultCanvas(AWTThread.getThreadAccess(), historian, null, null, trendSpec);
	        TrendNode trend = TrendInitializer.getTrendNode( ctx );    	
	    	trend.printing = true;
	    	if (milestones!=null) {
	    		trend.setMilestones(milestones);
	    	}
	    	if ( viewport!=null ) {
	    		trend.setViewport(viewport);
	    		page.autoscale = false;
	    	} else {
	    		page.autoscale = true;
	    	}
	    	page.trend = trend;
	    	return page;
	    }
				
		public void print(Graphics2D g2d, Rectangle2D pageArea)
		{
			Rectangle2D contentArea2 = new Rectangle2D.Float( 
					(float) PageDesc.toPoints(pageArea.getMinX()), 
					(float) PageDesc.toPoints(pageArea.getMinY()), 
					(float) PageDesc.toPoints(pageArea.getWidth()), 
					(float) PageDesc.toPoints(pageArea.getHeight()) );

			java.awt.Rectangle r2 = g2d.getClipBounds();
			java.awt.Rectangle r = g2d.getClipBounds();
			r.width *= 2;
			r.height *= 2;
			g2d.scale(0.5, 0.5);
//			r = g2d.getClipBounds();
//			System.out.println(r);
			g2d.setClip(r);
			try {
				trend.setSize(r.width, r.height);
//				trend.setSize(contentArea2.getWidth()*2, contentArea2.getHeight()*2);
//				g2d.setClip( contentArea2 );
				//trend.setSize(contentArea2.getWidth(), contentArea2.getHeight());
				if ( autoscale ) trend.zoomOut();
				trend.layout();
				trend.render( g2d );
			} finally {
				g2d.scale(2, 2);
				g2d.setClip(r2);
			}
		}
	}
	
}
