/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.trend.util;

import java.awt.Graphics2D;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

/**
 * DeviationBuilder builds a vertical string of rectangles as an AWT Shape.
 * Rectangles are added to the shape, each added to the right hand.
 * 
 * @author toni.kalajainen
 */
public class KvikDeviationBuilder {

	List<Rectangle2D.Double> rects = new ArrayList<Rectangle2D.Double>();
	int rectCount = 0;
    
    public KvikDeviationBuilder() {
    	reset();
    }
    
    public void addRectangle(double x1, double x2, double y1, double y2)
    {
    	if (y1>y2) {
    		double yy = y1;
    		y1 = y2;
    		y2 = yy;
    	}
    	
    	if (y2 == Double.NaN || y1 == Double.NaN || y2 == y1) {
    		return;
    	}
    	
    	Rectangle2D.Double lastRect = lastRect();
    	if ( lastRect!=null ) {
	    	double lx1 = lastRect.x;
	    	double lx2 = lastRect.x + lastRect.width;
	    	double ly1 = lastRect.y;
	    	double ly2 = lastRect.y + lastRect.height;
	    	
	    	// Extends min-max of prev rectangle
	    	if (x1==lx1 && x2==lx2) {
	    		if (y2<ly1) lastRect.y = y2;
	    		if (y1>ly2) lastRect.height = y1 - lastRect.y;
	    		return;
	    	}
	    	
	    	// Change the last values    	
	    	if (x1==lx2 && y1==ly1 && y2==ly2) {
	    		lastRect.width = x2 - lastRect.x;
	    		return;
	    	}
    	}

    	// Add new region
		Rectangle2D.Double r = newRect();
		r.x=x1;
		r.y=y1;
		r.width=x2-x1;
		r.height=y2-y1;
    }

    /**
     * Extend the previous rectangle to the left
     * 
     * @param x1
     * @param x2
     * @param low
     * @param high
     */
    public void appendRectangle(double xx, double low, double high)
    {
    	Rectangle2D.Double lastRect = lastRect();
    	if ( lastRect!=null ) {
    		addRectangle(lastRect.x+lastRect.width, xx, low, high);
    	} else {
    		addRectangle(xx, xx, low, high);
    	}
    }

    public void extendRectangle(double toX)
    {
    	Rectangle2D.Double lastRect = lastRect();
    	if ( lastRect == null ) return;
    	lastRect.width = toX - lastRect.x;
    }
    
    public boolean isEmpty() {
    	return rectCount == 0;
    }
    
    public void reset() {
    	rectCount = 0;
    }
    
    /**
     * Draw the shape with lines. This is the fastest for screen but not good for printer drawing
     * 
     * @param g
     */
    public void drawLines(Graphics2D g) {
    	Line2D.Double line = new Line2D.Double();
    	for (int i=0; i<rectCount; i++) {
    		Rectangle2D.Double r = rects.get(i);
    		int x1 = (int) Math.round(r.x);
    		int x2 = (int) Math.round(r.x);
    		line.y1 = r.y;
    		line.y2 = r.y + r.height;
    		for (int x = x1; x<=x2; x++) {
    			line.x1 = line.x2 = x;
    			g.draw(line);
    		}
    	}
    }
    
    /**
     * Draw the shape with rectangles. Correct, not the fastest
     */
    public void drawRectangles(Graphics2D g) {
    	for (int i=0; i<rectCount; i++) {
    		Rectangle2D.Double r = rects.get(i);
    		g.fill(r);
    	}
    }
    
    /**
     * Create new rectangle and add to the list. 
     * @return new rectangle
     */
    Rectangle2D.Double newRect() {
    	if (rectCount < rects.size()) {
    		return rects.get( rectCount++ );
    	}
    	Rectangle2D.Double rect = new Rectangle2D.Double();
    	rectCount++;
    	rects.add(rect);
    	return rect;
    }
    
    /**
     * Get the last rectangle
     * @return
     */
    Rectangle2D.Double lastRect() {
    	if ( rectCount == 0 ) return null;
    	return rects.get( rectCount-1 );
    }
    

}

