package org.simantics.scl.compiler.elaboration.expressions;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

import org.simantics.scl.compiler.compilation.CompilationContext;
import org.simantics.scl.compiler.elaboration.chr.CHRRuleset;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.java.SafeExternalConstant;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.exceptions.MatchException;

public class ECHRRuleset extends Expression implements SafeExternalConstant, Externalizable {

	transient public CHRRuleset ruleset;
	transient public String rulesetKey; 
    transient public Expression in;

    public ECHRRuleset() {
    }

    public ECHRRuleset(CHRRuleset ruleset, Expression in) {
        this.ruleset = ruleset;
        this.in = in;
    }
    
    private void ensureRuleset() {
    	if(ruleset == null) {
        	throw new IllegalStateException("Ruleset is null");
    	}
    }

    @Override
    protected void updateType() throws MatchException {
        setType(in.getType());
    }

    @Override
    public IVal toVal(CompilationContext context, CodeWriter w) {
    	ensureRuleset();
        ruleset.generateCode(w);
        return in.toVal(context, w);
    }

    @Override
    public Expression resolve(TranslationContext context) {
        if(context.currentRuleset != null) {
            context.getErrorLog().log(location, "Current version of SCL compiler does not support nested rulesets.");
            return this;
        }
    	ensureRuleset();
        context.currentRuleset = ruleset;
        
        context.pushFrame();
        context.pushCHRConstraintFrame();
        ruleset.resolve(context);
        in = in.resolve(context);
        context.popCHRConstraintFrame(ruleset);
        context.popFrame();
        
        context.currentRuleset = null;
        
        return this;
    }
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            this.location = loc;
        	ensureRuleset();
            ruleset.setLocationDeep(loc);
            in.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public Expression inferType(TypingContext context) {
    	ensureRuleset();
        ruleset.checkType(context);
        in = in.inferType(context);
        return this;
    }
    
    @Override
    public Expression checkBasicType(TypingContext context, Type requiredType) {
    	ensureRuleset();
        ruleset.checkType(context);
        in = in.checkType(context, requiredType);
        return this;
    }
    
    @Override
    public Expression simplify(SimplificationContext context) {
    	ensureRuleset();
        ruleset.simplifyAndCompileIfNeeded(context);
        in = in.simplify(context);
        return this;
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext context) {
        throw new UnsupportedOperationException();
    }

	@Override
	public void writeExternal(ObjectOutput out) throws IOException {
		out.writeUTF(ruleset.runtimeRulesetClassName);
	}

	@Override
	public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
		rulesetKey = in.readUTF();
	}

}
