/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.utils;

import java.awt.Color;

/**
 * @author Tuukka Lehtonen
 */
public class ColorUtil {

    /**
     * Blend two colors (AWT) by linearly interpolating between the specified
     * ratio (0-1). 0 means <code>src</code> color only and 1 means
     * <code>dst</code> color only.
     * 
     * @param src
     * @param dst
     * @param ratio a value between [0,1]
     * @return a new instance of Color, containing the interpolated result.
     *         Remember to dispose of this color when its no longer needed!
     */
    public static Color blend(Color src, Color dst, double ratio) {
        if (ratio < 0 || ratio > 1)
            throw new IllegalArgumentException("expected t in [0,1], got t " + ratio);

        double r1 = src.getRed();
        double g1 = src.getGreen();
        double b1 = src.getBlue();
        double a1 = src.getAlpha();
        double r2 = dst.getRed();
        double g2 = dst.getGreen();
        double b2 = dst.getBlue();
        double a2 = dst.getAlpha();

        double r = r1 + (r2 - r1) * ratio;
        double g = g1 + (g2 - g1) * ratio;
        double b = b1 + (b2 - b1) * ratio;
        double a = a1 + (a2 - a1) * ratio;

        return new Color((int) r, (int) g, (int) b, (int) a);
    }

    /**
     * @param c
     * @param alpha
     * @return
     */
    public static Color withAlpha(Color c, int alpha) {
        return new Color(c.getRed(), c.getGreen(), c.getBlue(), alpha);
    }

    /**
     * @param c
     * @param alpha
     * @return
     */
    public static Color withAlpha(Color c, float alpha) {
        return withAlpha(c, (int) (255.0f * alpha));
    }
    
    public static Color gamma(Color c, double y) {
    	double oneOverGamma = 1.0D / y;
    	int r = c.getRed();
    	int g = c.getGreen();
    	int b = c.getBlue();
    	r = (int) ( 255. * ( Math.pow( r / 255.0D, oneOverGamma ) + 0.5D ) );
    	g = (int) ( 255. * ( Math.pow( g / 255.0D, oneOverGamma ) + 0.5D ) );
    	b = (int) ( 255. * ( Math.pow( b / 255.0D, oneOverGamma ) + 0.5D ) );
	    if ( r > 255 ) r = 255;    	
	    if ( g > 255 ) g = 255;    	
	    if ( b > 255 ) b = 255;    	
    	
    	return new Color(r, g, b, c.getAlpha());
    }
    
    public static Color hexColor(String hex) {
    	if(hex.startsWith("#")) {
    		if(hex.length() == 4) {
    			int r = Character.digit(hex.charAt(1), 16);
    			int g = Character.digit(hex.charAt(2), 16);
    			int b = Character.digit(hex.charAt(3), 16);
    			return new Color(r+(r<<4), g+(g<<4), b+(b<<4));
    		} else if(hex.length() == 7) {
    			return new Color(Integer.parseInt(hex.substring(1), 16));
    		}
    	}
    	return Color.BLACK;
    }

}
