/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.events;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.simantics.scenegraph.g2d.events.FocusEvent.FocusGainedEvent;
import org.simantics.scenegraph.g2d.events.FocusEvent.FocusLostEvent;
import org.simantics.scenegraph.g2d.events.KeyEvent.KeyPressedEvent;
import org.simantics.scenegraph.g2d.events.KeyEvent.KeyReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseClickEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDoubleClickedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseEnterEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseExitEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseWheelMovedEvent;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;

/**
 * Event type masks are used for specifying the kinds of events an
 * {@link IEventHandler} wishes to receive.
 * 
 * @author Tuukka Lehtonen
 */
public final class EventTypes {

    /**
     * @see CommandEvent
     */
    public static final int Command                 = 1;
    public static final int CommandMask             = toMask(Command);

    /**
     * @see FocusGainedEvent
     */
    public static final int FocusGained             = 2;
    public static final int FocusGainedMask         = toMask(FocusGained);

    /**
     * @see FocusLostEvent
     */
    public static final int FocusLost               = 3;
    public static final int FocusLostMask           = toMask(FocusLost);

    /**
     * One of: {@link #FocusGained}, {@link #FocusLost}.
     */
    public static final int FocusMask               = FocusGainedMask | FocusLostMask;

    /**
     * @see KeyPressedEvent
     */
    public static final int KeyPressed              = 4;
    public static final int KeyPressedMask          = toMask(KeyPressed);

    /**
     * @see KeyReleasedEvent
     */
    public static final int KeyReleased             = 5;
    public static final int KeyReleasedMask         = toMask(KeyReleased);

    /**
     * One of: {@link #KeyPressed}, {@link #KeyReleased}
     */
    public static final int KeyMask                 = KeyPressedMask | KeyReleasedMask;

    /**
     * @see MouseButtonPressedEvent
     */
    public static final int MouseButtonPressed      = 6;
    public static final int MouseButtonPressedMask  = toMask(MouseButtonPressed);

    /**
     * @see MouseButtonReleasedEvent
     */
    public static final int MouseButtonReleased     = 7;
    public static final int MouseButtonReleasedMask = toMask(MouseButtonReleased);

    /**
     * @see MouseClickEvent
     */
    public static final int MouseClick              = 8;
    public static final int MouseClickMask          = toMask(MouseClick);

    /**
     * @see MouseDoubleClickedEvent
     */
    public static final int MouseDoubleClick        = 9;
    public static final int MouseDoubleClickMask    = toMask(MouseDoubleClick);

    /**
     * @see org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin
     */
    public static final int MouseDragBegin          = 10;
    public static final int MouseDragBeginMask      = toMask(MouseDragBegin);

    /**
     * One of: {@link #MouseButtonPressed}, {@link #MouseButtonReleased},
     * {@link #MouseClick}, {@link #MouseDoubleClick}, {@link #MouseDragBegin}
     */
    public static final int MouseButtonMask         = MouseButtonPressedMask | MouseButtonReleasedMask | MouseClickMask
                                                      | MouseDoubleClickMask | MouseDragBeginMask;

    /**
     * @see MouseEnterEvent
     */
    public static final int MouseEnter              = 11;
    public static final int MouseEnterMask          = toMask(MouseEnter);

    /**
     * @see MouseExitEvent
     */
    public static final int MouseExit               = 12;
    public static final int MouseExitMask           = toMask(MouseExit);

    /**
     * @see MouseMovedEvent
     */
    public static final int MouseMoved              = 13;
    public static final int MouseMovedMask          = toMask(MouseMoved);

    public static final int MouseMoveMask           = MouseEnterMask | MouseExitMask | MouseMovedMask;

    /**
     * @see MouseWheelMovedEvent
     */
    public static final int MouseWheel              = 14;
    public static final int MouseWheelMask          = toMask(MouseWheel);

    /**
     * One of: {@link #MouseButton}, {@link #MouseMove}, {@link #MouseWheel}
     */
    public static final int MouseMask               = MouseButtonMask | MouseMoveMask | MouseWheelMask;

    /**
     * @see TimeEvent
     */
    public static final int Time                    = 15;
    public static final int TimeMask                = toMask(Time);

    /**
     * One of: {@link #Command}, {@link #Focus}, {@link #Key}, {@link #Mouse},
     * {@link #Time}
     */
    public static final int AnyMask                 = CommandMask | FocusMask | KeyMask | MouseMask | TimeMask;

    /**
     * For testing whether the specified event mask passes the specified event.
     * 
     * @param eventMask
     * @param event
     * @return
     */
    public static boolean passes(IEventHandler handler, Event event) {
        int mask = handler.getEventMask();
        return passes(mask, event);
    }

    /**
     * For testing whether the specified event mask passes the specified event.
     * 
     * @param mask
     * @param event
     * @return
     */
    public static boolean passes(int mask, Event event) {
        int et = toTypeMask(event);
        return (et & mask) != 0;
    }

    /**
     * For testing whether the specified event mask passes the specified event.
     * 
     * @param eventMask
     * @param event
     * @return
     */
    public static boolean passes(IEventHandler handler, int eventTypeMask) {
        return (eventTypeMask & handler.getEventMask()) != 0;
    }

    /**
     * For testing whether the specified event type passes the specified event mask.
     * 
     * @param mask
     * @param event
     * @return
     */
    public static boolean passes(int mask, int eventTypeMask) {
        return (eventTypeMask & mask) != 0;
    }

    /**
     * Converts the specified event instance to an event type mask representing the event.
     * 
     * @param event
     * @return
     */
    public static int toTypeMask(Event event) {
        return event != null ? toTypeMask(event.getClass()) : 0;
    }

    /**
     * Converts the specified event instance to an integer describing the event.
     * 
     * @param event
     * @return
     */
    public static int toType(Event event) {
        return event != null ? toType(event.getClass()) : 0;
    }

    /**
     * Converts the specified event class to an event type mask representing the event.
     * 
     * @param event
     * @return
     */
    public static int toTypeMask(Class<? extends Event> clazz) {
        if (clazz == null)
            return 0;
        Integer type = maskCache.get(clazz);
        if (type != null)
            return type;
        type = calculateTypeMask(clazz);
        maskCache.putIfAbsent(clazz, type);
        return type;
    }

    /**
     * Converts the specified event class to an integer representing that specific event type.
     * 
     * @param event
     * @return
     */
    public static int toType(Class<? extends Event> clazz) {
        if (clazz == null)
            return 0;
        Integer type = typeCache.get(clazz);
        if (type != null)
            return type;
        type = calculateType(clazz);
        typeCache.putIfAbsent(clazz, type);
        return type;
    }

    /**
     * Converts the specified event instance to an event type mask representing
     * the event.
     * 
     * @param clazz class of event to get type for
     * @return event type mask
     * @throws IllegalArgumentException if event class is not recognized
     */
    private static int calculateTypeMask(Class<? extends Event> clazz) {
        if (Event.class.isAssignableFrom(clazz)) {
            if (MouseEvent.class.isAssignableFrom(clazz)) {
                if (MouseMovedEvent.class.isAssignableFrom(clazz))
                    return MouseMovedMask;
                if (MouseButtonPressedEvent.class.isAssignableFrom(clazz))
                    return MouseButtonPressedMask;
                if (MouseButtonReleasedEvent.class.isAssignableFrom(clazz))
                    return MouseButtonReleasedMask;
                if (MouseClickEvent.class.isAssignableFrom(clazz))
                    return MouseClickMask;
                if (MouseWheelMovedEvent.class.isAssignableFrom(clazz))
                    return MouseWheelMask;
                if (MouseDoubleClickedEvent.class.isAssignableFrom(clazz))
                    return MouseDoubleClickMask;
                if (org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin.class.isAssignableFrom(clazz))
                    return MouseDragBeginMask;
                if (MouseEnterEvent.class.isAssignableFrom(clazz))
                    return MouseEnterMask;
                if (MouseExitEvent.class.isAssignableFrom(clazz))
                    return MouseExitMask;
                return MouseMask;
            }

            if (KeyEvent.class.isAssignableFrom(clazz)) {
                if (KeyPressedEvent.class.isAssignableFrom(clazz))
                    return KeyPressedMask;
                if (KeyReleasedEvent.class.isAssignableFrom(clazz))
                    return KeyReleasedMask;
                return KeyMask;
            }

            if (FocusEvent.class.isAssignableFrom(clazz)) {
                if (FocusGainedEvent.class.isAssignableFrom(clazz))
                    return FocusGainedMask;
                if (FocusLostEvent.class.isAssignableFrom(clazz))
                    return FocusLostMask;
                return FocusMask;
            }

            if (CommandEvent.class.isAssignableFrom(clazz))
                return CommandMask;

            if (TimeEvent.class.isAssignableFrom(clazz))
                return TimeMask;

            return AnyMask;
        }
        throw new IllegalArgumentException("unrecognized event class: " + clazz);
    }


    /**
     * Converts the specified event instance to an event type mask representing
     * the event.
     * 
     * @param clazz class of event to get type for
     * @return event type mask
     * @throws IllegalArgumentException if event class is not recognized
     */
    private static int calculateType(Class<? extends Event> clazz) {
        if (Event.class.isAssignableFrom(clazz)) {
            if (MouseEvent.class.isAssignableFrom(clazz)) {
                if (MouseMovedEvent.class.isAssignableFrom(clazz))
                    return MouseMoved;
                if (MouseButtonPressedEvent.class.isAssignableFrom(clazz))
                    return MouseButtonPressed;
                if (MouseButtonReleasedEvent.class.isAssignableFrom(clazz))
                    return MouseButtonReleased;
                if (MouseClickEvent.class.isAssignableFrom(clazz))
                    return MouseClick;
                if (MouseWheelMovedEvent.class.isAssignableFrom(clazz))
                    return MouseWheel;
                if (MouseDoubleClickedEvent.class.isAssignableFrom(clazz))
                    return MouseDoubleClick;
                if (org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin.class.isAssignableFrom(clazz))
                    return MouseDragBegin;
                if (MouseEnterEvent.class.isAssignableFrom(clazz))
                    return MouseEnter;
                if (MouseExitEvent.class.isAssignableFrom(clazz))
                    return MouseExit;
            }

            if (KeyEvent.class.isAssignableFrom(clazz)) {
                if (KeyPressedEvent.class.isAssignableFrom(clazz))
                    return KeyPressed;
                if (KeyReleasedEvent.class.isAssignableFrom(clazz))
                    return KeyReleased;
            }

            if (FocusEvent.class.isAssignableFrom(clazz)) {
                if (FocusGainedEvent.class.isAssignableFrom(clazz))
                    return FocusGained;
                if (FocusLostEvent.class.isAssignableFrom(clazz))
                    return FocusLost;
            }

            if (CommandEvent.class.isAssignableFrom(clazz))
                return Command;

            if (TimeEvent.class.isAssignableFrom(clazz))
                return Time;
        }
        throw new IllegalArgumentException("unrecognized event class: " + clazz);
    }

    private static ConcurrentMap<Class<?>, Integer> maskCache = new ConcurrentHashMap<Class<?>, Integer>();
    private static ConcurrentMap<Class<?>, Integer> typeCache = new ConcurrentHashMap<Class<?>, Integer>();

    private static int toMask(int eventType) {
        return (1 << eventType);
    }

}
