/*******************************************************************************
 *  Copyright (c) 2023, 2024 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.web;

import java.awt.Color;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.eclipse.core.runtime.IAdaptable;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.diagram.adapter.GraphToDiagramSynchronizer;
import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.connection.RouteLine;
import org.simantics.diagram.connection.RouteLink;
import org.simantics.diagram.connection.RoutePoint;
import org.simantics.diagram.connection.RouteTerminal;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.content.ResourceTerminal;
import org.simantics.diagram.content.TerminalMap;
import org.simantics.diagram.elements.DiagramNodeUtil;
import org.simantics.diagram.elements.ElementTransforms;
import org.simantics.diagram.elements.TextNode;
import org.simantics.diagram.handler.CopyPasteHandler;
import org.simantics.diagram.participant.ConnectionBuilder;
import org.simantics.diagram.participant.ControlPoint;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.diagram.synchronization.graph.RemoveElement;
import org.simantics.diagram.synchronization.graph.TranslateElement;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.IContentContext;
import org.simantics.g2d.canvas.IContentContext.IContentListener;
import org.simantics.g2d.canvas.impl.CanvasContext;
import org.simantics.g2d.connection.IConnectionAdvisor;
import org.simantics.g2d.connection.TerminalKeyOf;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.DiagramUtils;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.Topology.Terminal;
import org.simantics.g2d.diagram.participant.ElementJSON;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.diagram.participant.ZOrderHandler;
import org.simantics.g2d.diagram.participant.ZOrderListener;
import org.simantics.g2d.diagram.participant.pointertool.TerminalUtil;
import org.simantics.g2d.diagram.participant.pointertool.TerminalUtil.TerminalInfo;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.elementclass.BranchPoint.Direction;
import org.simantics.g2d.elementclass.RouteGraphConnectionClass;
import org.simantics.g2d.layers.ILayers;
import org.simantics.g2d.layers.ILayers.ILayersListener;
import org.simantics.g2d.layers.ILayersEditor;
import org.simantics.g2d.layers.LayersConfiguration;
import org.simantics.g2d.participant.MouseUtil;
import org.simantics.g2d.participant.PageBorderParticipant;
import org.simantics.g2d.scenegraph.ICanvasSceneGraphProvider;
import org.simantics.g2d.tooltip.TooltipParticipant;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.triggers.IActivation;
import org.simantics.layer0.utils.triggers.IActivationManager;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.sg.DiagramSceneGraphProvider;
import org.simantics.modeling.web.serializer.RouteGraphSerializer;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DSceneGraph;
import org.simantics.scenegraph.g2d.G2DWebalizerHints;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.IG2DNodeVisitor;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.g2d.events.command.Command;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.scenegraph.g2d.nodes.ConnectionNode;
import org.simantics.scenegraph.g2d.nodes.SingleElementNode;
import org.simantics.scenegraph.g2d.nodes.connection.RouteGraphNode;
import org.simantics.scl.runtime.function.Function2;
import org.simantics.scl.runtime.tuple.Tuple2;
import org.simantics.structural2.StructuralVariables;
import org.simantics.structural2.modelingRules.CPTerminal;
import org.simantics.structural2.modelingRules.ConnectionJudgement;
import org.simantics.structural2.modelingRules.IConnectionPoint;
import org.simantics.structural2.modelingRules.IModelingRules;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.Triple;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.threads.ThreadUtils;
import org.simantics.utils.threads.WorkerThread;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * @author Jussi Koskela
 * @since 1.57.0
 */
public class SceneGraphWebalizer {
	private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(SceneGraphWebalizer.class);

	private WorkerThread thread = new WorkerThread("Web Diagram Painter");
	private CanvasContext ctx = new CanvasContext(thread);
	private IActivation activation;
	private Function2<String, String, Boolean> listener;

	private Resource diagram;
	private ICanvasSceneGraphProvider provider;

	private ObjectMapper mapper = new NodeMapper();

	// client state
	private Map<Long, IG2DNode> nodeMap = new HashMap<>();
	private Map<Long, Map<Long, ObjectNode>> fullClientNodeState = new HashMap<>();
	private DirtyElementsTracker dirtyElementsTracker = new DirtyElementsTracker();
	private String border = null;
	private String clientBorder = null;
	private Map<Long, Integer> elementZMap = new HashMap<>();
	private boolean orderChanged = false;
	private boolean crossingChanged = false;
	private String crossingStyle = null;
	private Double crossingWidth = 0.0;
	private Color backgroundColor = null;
	private boolean backgroundColorChanged = true;
	private boolean defaultTransformations = true;
	private LayersConfiguration layersConfiguration = null;

	private boolean init = true;

	public SceneGraphWebalizer(Resource diagram, Function2<String, String, Boolean> listener) throws Exception {
		this(diagram, null, listener);
	}

	public SceneGraphWebalizer(Resource diagram, Variable __context, Function2<String, String, Boolean> listener) throws Exception {
		this.diagram = diagram;
		this.listener = listener;
		thread.start();

		ctx.add(dirtyElementsTracker);

		ctx.add(new ElementJSONImpl());

		final ISessionContext sessionContext = Simantics.getSessionContext();

		// IMPORTANT: Load diagram in a different thread than the canvas context thread!

		Runnable r = new Runnable() {
			@Override
			public void run() {
				try {
					
					if(ctx.isDisposed())
						return;
					
					Pair<Resource, String> modelAndRVI = sessionContext.getSession()
							.syncRequest(new UniqueRead<Pair<Resource, String>>() {
								@Override
								public Pair<Resource, String> perform(ReadGraph graph) throws DatabaseException {
									Resource model = resolveModel(graph, diagram);
									return new Pair<Resource, String>(model, resolveRVI(graph, model, diagram));
								}
							});
					provider = DiagramNodeUtil.loadSceneGraphProviderForDiagram(ctx,
							modelAndRVI.first, diagram, modelAndRVI.second, layersConfiguration);

					IDiagram d = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
					ILayersEditor layersEditor = d.getHint(DiagramHints.KEY_LAYERS);
					if(layersEditor != null) {
						layersEditor.addLayersListener(new ILayersListener() {

							@Override
							public void changed() {
								ThreadUtils.asyncExec(thread, new Runnable() {
									@Override
									public void run() {
										for (IElement element : d.getElements()) {
											element.setHint(Hints.KEY_DIRTY, Hints.VALUE_SG_DIRTY);
										}
										provider.getCanvasContext().getContentContext().setDirty();
									}
								});
							}

						});
					}

					ctx.add( new PageBorderParticipant() {
						@Override
						protected void updateNode(boolean markDirty) {
							super.updateNode(markDirty);
							runMutex(() -> {
								ObjectMapper mapper = new NodeMapper();
								JSONGenerator jsonGenerator = new JSONGenerator(mapper, null);
								border =  jsonGenerator.asSVG(node, false);
								provider.getCanvasContext().getContentContext().setDirty();
							});
						}
					});
					ZOrderHandler zoh = ctx.getAtMostOneItemOfClass(ZOrderHandler.class);
					zoh.addOrderListener(new ZOrderListener() {

						@Override
						public void orderChanged(IDiagram diagram) {
							runMutex(() -> {
								orderChanged = true;
								provider.getCanvasContext().getContentContext().setDirty();
							});
						}
					});

					Simantics.getSession().async(new UnaryRead<Resource, Pair<Double, String>>(diagram){

						@Override
						public Pair<Double, String> perform(ReadGraph graph) throws DatabaseException {
							DiagramResource DIA = DiagramResource.getInstance(graph);
							Layer0 L0 = Layer0.getInstance(graph);
							Double width = graph.getPossibleRelatedValue(diagram, DIA.ConnectionCrossingStyle_Width);
							Resource typeRes = graph.getPossibleObject(diagram, DIA.ConnectionCrossingStyle_HasType);
							String style = graph.getPossibleRelatedValue(typeRes, L0.HasName);
							return new Pair<>(width, style);
						}

					}, new Listener<Pair<Double, String>>() {

						@Override
						public void execute(Pair<Double, String> result) {
							runMutex(() -> {
								crossingChanged = true;
								crossingWidth = result.first;
								crossingStyle = result.second;
								provider.getCanvasContext().getContentContext().setDirty();
							});
						}

						@Override
						public void exception(Throwable t) {
						}

						@Override
						public boolean isDisposed() {
							if(ctx == null)
								return true;
							return ctx.isDisposed();
						}
					});

					IHintContext h = ctx.getDefaultHintContext();
					backgroundColor = h.getHint(Hints.KEY_BACKGROUND_COLOR);
					h.addKeyHintListener(Hints.KEY_BACKGROUND_COLOR, new IHintListener() {
						public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
							runMutex(() -> {
								backgroundColor = (Color)newValue;
								backgroundColorChanged = true;
								provider.getCanvasContext().getContentContext().setDirty();
							});
						}
						public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {
							runMutex(() -> {
								backgroundColor = null;
								backgroundColorChanged = true;
								provider.getCanvasContext().getContentContext().setDirty();
							});
						}
					});


					IActivationManager activationManager = sessionContext.getSession().peekService(IActivationManager.class);
					if (activationManager != null) {
						activation = activationManager.activate(diagram);
					}

					// update state
					Runnable repaint = () -> {
						if(isDisposed())
							return;
						Pair<Set<IElement>, Set<Long>> result = dirtyElementsTracker.fetchChanges();
						Set<IElement> dirtyElements = result.first;
						Set<Long> removedElements = result.second;
						if (dirtyElements.size() > 0 || removedElements.size() > 0 || !Objects.equals(border, clientBorder) || orderChanged || crossingChanged || backgroundColorChanged) {
							onRepaint(dirtyElements, removedElements);
						}
					};

					runMutex(repaint);

					provider.getCanvasContext().getContentContext().addPaintableContextListener(new IContentListener() {
						@Override
						public void onDirty(IContentContext sender) {
							runMutex(repaint);
						}
					});

				} catch (DatabaseException | InterruptedException e) {
					LOGGER.error("Failed to load diagram.", e);
				}
			}
		};

		Thread t = new Thread(r);
		t.start();

	}

	public class ElementJSONImpl implements ElementJSON {

		Set<Key> rejectSet = new HashSet<IHintContext.Key>();

		public ElementJSONImpl() {
			// TODO : we should use accept list instead of rejectList.
			//        The problem is that we need a way to inject accepted hint keys, without adding additional dependecies to this plug-in.
			rejectSet.add(ElementHints.KEY_IMAGE);
			rejectSet.add(ElementHints.KEY_COMPOSITE);
			rejectSet.add(ElementHints.KEY_DECORATORS);
			rejectSet.add(ElementHints.KEY_OBJECT);
			rejectSet.add(ElementHints.KEY_TRANSFORM);
			rejectSet.add(ElementHints.KEY_SG_NODE);
			rejectSet.add(ElementHints.KEY_PARENT_ELEMENT);
			rejectSet.add(ElementHints.KEY_COPY_OF_OBJECT);
			rejectSet.add(ElementHints.KEY_ANCHOR);
			rejectSet.add(ElementHints.KEY_FOCUS_LAYERS);
			rejectSet.add(ElementHints.KEY_VISIBLE_LAYERS);
			rejectSet.add(ElementHints.KEY_HOVER);
			rejectSet.add(ElementHints.KEY_TYPE_CLASS);
			rejectSet.add(ElementHints.KEY_SG_NAME);
			rejectSet.add(ElementHints.KEY_SG_CALLBACK);
			rejectSet.add(ElementHints.KEY_CONNECTION_ENTITY);
			rejectSet.add(SynchronizationHints.HINT_SYNCHRONIZER);
			rejectSet.add(TooltipParticipant.TOOLTIP_KEY);
			rejectSet.add(RouteGraphConnectionClass.KEY_PICK_TOLERANCE);
			rejectSet.add(RouteGraphConnectionClass.KEY_RENDERER);
			rejectSet.add(RouteGraphConnectionClass.KEY_RG_LISTENER);
			rejectSet.add(RouteGraphConnectionClass.KEY_RG_NODE);
			rejectSet.add(RouteGraphConnectionClass.KEY_ROUTEGRAPH);
			rejectSet.add(RouteGraphConnectionClass.KEY_USE_TOLERANCE_IN_SELECTION);
			rejectSet.add(Hints.KEY_DIRTY);
		}

		@Override
		public void removedFromContext(ICanvasContext ctx) {

		}

		@Override
		public void addedToContext(ICanvasContext ctx) {

		}

		@Override
		public Optional<String> getJSON(IElement element) {
			ObjectNode node = new ObjectNode(JsonNodeFactory.instance);

			for (Entry<Key, Object> entry : element.getHints().entrySet()) {
				if (rejectSet.contains(entry.getKey()))
					continue;
				if (entry.getKey() instanceof TerminalKeyOf)
					continue;
				if (entry.getValue() == null)
					continue;
				if (entry.getValue() instanceof INode)
					continue;
				if (entry.getValue() instanceof ICanvasContext)
					continue;
				JsonNode n = mapper.valueToTree(entry.getValue());
				if (n != null) {
					String name = entry.getKey().toString();
					if (name.indexOf("(") > 0)
						name = name.substring(0, name.indexOf("("));
					else {
						name = name.replaceAll("(", "");
						name = name.replaceAll(")", "");
					}
					node.set(name, n);
				}
			}

			try {
				return Optional.of(mapper.writeValueAsString(node));
			} catch (Exception e) {
				return Optional.empty();
			}
		}
	}

	public void resetClientState() {
		runMutex(() -> {
			fullClientNodeState.clear();
			nodeMap.clear();
			clientBorder = null;
			dirtyElementsTracker.reset();
			provider.getCanvasContext().getContentContext().setDirty();
		});
	}

	private void onRepaint(Set<IElement> dirtyElements, Set<Long> removedElements) {
		ObjectNode root = new ObjectNode(JsonNodeFactory.instance);

		ICanvasContext ctx = provider.getCanvasContext();
		if(ctx == null)
			return;
		
		IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
		if(diagram == null)
			return;
		
		ILayers layers = diagram.getHint(DiagramHints.KEY_LAYERS);
		JSONGenerator jsonGenerator = new JSONGenerator(mapper, diagram);

		assert (((G2DSceneGraph) ctx.getSceneGraph() != null));
		assert (((G2DSceneGraph) ctx.getSceneGraph().getRootNode() != null));
		((G2DSceneGraph) ctx.getSceneGraph().getRootNode()).performCleanup();

		if (!Objects.equals(border, clientBorder)) {
			root.put("border", border);
			clientBorder = border;
		}

		if (orderChanged) {
			ArrayNode reorder = root.putArray("reorder");
			for (IElement element : diagram.getElements()) {
				IG2DNode node = element.getHint(ElementHints.KEY_SG_NODE);
				Resource res = (Resource) ElementUtils.getObject(element);
				Integer currentZ = node.getZIndex();
				long elementId = res.getResourceId();
				Integer prevZ = elementZMap.put(elementId, currentZ);
				if (!Objects.equals(prevZ, currentZ)) {
					ObjectNode elem = reorder.addObject();
					IG2DNode n = element.getHint(ElementHints.KEY_SG_NODE);
					elem.put("id", elementId);
					elem.put("z", n.getZIndex());

				}
			}
			orderChanged = false;
		}

		if (crossingChanged) {
			ObjectNode crossings = root.putObject("crossing");
			crossings.put("width", crossingWidth);
			crossings.put("style", crossingStyle);
			crossingChanged = false;
		}

		if (backgroundColorChanged) {
			ObjectNode background = root.putObject("background");
			background.put("color", (backgroundColor != null ? String.format("#%02x%02x%02x", backgroundColor.getRed(), backgroundColor.getGreen(), backgroundColor.getBlue()) : "#ffffff"));
			backgroundColorChanged = false;
		}

		ArrayNode elements = root.putArray("elements");
		List<Triple<IElement, Terminal, ObjectNode>> judgeTasks = new ArrayList<>();

		for (IElement dirtyElement : dirtyElements) {
			
			Resource res = (Resource) ElementUtils.getObject(dirtyElement);
			
			long elementId = 0;
			
			if(res != null) {
				elementId = res.getResourceId();
				if (removedElements.contains(elementId))
					continue;
			} else {
				continue;
			}

			IG2DNode node = dirtyElement.getHint(ElementHints.KEY_SG_NODE);

			if (node instanceof SingleElementNode) {
				SingleElementNode sne = (SingleElementNode)node;
				boolean notLayersIgnored = layers != null ? !layers.getIgnoreVisibilitySettings() : false;
				if (notLayersIgnored && (sne.isHidden() || !sne.isVisible())) {
					if (fullClientNodeState.containsKey(elementId)) { // consider as removed element if it was present earlier 
						removedElements.add(elementId);
					}
					continue;
				}
			}

			ObjectNode element = elements.addObject();

			Map<Long, ObjectNode> clientNodeState = fullClientNodeState.get(elementId);
			if (clientNodeState == null) {
				clientNodeState = new HashMap<Long, ObjectNode>();
				fullClientNodeState.put(elementId, clientNodeState);
			}
			
			boolean isConnection = node instanceof ConnectionNode;
			boolean selectable = true;
			
			if (node instanceof SingleElementNode) {
				SingleElementNode sne = (SingleElementNode)node;
				if(sne.getNodeCount() == 1) {
					for(IG2DNode child : sne.getNodes()) {
						if(child instanceof IAdaptable) {
							G2DWebalizerHints hints = ((IAdaptable) child).getAdapter(G2DWebalizerHints.class);
							if(hints != null) {
								selectable = hints.isSelectable();
							}
						}
					}
				}
			}

			element.put("id", elementId);
			element.put("root", node.getId());
			element.put("perPixel", isConnection);
			element.put("selectable", selectable);
			element.put("z", node.getZIndex());
			element.put("draggable", !isConnection);

			ArrayNode terminalsNode = element.putArray("terminals");

			List<Terminal> terminals = new ArrayList<Terminal>();
			ElementUtils.getTerminals(dirtyElement, terminals, false);

			for (Terminal terminal : terminals) {
				long terminalId = ((ResourceTerminal)terminal).getResource().getResourceId();

				Shape shape = TerminalUtil.getTerminalShape(dirtyElement, terminal);
				AffineTransform transform = TerminalUtil.getTerminalPosOnDiagram(dirtyElement,
						terminal);

				Path2D shapePath = new Path2D.Double(shape, transform);
				Rectangle2D bbox = shapePath.getBounds2D();

				// This is legacy. Obviously we cannot determine allowed directions from the element.
				// DirectionSet directions = TerminalUtil.getTerminalDirectionSet(dirtyElement,
				//        terminal, null);

				ObjectNode terminalNode = terminalsNode.addObject();
				terminalNode.put("id", terminalId);
				terminalNode.set("shape", mapper.valueToTree(shapePath));

				ObjectNode bboxNode = terminalNode.putObject("bbox");
				bboxNode.put("x", bbox.getX());
				bboxNode.put("y", bbox.getY());
				bboxNode.put("width", bbox.getWidth());
				bboxNode.put("height", bbox.getHeight());

				judgeTasks.add(new Triple<>(dirtyElement, terminal, terminalNode));

			}

			ArrayNode nodes = element.putArray("nodes");
			ArrayNode removed = element.putArray("removedNodes");

			jsonGenerator.getNodes().clear();
			node.accept(jsonGenerator);

			synchronized (nodeMap) {
				node.accept(new IG2DNodeVisitor() {

					@Override
					public void leave(IG2DNode node) {
					}

					@Override
					public void enter(IG2DNode node) {
						nodeMap.put(node.getId(), node);
					}
				});
			}

			Set<Long> removedNodes = new HashSet<Long>();
			removedNodes.addAll(clientNodeState.keySet());

			for (ObjectNode currentNode : jsonGenerator.getNodes()) {
				Long id = currentNode.get("id").asLong();
				ObjectNode oldNode = clientNodeState.get(id);
				if (!currentNode.equals(oldNode)) {
					nodes.add(currentNode);
					clientNodeState.put(id, currentNode);
				}
				removedNodes.remove(id);
			}
			synchronized (nodeMap) {
				for (Long id : removedNodes) {
					removed.add(id);
					clientNodeState.remove(id);
					elementZMap.remove(id);
					nodeMap.remove(id);
				}
			}
		}


		try {
			Simantics.sync(new ReadRequest() {

				@Override
				public void run(ReadGraph graph) throws DatabaseException {

					IDiagram diagram = provider.getCanvasContext().getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
					IModelingRules modelingRules = diagram.getHint(DiagramModelHints.KEY_MODELING_RULES);
					IConnectionAdvisor connectionAdvisor = diagram.getHint(DiagramHints.CONNECTION_ADVISOR);
					if (connectionAdvisor != null) {
						for (Triple<IElement, Terminal, ObjectNode> judgeTask : judgeTasks) {
							IElement element = judgeTask.first;
							Terminal terminal = judgeTask.second;
							ObjectNode terminalNode = judgeTask.third;

							terminalNode.put("canBegin", connectionAdvisor.canBeginConnection(graph, element, terminal));
						}

					} else {
						for (Triple<IElement, Terminal, ObjectNode> judgeTask : judgeTasks) {
							IElement element = judgeTask.first;
							Terminal terminal = judgeTask.second;
							ObjectNode terminalNode = judgeTask.third;

							TerminalInfo ti = new TerminalInfo();
							ti.e = element;
							ti.t = terminal;

							ConnectionJudgement judgement = modelingRules.judgeConnection(graph,
									Arrays.asList(ConnectionUtil.toConnectionPoint(graph, ti)));

							terminalNode.put("canBegin", judgement != null && judgement != ConnectionJudgement.ILLEGAL);
						}
					}
				}
			});
		} catch (DatabaseException e1) {
			LOGGER.error("Failed to judge connections.", e1);
		}

		ArrayNode removed = root.putArray("removed");
		for (long elementId : removedElements) {
			removed.add(elementId);
			fullClientNodeState.remove(elementId);
		}

		ArrayNode newDefs = root.putArray("defs");
		for (Map.Entry<String, String> entry : jsonGenerator.flushNewDefs().entrySet()) {
			ObjectNode newDef = newDefs.addObject();
			newDef.put("id", entry.getKey());
			newDef.put("def", entry.getValue());
		}

		try {
			String s = mapper.writeValueAsString(root);
			if (!listener.apply(init ? "init" : "update", s)) {
				cleanUp();
			}
			init = false;
		} catch (JsonProcessingException e) {
			LOGGER.error("Failed to write JSON string.", e);
			cleanUp();
		}

	}

	public Resource elementIdToElement(ReadGraph graph, String id) throws DatabaseException {
		SerialisationSupport ss = graph.getService(SerialisationSupport.class);
		return ss.getResource(Long.parseLong(id));
	}

	public Resource terminalIdToElement(ReadGraph graph, String id) throws DatabaseException {
		SerialisationSupport ss = graph.getService(SerialisationSupport.class);
		return ss.getResource(Long.parseLong(id));
	}

	public Resource connectIdToConnection(ReadGraph graph, String id) throws DatabaseException {
		SerialisationSupport ss = graph.getService(SerialisationSupport.class);
		return ss.getResource(Long.parseLong(id));
	}

	private Resource[] elementIdsToElementArray(ReadGraph graph, List<String> ids) throws DatabaseException {
		List<Resource> elements = new ArrayList<>();
		for (String id : ids) {
			Resource res = elementIdToElement(graph, id);
			if (res != null) {
				elements.add(res);
			}
		}
		return elements.toArray(new Resource[elements.size()]);
	}

	public void translateElements(WriteGraph graph, List<String> ids, double dx, double dy) throws DatabaseException {
		if(!defaultTransformations)
			return;
		// TODO: refactor org.simantics.diagram.participant.TranslateMode2
		DiagramResource DIA = DiagramResource.getInstance(graph);
		ConnectionUtil cu = new ConnectionUtil(graph);
		graph.markUndoPoint();
		// AffineTransform at = AffineTransform.getTranslateInstance(dx, dy);
		for (String id : ids) {
			Resource element = elementIdToElement(graph, id);
			// DiagramGraphUtil.changeTransform(graph, element, at);


			if (graph.isInstanceOf(element, DIA.RouteGraphConnection)) {
				cu.translateRouteNodes(element, dx, dy);
			} else {
				TranslateElement.offset(element, dx, dy).perform(graph);
			}
		}
	}

	public void transformElements(WriteGraph graph, List<Tuple2> transformations) throws DatabaseException {
		if(!defaultTransformations)
			return;
		graph.markUndoPoint();
		for (Tuple2 tuple: transformations) {
			String id = (String) tuple.c0;
			@SuppressWarnings("unchecked")
			List<Double> t = (List<Double>) tuple.c1;
			Resource element = elementIdToElement(graph, id);
			AffineTransform at = DiagramGraphUtil.getTransform(graph, element);
			AffineTransform at2 = new AffineTransform(t.get(0), t.get(1), t.get(2), t.get(3), t.get(4), t.get(5));
			DiagramGraphUtil.changeTransform(graph, element, at2);

			System.out.println(at.toString());
			System.out.println(t.toString());
		}
	}

	public void rotateElements(WriteGraph graph, List<String> ids, boolean clockwise) throws DatabaseException {
		if(!defaultTransformations)
			return;
		IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
		if(diagram != null)
			ElementTransforms.rotate(diagram, elementIdsToElementArray(graph, ids), clockwise);
	}

	public void flipElements(WriteGraph graph, List<String> ids, boolean horizontal) throws DatabaseException {
		if(!defaultTransformations)
			return;
		IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
		if(diagram != null)
			ElementTransforms.flip(diagram, elementIdsToElementArray(graph, ids), !horizontal);
	}

	private Collection<IElement> elementIdsToIElements(CanvasContext ctx, Collection<String> ids) throws DatabaseException {
		return Simantics.sync(new UnaryRead<Collection<String>, Collection<IElement>>(ids) {

			@Override
			public Collection<IElement> perform(ReadGraph graph) throws DatabaseException {
				Collection<IElement> result = new ArrayList<>();
				for (String elementId : ids) {
					Resource er = elementIdToElement(graph, elementId);
					IElement e = DiagramNodeUtil.findElement(ctx, er);
					result.add(e);
				}
				return result;
			}
		});
	}

	private void handleZCommand(List<String> ids, Command command) {
		ThreadUtils.asyncExec(thread, new Runnable() {
			@Override
			public void run() {
				ZOrderHandler zoh = ctx.getAtMostOneItemOfClass(ZOrderHandler.class);
				try {
					Collection<IElement> elements = elementIdsToIElements(ctx, ids); 
					Selection selection = ctx.getAtMostOneItemOfClass(Selection.class);
					selection.setSelection(0, elements);
					zoh.handleCommand(new CommandEvent(null, 0, command));
					selection.setSelection(0, Collections.emptySet());
				} catch (DatabaseException e) {
					LOGGER.error("Failed to change the Z order.", e);
				}
			}
		});
	}

	public void moveElementsUp(List<String> ids) {
		handleZCommand(ids, Commands.BRING_UP);
	}

	public void moveElementsDown(List<String> ids) {
		handleZCommand(ids, Commands.SEND_DOWN);
	}

	public void moveElementsToTop(List<String> ids) {
		handleZCommand(ids, Commands.BRING_TO_TOP);
	}

	public void moveElementsToBottom(List<String> ids) {
		handleZCommand(ids, Commands.SEND_TO_BOTTOM);
	}

	public void connectFrom(ReadGraph graph, String elementId, String terminalId) throws DatabaseException {

		DiagramResource DIA = DiagramResource.getInstance(graph);
		IDiagram idiagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
		IModelingRules modelingRules = idiagram.getHint(DiagramModelHints.KEY_MODELING_RULES);
		IConnectionAdvisor connectionAdvisor = idiagram.getHint(DiagramHints.CONNECTION_ADVISOR);

		ObjectMapper mapper = new ObjectMapper();
		ObjectNode result = mapper.createObjectNode();
		result.put("source", Long.parseLong(terminalId));
		ArrayNode targets = result.putArray("targets");

		Resource element = elementIdToElement(graph, elementId);
		IElement e = DiagramNodeUtil.findElement(ctx, element);
		Resource terminal = terminalIdToElement(graph, terminalId);
		Terminal t = new ResourceTerminal(terminal);
		Resource cp = DiagramGraphUtil.getConnectionPointOfTerminal(graph, terminal);
		CPTerminal cpt = new CPTerminal(element, cp);


		for (Resource targetElement : OrderedSetUtils.toList(graph, diagram)) {
			if (graph.isInstanceOf(targetElement, DIA.Connection)) continue;
			if (targetElement.equals(element))
				continue;
			IElement e2 = DiagramNodeUtil.findElement(ctx, targetElement);
			ModelingResources MOD = ModelingResources.getInstance(graph);

			// workaround to skip reading terminals from elements with no corresponding component
			Resource component = graph.getPossibleObject(targetElement, MOD.ElementToComponent);
			if (component == null) continue;

			TerminalMap terminals = DiagramGraphUtil.getElementTerminals(graph, targetElement);
			for (Resource targetTerminal : terminals.getTerminals()) {

				Terminal t2 = new ResourceTerminal(targetTerminal);
				Resource targetCP = DiagramGraphUtil.getConnectionPointOfTerminal(graph, targetTerminal);

				CPTerminal targetCPT = new CPTerminal(targetElement, targetCP);

				ConnectionJudgement judgement = null;
				if (connectionAdvisor != null) {
					judgement = (ConnectionJudgement)connectionAdvisor.canBeConnected(graph, e, t, e2, t2);
				} else {

					judgement = modelingRules.judgeConnection(graph, Arrays.asList(cpt, targetCPT));
				}

				if (judgement != null && judgement != ConnectionJudgement.ILLEGAL) {
					Long targetTerminalId = targetTerminal.getResourceId();
					ObjectNode target = targets.addObject();

					target.put("e", targetElement.getResourceId());
					target.put("t", targetTerminalId);
				}
			}
		}

		try {
			listener.apply("mayConnectTo", mapper.writeValueAsString(result));
		} catch (JsonProcessingException ex) {
			LOGGER.error("Failed to create Json string.", ex);
		}

	}

	public void connectFromFlag(WriteGraph graph, double flagX, double flagY, String elementId, String terminalId) throws DatabaseException {
		connectNewFlag(flagX, flagY, elementId, terminalId, true);
	}

	public void connectToFlag(double flagX, double flagY, String elementId, String terminalId) throws DatabaseException {
		connectNewFlag(flagX, flagY, elementId, terminalId, false);
	}

	public void connectNewFlag(double flagX, double flagY, String elementId, String terminalId, boolean startFromFlag) {
		IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
		Boolean flags = diagram.getHint(DiagramHints.KEY_USE_CONNECTION_FLAGS);
		if (flags != null && flags.booleanValue() == false)
			return;

		ThreadUtils.asyncExec(thread, new Runnable() {
			@Override
			public void run() {

				try {
					Simantics.sync(new WriteRequest() {

						@Override
						public void perform(WriteGraph graph) throws DatabaseException {
							IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
							ConnectionBuilder builder = new ConnectionBuilder(diagram);

							IModelingRules modelingRules = diagram.getHint(DiagramModelHints.KEY_MODELING_RULES);
							IConnectionAdvisor connectionAdvisor = diagram.getHint(DiagramHints.CONNECTION_ADVISOR);

							if (modelingRules == null) return;

							Resource er = elementIdToElement(graph, elementId);

							IElement e = DiagramNodeUtil.findElement(ctx, er);

							Terminal t = new ResourceTerminal(terminalIdToElement(graph, terminalId));

							TerminalInfo ti = new TerminalInfo();
							ti.e = e;
							ti.t = t;

							Collection<IConnectionPoint> pts = new ArrayList<>();
							pts.add(ConnectionUtil.toConnectionPoint(graph, ti));
							ConnectionJudgement judgement = modelingRules.judgeConnection(graph, pts);

							if (judgement != null && judgement != ConnectionJudgement.ILLEGAL) {
								Deque<ControlPoint> cps = new LinkedList<ControlPoint>();
								cps.add(new ControlPoint(new Point2D.Double(flagX, flagY), Direction.Any));
								if (startFromFlag) {
									builder.create(graph, judgement, cps, null, ti);
								} else {
									builder.create(graph, judgement, cps, ti, null);
								}
							}

						}
					});
				} catch (DatabaseException e) {
					LOGGER.error("Failed to create connection.", e);
				}
			}
		});
	}

	public void connect(String elementId1, String terminalId1, String elementId2, String terminalId2) throws DatabaseException {

		ThreadUtils.asyncExec(thread, new Runnable() {
			@Override
			public void run() {

				try {
					Simantics.sync(new WriteRequest() {

						@Override
						public void perform(WriteGraph graph) throws DatabaseException {
							IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
							ConnectionBuilder builder = new ConnectionBuilder(diagram);

							IModelingRules modelingRules = diagram.getHint(DiagramModelHints.KEY_MODELING_RULES);

							Resource er1 = elementIdToElement(graph, elementId1);
							Resource er2 = elementIdToElement(graph, elementId2);

							TerminalInfo ti1 = new TerminalInfo();
							ti1.e = DiagramNodeUtil.findElement(ctx, er1);
							ti1.t = new ResourceTerminal(terminalIdToElement(graph, terminalId1));;

							TerminalInfo ti2 = new TerminalInfo();
							ti2.e = DiagramNodeUtil.findElement(ctx, er2);
							ti2.t = new ResourceTerminal(terminalIdToElement(graph, terminalId2));;


							Collection<IConnectionPoint> pts = new ArrayList<>();
							pts.add(ConnectionUtil.toConnectionPoint(graph, ti1));
							pts.add(ConnectionUtil.toConnectionPoint(graph, ti2));
							ConnectionJudgement judgement = modelingRules.judgeConnection(graph, pts);

							if (judgement != null && judgement != ConnectionJudgement.ILLEGAL) {
								builder.create(graph, judgement, new LinkedList<ControlPoint>(), ti1, ti2);
							}

						}
					});
				} catch (DatabaseException e) {
					LOGGER.error("Failed to create connection.", e);
				}
			}
		});
	}    

	public void modifyRouteGraph(String nodeId, String partId, double x, double y) throws DatabaseException {
		ThreadUtils.syncExec(thread, new Runnable() {
			@Override
			public void run() {
				IG2DNode node;
				synchronized (nodeMap) {
					node = nodeMap.get(Long.parseLong(nodeId));
				}
				if (node instanceof RouteGraphNode) {
					RouteGraphNode rgn = (RouteGraphNode) node;
					RouteGraph before = rgn.getRouteGraph();
					RouteGraph after = before.copy();

					boolean changed = false;
					for (RouteLine line : after.getAllLines()) {
						String id = RouteGraphSerializer.getRouteNodeId(line);
						if (partId.equals(id)) {
							after.setLocation(line, x, y);
							changed = true;
							break;
						}
						for (RoutePoint point : line.getPoints()) {
							if (partId.equals(RouteGraphSerializer.getRoutePointId(point))) {
								if (point instanceof RouteTerminal) {
									//after.setLocation((RouteTerminal)point, x, y);
									break;
								} else if (point instanceof RouteLink) {
									after.setLocation(((RouteLink)point).getA(), x, y);
									after.setLocation(((RouteLink)point).getB(), x, y);
									changed = true;
									break;
								}
							}
						}
					}
					// FIXME : hack! In some cases, where we have on transient line between terminals,
					// there is a difference between RG structures, and the single line gets different id (based on different terminal).  
					if (!changed && after.getAllLines().size() == 1) {
						RouteLine line = after.getAllLines().iterator().next();
						after.setLocation(line, x, y);
					}
					// fire changes even if nothing changed to keep client and server in sync
					rgn.setRouteGraphAndFireChanges(before, after);
				}
			}
		});
	}

	public void deleteRouteGraphPart(String nodeId, String partId) throws DatabaseException {
		ThreadUtils.syncExec(thread, new Runnable() {
			@Override
			public void run() {
				IG2DNode node;
				synchronized (nodeMap) {
					node = nodeMap.get(Long.parseLong(nodeId));
				}
				if (node instanceof RouteGraphNode) {
					RouteGraphNode rgn = (RouteGraphNode) node;
					RouteGraph rg = rgn.getRouteGraph();

					for (RouteLine line : rg.getAllLines()) {
						if (partId.equals(RouteGraphSerializer.getRouteNodeId(line))) {
							//rgn.deleteTarget(line); // refactor to public
							RouteGraph before = rg.copy();
							rg.merge(line);
							rgn.setRouteGraphAndFireChanges(before, rg);

							break;
						}
						for (RoutePoint point : line.getPoints()) {
							if (point instanceof RouteLink) { // don't allow deleting routeterminals
								if (partId.equals(RouteGraphSerializer.getRoutePointId(point))) {
									//rgn.deleteTarget(point); // refactor to public
									RouteGraph before = rg.copy();
									rg.deleteCorner((RouteLink) point);
									rgn.setRouteGraphAndFireChanges(before, rg);
								}
							}
						}
					}
				}
			}
		});
	}

	public void clearRouteGraph(String nodeId, String partId) throws DatabaseException {
		ThreadUtils.syncExec(thread, new Runnable() {
			@Override
			public void run() {
				IG2DNode node;
				synchronized (nodeMap) {
					node = nodeMap.get(Long.parseLong(nodeId));
				}
				if (node instanceof RouteGraphNode) {
					RouteGraphNode rgn = (RouteGraphNode) node;
					RouteGraph rg = rgn.getRouteGraph();

					boolean changed = false;
					RouteGraph before = rg.copy();
					for (RouteLine line : rg.getAllLines()) {
						rg.merge(line);
						changed = true;
						//                        for (RoutePoint point : line.getPoints()) {
						//                            if (point instanceof RouteLink) { // don't allow deleting routeterminals
						//                                rg.deleteCorner((RouteLink) point);
						//                            }
						//                        }
					}
					if (changed)
						rgn.setRouteGraphAndFireChanges(before, rg);
				}
			}
		});
	}

	public void splitRouteGraphPart(String nodeId, String partId, double position) throws DatabaseException {
		ThreadUtils.syncExec(thread, new Runnable() {
			@Override
			public void run() {
				IG2DNode node;
				synchronized (nodeMap) {
					node = nodeMap.get(Long.parseLong(nodeId));
				}
				if (node instanceof RouteGraphNode) {
					RouteGraphNode rgn = (RouteGraphNode) node;
					RouteGraph rg = rgn.getRouteGraph();

					for (RouteLine line : rg.getAllLines()) {
						if (partId.equals(RouteGraphSerializer.getRouteNodeId(line))) {
							if (line instanceof RouteLine) {
								RouteGraph before = rg.copy();
								rg.split(line, position);
								rgn.setRouteGraphAndFireChanges(before, rg);
							}
							break;
						}
					}
				}
			}
		});
	}

	public Tuple2 modifyText(String nodeId, String text) { // transaction is not allowed here due to validators
		Tuple2[] result = {null};
		ThreadUtils.syncExec(thread, new Runnable() {
			@Override
			public void run() {
				IG2DNode node;
				synchronized (nodeMap) {
					node = nodeMap.get(Long.parseLong(nodeId));
				}
				if (node instanceof TextNode) {
					TextNode n = (TextNode) node;
					String error = n.editText(text);
					result[0] = new Tuple2(n.getText(), error);
				}
			}
		});
		return result[0];
	}

	public void deleteElements(WriteGraph graph, List<String> ids) throws DatabaseException {
		graph.markUndoPoint();
		for (String id : ids) {
			Resource res = elementIdToElement(graph, id);
			RemoveElement.removeElement(graph, diagram, res);
		}
	}

	public void addElements(List<String> ids, double x, double y) throws DatabaseException {
		IDiagram d = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);

		GraphToDiagramSynchronizer synchronizer = ((DiagramSceneGraphProvider)provider).getGraphToDiagramSynchronizer();

		List<ElementClass> ecs = Simantics.getSession().sync(new UniqueRead<List<ElementClass>>() {

			@Override
			public List<ElementClass> perform(ReadGraph graph) throws DatabaseException {
				List<ElementClass> ecs = new ArrayList<>();
				SerialisationSupport ss = graph.getService(SerialisationSupport.class);
				for (String id: ids) { 
					Resource elementType = ss.getResource(Long.parseLong(id));
					if (elementType != null) {
						ElementClass ec = synchronizer.getNodeClass(graph, elementType);
						if (ec != null) {
							ecs.add(ec);
						}
					}
				}
				return ecs;
			}

		});

		DiagramUtils.mutateDiagram(d, m -> {

			if (ctx != null) {

				for (ElementClass ec : ecs) {
					Point2D pos = new Point2D.Double(x, y);
					IElement element = m.newElement(ec);
					//setupDroppedElement(element, pos);
					IElement parent = element.getHint(ElementHints.KEY_PARENT_ELEMENT);
					if (parent != null) {
						Point2D parentPos = ElementUtils.getPos(parent);
						Point2D pos2 = new Point2D.Double(pos.getX() - parentPos.getX(), pos.getY() - parentPos.getY());
						ElementUtils.setPos(element, pos2);
					} else {
						ElementUtils.setPos(element, pos);
					}
				}
			}
		});
	}

	private void copyOrCutElements(List<String> ids, boolean cut) {
		ThreadUtils.asyncExec(thread, new Runnable() {


			@Override
			public void run() {
				CopyPasteHandler cph = ctx.getAtMostOneItemOfClass(CopyPasteHandler.class);


				try {
					Collection<IElement> elements = Simantics.sync(new UniqueRead<Collection<IElement>>() {

						@Override
						public Collection<IElement> perform(ReadGraph graph) throws DatabaseException {
							Collection<IElement> result = new ArrayList<>();
							for (String elementId : ids) {
								Resource er = elementIdToElement(graph, elementId);

								IElement e = DiagramNodeUtil.findElement(ctx, er);
								result.add(e);
							}
							return result;

						}
					});

					Selection selection = ctx.getAtMostOneItemOfClass(Selection.class);
					selection.setSelection(0, elements);

					cph.handleCommand(new CommandEvent(null, 0, cut ? Commands.CUT : Commands.COPY));

					selection.setSelection(0, Collections.emptySet());

				} catch (DatabaseException e) {
					LOGGER.error("Failed to copy/cut elements.", e);
				}
			}
		});

	}

	public void copyElements(List<String> ids) {
		copyOrCutElements(ids, false);
	}

	public void cutElements(List<String> ids) {
		copyOrCutElements(ids, true);
	}

	public void pasteElements(double x, double y) {
		ThreadUtils.asyncExec(thread, new Runnable() {

			@Override
			public void run() {
				CopyPasteHandler cph = ctx.getAtMostOneItemOfClass(CopyPasteHandler.class);
				MouseUtil mouseUtil = ctx.getSingleItem(MouseUtil.class);
				mouseUtil.handleMouseEvent(new MouseMovedEvent(null, 0, 0, 0, 0, new Point2D.Double(x, y), null));

				cph.handleCommand(new CommandEvent(null, 0, Commands.PASTE));

				Selection selection = ctx.getAtMostOneItemOfClass(Selection.class);
				selection.setSelection(0, Collections.emptySet());
			}
		});       

	}

	private static Resource resolveModel(ReadGraph graph, Resource diagram) throws DatabaseException {
		Resource model = graph.syncRequest(new PossibleIndexRoot(diagram));
		if (model == null)
			throw new ValidationException("no model found for composite " + NameUtils.getSafeName(graph, diagram));
		return model;
	}

	private static String resolveRVI(ReadGraph graph, Resource model, Resource diagram) throws DatabaseException {
		/*
		 * ModelingResources mod = ModelingResources.getInstance(graph); Resource
		 * composite = graph.getSingleObject(diagram, mod.DiagramToComposite); String
		 * modelURI = graph.getURI(model); String compositeURI =
		 * graph.getURI(composite); return compositeURI.substring(modelURI.length());
		 */

		ModelingResources mod = ModelingResources.getInstance(graph);
		Resource composite = graph.getPossibleObject(diagram, mod.DiagramToComposite);
		if(composite == null)
			return null;
		final ResourceArray compositePath = StructuralVariables.getCompositeArray(graph, composite);
		final ResourceArray variablePath = compositePath.removeFromBeginning(1);
		return StructuralVariables.getRVI(graph, variablePath);
	}

	public Resource getDiagram() {
		return diagram;
	}

	public void dispose() {
		listener.apply("reset", "{}");
		cleanUp();
	}
	
	public boolean isDisposed() {
		return ctx == null;
	}

	private void cleanUp() {
		if (activation != null) {
			activation.deactivate();
			activation = null;
		}
		if (ctx != null) {
			ctx.dispose();
			ctx = null;
		}
		if (provider != null) {
			provider.dispose();
			provider = null;
		}
		if (thread != null) {
			thread.stopDispatchingEvents(false);
			thread = null;
		}
	}

	public ICanvasContext getCanvasContext() {
		return ctx;
	}

	public ICanvasSceneGraphProvider getSceneGraphProvider() {
		return provider;
	}
	
	private void runMutex(Runnable r) {
		ThreadUtils.asyncExec(thread,r);		
	}
	
	public void configureDefaultTransformations(boolean value) {
		defaultTransformations = value;
	}

}
