/*******************************************************************************
 *  Copyright (c) 2023 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.web.serializer;

import java.io.IOException;
import java.util.Iterator;

import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.connection.RouteLine;
import org.simantics.diagram.connection.RouteLink;
import org.simantics.diagram.connection.RouteNode;
import org.simantics.diagram.connection.RoutePoint;
import org.simantics.diagram.connection.RouteTerminal;
import org.simantics.modeling.web.JSONGenerator;
import org.simantics.modeling.web.NodeMapper;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.NullNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;

/**
 * @author Jussi Koskela
 * @since 1.57.0
 */
public class RouteGraphSerializer extends StdSerializer<RouteGraph> {

	private static final long serialVersionUID = -2239328941687246672L;
	private NodeMapper mapper;

	public RouteGraphSerializer() {
		this(null);
	}

	public RouteGraphSerializer(Class<RouteGraph> object) {
		super(object);
	}

	public void setNodeMapper(NodeMapper mapper) {
		this.mapper = mapper;
	}

	@Override
	public void serialize(RouteGraph rg, JsonGenerator jgen, SerializerProvider provider) throws IOException {
		ObjectNode rgNode = new ObjectNode(JsonNodeFactory.instance);

		ArrayNode linkNodes = rgNode.putArray("links");

		ArrayNode routeLineNodes = rgNode.putArray("routeLines");
		for (RouteLine routeLine : rg.getLines()) {
			ObjectNode routeLineNode = routeLineNodes.addObject();
			routeLineNode.put("isHorizonal", routeLine.isHorizontal());
			routeLineNode.put("position", routeLine.getPosition());
			routeLineNode.put("data", (Long) routeLine.getData());

			for (RoutePoint point : routeLine.getPoints()) {
				if (point instanceof RouteLink) {
					RouteLink rl = (RouteLink) point;
					if (rl.getA().equals(routeLine)) {
						ObjectNode linkNode = linkNodes.addObject();
						linkNode.put("n1", RouteGraphSerializer.getRouteNodeId(rl.getA()));
						linkNode.put("n2", RouteGraphSerializer.getRouteNodeId(rl.getB()));
					}
				}
			}
		}

		ArrayNode routeTerminalNodes = rgNode.putArray("routeTerminals");
		for (RouteTerminal routeTerminal : rg.getTerminals()) {
			ObjectNode routeTerminalNode = routeTerminalNodes.addObject();
			routeTerminalNode.put("x", routeTerminal.getX());
			routeTerminalNode.put("y", routeTerminal.getY());
			routeTerminalNode.put("minX", routeTerminal.getMinX());
			routeTerminalNode.put("minY", routeTerminal.getMinY());
			routeTerminalNode.put("maxX", routeTerminal.getMaxX());
			routeTerminalNode.put("maxY", routeTerminal.getMaxY());
			routeTerminalNode.put("allowedDirections", routeTerminal.getAllowedDirections());
			routeTerminalNode.put("data", (Long) routeTerminal.getData());
			routeTerminalNode.put("routeToBounds", routeTerminal.isRouteToBounds());
			routeTerminalNode.set("style",
					routeTerminal.getStyle() != null ? JSONGenerator.toJSON(mapper, routeTerminal.getStyle())
							: NullNode.getInstance());
			ArrayNode lineEndLengths = routeTerminalNode.putArray("lineEndLengths");
			for (int dir = 0; dir < 4; dir++) {
				lineEndLengths
						.add(routeTerminal.getStyle() != null ? routeTerminal.getStyle().getLineEndLength(dir) : 0.0);
			}

			RouteLine line = routeTerminal.getLine();
			if (line != null && rg.getLines().contains(line)) {
				// if (routeTerminal.getLine() != null &&
				// !"_transient_".equals(RouteGraphSerializer.getRouteNodeId(routeTerminal.getLine())))
				// {
				// don't add link to transient line - routeTerminal.getLine().isTransient() does
				// not work with transient line that is connected to two terminals!
				ObjectNode linkNode = linkNodes.addObject();
				linkNode.put("n1", RouteGraphSerializer.getRouteNodeId(routeTerminal));
				linkNode.put("n2", RouteGraphSerializer.getRouteNodeId(routeTerminal.getLine()));
			}
		}

		if (rg.getLines().isEmpty() && rg.getTerminals().size() == 2) {
			ObjectNode linkNode = linkNodes.addObject();
			Iterator<RouteTerminal> iter = rg.getTerminals().iterator();
			linkNode.put("n1", RouteGraphSerializer.getRouteNodeId(iter.next()));
			linkNode.put("n2", RouteGraphSerializer.getRouteNodeId(iter.next()));
		}

		jgen.writeRawValue(mapper.writeValueAsString(rgNode));
	}

	public static String getRouteNodeId(RouteNode node) {
		if (node.getData() != null) {
			return node.getData().toString();
		} else if (node instanceof RouteLine) {
			RouteLine rl = (RouteLine) node;
			if (rl.getData() != null) {
				return rl.getData().toString();
			} else if (rl.getTerminal() != null) {
				String id = rl.getTerminal().getData().toString();
				int count = 0;
				RouteLine next = rl.getNextTransient();
				while (next != null) {
					count++;
					next = next.getNextTransient();
				}
				return id + "_" + count;
			} else {
				return "_transient_"; // handle bug in transient line with two terminals
			}
		} else {
			return null;
		}
	}

	public static String getRoutePointId(RoutePoint node) {
		if (node instanceof RouteTerminal) {
			RouteTerminal t = (RouteTerminal) node;
			return t.getData().toString();
		} else if (node instanceof RouteLink) {
			RouteLink rl = (RouteLink) node;
			return getRouteNodeId(rl.getA()) + "_" + getRouteNodeId(rl.getB());
		} else {
			return null;
		}
	}

}