/*******************************************************************************
 *  Copyright (c) 2023 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.web;

import java.util.HashSet;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.IDiagram.CompositionListener;
import org.simantics.g2d.diagram.participant.AbstractDiagramParticipant;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;

/**
 * @author Jussi Koskela
 * @since 1.57.0
 */
public class DirtyElementsTracker extends AbstractDiagramParticipant {

	private Set<IElement> dirtyElements = new HashSet<>();
	private Set<Long> removedElements = new HashSet<>();
	private IDiagram diagram;

	public synchronized Pair<Set<IElement>, Set<Long>> fetchChanges() {
		Pair<Set<IElement>, Set<Long>> result = new Pair<>(dirtyElements, removedElements);

		dirtyElements = new HashSet<>();
		removedElements = new HashSet<>();
		return result;
	}

	public synchronized void reset() {
		dirtyElements.clear();
		if (diagram != null) {
			for (IElement e : diagram.getElements()) {
				dirtyElements.add(e);
			}
		}
		removedElements.clear();
	}

	class ChangeListener implements CompositionListener, IHintListener {

		@Override
		public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
			if (key == Hints.KEY_DIRTY && newValue == Hints.VALUE_SG_DIRTY && sender instanceof IElement) {
				addDirtyElement((IElement)sender);
			}
		}

		@Override
		public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {

		}

		@Override
		public void onElementAdded(IDiagram d, IElement e) {
			e.addHintListener(this);
			addDirtyElement(e);
		}

		@Override
		public void onElementRemoved(IDiagram d, IElement e) {
			e.removeHintListener(this);
			addRemovedElement(e);
		}
	}

	private final ChangeListener changeListener = new ChangeListener();

	private void addDirtyElement(IElement e) {
		synchronized (DirtyElementsTracker.this) {
			dirtyElements.add(e);
		}
	}

	private void addRemovedElement(IElement e) {
		synchronized (DirtyElementsTracker.this) {
			Resource res = (Resource)ElementUtils.getObject(e);
			if(res != null)
				removedElements.add(res.getResourceId());
		}
	}

	@Override
	protected void onDiagramSet(IDiagram newValue, IDiagram oldValue) {
		diagram = newValue;
		if (oldValue == newValue)
			return;

		if (oldValue != null) {
			for (IElement e : oldValue.getElements()) {
				removeElement(e);
			}
			oldValue.removeCompositionListener(changeListener);
		}

		if (newValue != null) {
			for (IElement e : newValue.getElements()) {
				addElement(e);
				dirtyElements.add(e);
			}
			newValue.addCompositionListener(changeListener);
		}
	}

	protected void addElement(IElement e) {
		e.addKeyHintListener(Hints.KEY_DIRTY, changeListener);
	}

	protected void removeElement(IElement e) {
		e.removeKeyHintListener(Hints.KEY_DIRTY, changeListener);
	}

}