/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.preferences;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.connection.rendering.ConnectionCrossings;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.modeling.ui.Activator;
import org.simantics.utils.page.PageDesc;

/**
 * @author Tuukka Lehtonen
 */
public final class DiagramPreferenceUtil {

    public static final String PLUGIN_ID = Activator.PLUGIN_ID;

    public static DiagramPreferences getDefaultPreferences() {
        return getPreferences(DefaultScope.INSTANCE);
    }

    /**
     * @return
     */
    public static DiagramPreferences getPreferences() {
        return getPreferences(InstanceScope.INSTANCE);
    }

    /**
     * @return
     */
    public static DiagramPreferences getPreferences(IScopeContext context) {
        IEclipsePreferences node = context.getNode(PLUGIN_ID);
        double gridSize = node.getDouble(DiagramPreferences.P_SNAP_GRID_SIZE, DiagramPreferences.DEFAULT_SNAP_GRID_SIZE);
        PageDesc pageDesc = getPageDesc(node);
        //System.out.println("GET pagedesc: " + pageDesc);
        //String pos = node.get(DiagramPreferences.P_DEFAULT_PAGE_ORIENTATION, DiagramPreferences.DEFAULT_PAGE_ORIENTATION.toString());
        //PageOrientation po = PageOrientation.valueOf(pos);

//        Margins margins = DiagramPreferences.DEFAULT_PAGE_MARGINS;
//        String marginString = node.get(DiagramPreferences.P_DEFAULT_PAGE_MARGINS, null);
//        if (marginString != null)
//            margins = PageDesc.deserializeMargins(marginString, margins);
//        pageDesc = pageDesc.withMargins(margins);

        boolean displayPage = node.getBoolean(DiagramPreferences.P_DISPLAY_PAGE_SIZE, DiagramPreferences.DEFAULT_DISPLAY_PAGE_SIZE);
        boolean displayMargins = node.getBoolean(DiagramPreferences.P_DISPLAY_MARGINS, DiagramPreferences.DEFAULT_DISPLAY_MARGINS);
        boolean displayGrid = node.getBoolean(DiagramPreferences.P_DISPLAY_MARGINS, DiagramPreferences.DEFAULT_DISPLAY_GRID);
        boolean displayRuler = node.getBoolean(DiagramPreferences.P_DISPLAY_MARGINS, DiagramPreferences.DEFAULT_DISPLAY_RULER);

        Map<String, Object> map = new HashMap<String, Object>();
        map.put(DiagramPreferences.P_SNAP_GRID_SIZE, gridSize);
        map.put(DiagramPreferences.P_DEFAULT_PAGE_SIZE, pageDesc);
//        map.put(DiagramPreferences.P_DEFAULT_PAGE_ORIENTATION, po);
//        map.put(DiagramPreferences.P_DEFAULT_PAGE_MARGINS, margins);
        map.put(DiagramPreferences.P_DISPLAY_PAGE_SIZE, displayPage);
        map.put(DiagramPreferences.P_DISPLAY_MARGINS, displayMargins);
        map.put(DiagramPreferences.P_DISPLAY_GRID, displayGrid);
        map.put(DiagramPreferences.P_DISPLAY_RULER, displayRuler);

        return new DiagramPreferences(node, map);
    }

    /**
     * @return
     * @throws BackingStoreException
     */
    public static void setPreferences(DiagramPreferences prefs) {
        _setPreferences(InstanceScope.INSTANCE, prefs);
    }

    /**
     * @return
     * @throws BackingStoreException
     */
    public static void flushPreferences(DiagramPreferences prefs) throws BackingStoreException {
        Preferences p = _setPreferences(InstanceScope.INSTANCE, prefs);
        p.flush();
    }

    /**
     * @return
     * @throws BackingStoreException
     */
    public static void setPreferences(IScopeContext context, DiagramPreferences prefs) {
        _setPreferences(context, prefs);
    }

    /**
     * @return
     * @throws BackingStoreException
     */
    private static IEclipsePreferences _setPreferences(IScopeContext context, DiagramPreferences prefs) {
        IEclipsePreferences node = context.getNode(PLUGIN_ID);
        //System.out.println("SET pagedesc: " + prefs.get(DiagramPreferences.P_DEFAULT_PAGE_SIZE));
        node.putDouble(DiagramPreferences.P_SNAP_GRID_SIZE, (Double) prefs.get(DiagramPreferences.P_SNAP_GRID_SIZE));
        node.put(DiagramPreferences.P_DEFAULT_PAGE_SIZE, PageDesc.serialize((PageDesc) prefs.get(DiagramPreferences.P_DEFAULT_PAGE_SIZE)));
//        node.put(DiagramPreferences.P_DEFAULT_PAGE_ORIENTATION, prefs.get(DiagramPreferences.P_DEFAULT_PAGE_ORIENTATION).toString());
//        node.put(DiagramPreferences.P_DEFAULT_PAGE_MARGINS, PageDesc.serialize((Margins) prefs.get(DiagramPreferences.P_DEFAULT_PAGE_MARGINS)));
        node.putBoolean(DiagramPreferences.P_DISPLAY_PAGE_SIZE, (Boolean) prefs.get(DiagramPreferences.P_DISPLAY_PAGE_SIZE));
        node.putBoolean(DiagramPreferences.P_DISPLAY_MARGINS, (Boolean) prefs.get(DiagramPreferences.P_DISPLAY_MARGINS));
        node.putBoolean(DiagramPreferences.P_DISPLAY_GRID, (Boolean) prefs.get(DiagramPreferences.P_DISPLAY_GRID));
        node.putBoolean(DiagramPreferences.P_DISPLAY_RULER, (Boolean) prefs.get(DiagramPreferences.P_DISPLAY_RULER));
        return node;
    }

    private static PageDesc getPageDesc(IEclipsePreferences node) {
        String desc = node.get(DiagramPreferences.P_DEFAULT_PAGE_SIZE, null);
        return PageDesc.deserialize(desc, DiagramPreferences.DEFAULT_PAGE_SIZE);
    }
    
    
    public static void applyDefaultCrossingPreferences(WriteGraph graph, Resource diagram) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        Double currentWidth = graph.getPossibleRelatedValue(diagram, DIA.ConnectionCrossingStyle_Width, Bindings.DOUBLE);
        Resource currentType = graph.getPossibleObject(diagram, DIA.ConnectionCrossingStyle_HasType);
        
        IScopeContext context = InstanceScope.INSTANCE;
        Preferences node = context.getNode(Activator.PLUGIN_ID);
        String typeName = node.get(DiagramPreferences.P_DEFAULT_CONNECTION_CROSSING_STYLE, DiagramPreferences.DEFAULT_CONNECTION_CROSSING_STYLE);
        ConnectionCrossings.Type cctype = ConnectionCrossings.Type.valueOf(typeName);
        Resource typeRes = getConnectionCrossingsType(cctype);
        Double width = node.getDouble(DiagramPreferences.P_DEFAULT_CONNECTION_CROSSING_WIDTH, DiagramPreferences.DEFAULT_CONNECTION_CROSSING_WIDTH);
        
        if (!width.equals(currentWidth) || !typeRes.equals(currentType)) {
            graph.deny(diagram, DIA.ConnectionCrossingStyle_HasType);
            graph.deny(diagram, DIA.ConnectionCrossingStyle_Width);
            graph.claim(diagram, DIA.ConnectionCrossingStyle_HasType, typeRes);
            graph.claimLiteral(diagram, DIA.ConnectionCrossingStyle_Width, width, Bindings.DOUBLE);
        }
    }
    
    public static Resource getConnectionCrossingsType(ConnectionCrossings.Type cctype) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(Simantics.getSession());
        return switch (cctype) {
        case NONE   -> DIA.ConnectionCrossingStyle_Type_None;
        case ARC    -> DIA.ConnectionCrossingStyle_Type_Arc;
        case GAP    -> DIA.ConnectionCrossingStyle_Type_Gap;
        case SQUARE -> DIA.ConnectionCrossingStyle_Type_Square;
        default     -> DIA.ConnectionCrossingStyle_Type_None;
        };
    }

}
