/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - (#7084) page numbering, gitlab #987
 *******************************************************************************/
package org.simantics.modeling.ui.pdf;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreePath;
import org.simantics.db.common.NamedResource;
import org.simantics.db.management.ISessionContext;
import org.simantics.export.core.pdf.PageNumbering;
import org.simantics.modeling.requests.CollectionResult;
import org.simantics.modeling.requests.Node;
import org.simantics.project.IProject;

public class PDFExportPlan {

    public static class StoredNodeSelection {
        public final CollectionResult nodes;
        public final Set<Node> selectedNodeSet;
        public final TreePath[] expandedNodes;

        public StoredNodeSelection(PDFExportPlan from, TreePath[] expandedNodes) {
            this.nodes = from.nodes;
            this.selectedNodeSet = new HashSet<Node>(from.selectedNodeSet);
            this.expandedNodes = expandedNodes;
        }
    }

    // Input
    public ISessionContext           sessionContext;
    public IProject                  project;
    public IStructuredSelection      initialSelection;
    public NamedResource             initialModelSelection;
    public List<NamedResource>       selectableModels = new ArrayList<>();
    public NamedResource             selection;
    public Deque<String>             recentLocations = new LinkedList<>();

    // Temporary data
    public CollectionResult          nodes           = new CollectionResult();
    public Set<Node>                 selectedNodeSet = new HashSet<>();

    public Map<NamedResource, StoredNodeSelection> tempCache = new HashMap<>();

    // Output
    public NamedResource             selectedModel;
    public File                      exportLocation;
    public Collection<Node>          selectedNodes;

    // I/O

    /**
     * <code>false</code> to print content as is regardless of whether it goes
     * outside printed page boundaries. <code>true</code> to scale the content
     * so that it fills the printed page size and the specified margins. If the
     * content goes over page boundaries, it is shrunk and if it doesn't fill
     * the page boundaries, it is scaled up uniformly to fill them either
     * horizontally or vertically.
     */
    public boolean                   fitContentToPageMargins    = false;

    /**
     * <code>true</code> to attach TG of the diagram and the model to the exported
     * PDF.
     */
    public boolean attachTG = false;

    /**
     * Global enable-selector for including diagram in the export. <code>true</code>
     * to include, <code>false</code> to exclude. {@value #includeDiagrams} by
     * default.
     */
    public boolean includeDiagrams = true;

    /**
     * Global enable-selector for including model, library and composite level wiki
     * documentation in the export. <code>true</code> to include, <code>false</code>
     * to exclude. {@value #includeDocumentation} by default.
     */
    public boolean includeDocumentation = true;

    /**
     * Global enable-selector for including composite content, i.e. component level
     * wiki documentation in the export. <code>true</code> to include,
     * <code>false</code> to exclude. {@value #includeComponentLevelDocumentation}
     * by default. The value is only meaningful if {@link #includeDocumentation} is
     * <code>true</code>.
     */
    public boolean includeComponentLevelDocumentation = true;

    /**
     * Whether or not to add page numbers to the exported PDF. Default value is
     * {@value #addPageNumbers}.
     * 
     * @since 1.28.0
     */
    public boolean addPageNumbers = true;

    /**
     * This is ignored if {@link #addPageNumbers} is <code>false</code>.
     */
    public PageNumbering.Position pageNumberPosition = PageNumbering.Position.BOTTOM_RIGHT;

    /**
     * This is ignored if {@link #addPageNumbers} is <code>false</code>.
     */
    public PageNumbering.NumberingFormat pageNumberFormat = PageNumbering.NumberingFormat.PAGE_SLASH_TOTAL_PAGES;


    public PDFExportPlan(ISessionContext sessionContext) {
        this.sessionContext = sessionContext;
    }

}
