package org.simantics.modeling.ui.actions;

import java.awt.geom.Point2D;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.graph.AddElement;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.participant.MouseUtil;
import org.simantics.g2d.participant.MouseUtil.MouseInfo;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.direct.GraphUtils;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.ResourceEditorInput;

/**
 * Base class for shape creation handlers of the symbol editor.
 * 
 * @author Tuukka Lehtonen
 */
public abstract class CreateShapeHandler extends AbstractHandler {

    public abstract String getDefaultElementData();

    public Object addElement(final double mposX, final double mposY, final Resource composite) {
        final String data = getDefaultElementData();
        Simantics.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph g) throws DatabaseException {
                Layer0 L0 = Layer0.getInstance(g);
                DiagramResource dr = DiagramResource.getInstance(g);
                G2DResource g2d = G2DResource.getInstance(g);
                Resource svgElement = GraphUtils.create(g,
                        L0.InstanceOf, dr.SVGElement,
                        g2d.HasSVGDocument, data);
                g.addLiteral(svgElement, dr.HasTransform, dr.IsTransformOf,
                        g2d.Transform,
                        new double[] {1.0, 0.0, 0.0, 1.0, mposX, mposY},
                        Bindings.DOUBLE_ARRAY);
                OrderedSetUtils.add(g, composite, svgElement);
                g.claim(composite, L0.ConsistsOf, svgElement);
                AddElement.claimFreshElementName(g, composite, svgElement);
            }
        });

        return null;
    }

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        IWorkbenchPart ap = HandlerUtil.getActivePart(event);
        IEditorPart viewer = null;
        if (ap instanceof MultiPageEditorPart) {
            MultiPageEditorPart rfe = (MultiPageEditorPart) ap;
            IResourceEditorInput in = (ResourceEditorInput) rfe.getEditorInput();
            final ResourceArray ra = in.getResourceArray();
            ResourceArray symbolEditorInput;
            try {
                symbolEditorInput = Simantics.getSession().syncRequest(new Read<ResourceArray>() {
                    @Override
                    public ResourceArray perform(ReadGraph graph) throws DatabaseException {
                        StructuralResource2 sr = StructuralResource2.getInstance(graph);
                        ModelingResources mr = ModelingResources.getInstance(graph);
                        Resource symbol = graph.getPossibleObject(ra.resources[0], mr.ComponentTypeToSymbol);
                        Resource definedBy = symbol != null ? graph.getPossibleObject(symbol, sr.IsDefinedBy) : null;
                        ResourceArray result = definedBy != null ? new ResourceArray(definedBy) : ResourceArray.EMPTY;
                        return result;
                    }
                });

                IEditorPart[] eps = rfe.findEditors(new ResourceEditorInput("org.simantics.modeling.ui.symbolEditor", symbolEditorInput)); //$NON-NLS-1$
                if (eps.length == 0) {
                    System.out.println("symbol editor part not found from multi page editor part: " + ap); //$NON-NLS-1$
                    return null;
                }
                viewer = eps[0];
            } catch (DatabaseException e) {
                e.printStackTrace();
                return null;
            }
        } else if (ap instanceof IEditorPart) {
            viewer = (IEditorPart) ap;
        } else {
            return null;
        }
        ICanvasContext ctx = (ICanvasContext) viewer.getAdapter(ICanvasContext.class);
        if (ctx == null) {
            System.out.println("No canvas context"); //$NON-NLS-1$
            return null;
        }
        MouseInfo minfo = ctx.getSingleItem(MouseUtil.class).getMousePressedInfo(0);
        if(minfo == null) {
            System.out.println("No mouse info"); //$NON-NLS-1$
            return null;
        }
        final Point2D mpos = minfo.canvasPosition;
        IResourceEditorInput input = (IResourceEditorInput)viewer.getEditorInput();
        Resource composite = input.getResource();

        return addElement(mpos.getX(), mpos.getY(), composite);
    }

}