package org.simantics.modeling.ui;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.HandlerUtil;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.diagramEditor.DiagramEditor;
import org.simantics.modeling.ui.diagramEditor.DiagramViewer;
import org.simantics.ui.workbench.dialogs.ResourceSelectionDialog3;
import org.simantics.utils.datastructures.Pair;

public class ModelingUIUtils {

    public static DiagramViewer tryGetDiagramViewer(ExecutionEvent event) {

    	IEditorPart editorPart = HandlerUtil.getActiveEditor(event);
    	if(editorPart == null) return null;
    	if(editorPart instanceof DiagramEditor) {
    		return (DiagramViewer)editorPart.getAdapter(DiagramViewer.class);
    	}
    	
    	return null;
    	
    }
    
    public static ICanvasContext tryGetCanvasContext(ExecutionEvent event) {
    	
    	DiagramViewer viewer = tryGetDiagramViewer(event);
    	if(viewer != null) return (ICanvasContext)viewer.getAdapter(ICanvasContext.class);
    	else return null;
    	
    }
	
	protected static void findSCLQueryTypes(ReadGraph graph, Resource indexRoot, Map<Resource, Pair<String, ImageDescriptor>> map) throws DatabaseException {

		Layer0 L0 = Layer0.getInstance(graph);
		ModelingResources MOD = ModelingResources.getInstance(graph);

		Instances query = graph.adapt(MOD.SCLQueryType, Instances.class);

		String modelURI = graph.getURI(indexRoot);
		
		for(Resource _res : query.find(graph, indexRoot)) {
			
			// Don't allow instantiation of abstract query types.
			if (graph.hasStatement(_res, L0.Abstract))
				continue;

			Resource res = graph.getPossibleObject(_res, L0.HasRange_Inverse);
			if(res == null) {
				
				// Entry type

				String name = graph.getPossibleRelatedValue(_res, L0.HasName, Bindings.STRING);
				if (name == null)
					continue;
				String label = graph.getPossibleRelatedValue2(_res, L0.HasLabel, Bindings.STRING);

				if (label != null && !name.equals(label)) {
					name = label + " (" + name + ")"; //$NON-NLS-1$ //$NON-NLS-2$
				}

				Resource parent = graph.getPossibleObject(_res, L0.PartOf);
				if(parent == null) continue;

				String parentURI = graph.getURI(parent);
				if(parentURI.startsWith(modelURI)) {
					parentURI = parentURI.substring(modelURI.length());
					if(parentURI.startsWith("/")) parentURI = parentURI.substring(1); //$NON-NLS-1$
				}

				name = name + " - " + URIStringUtils.unescape(parentURI); //$NON-NLS-1$

				map.put(_res, new Pair<String, ImageDescriptor>(name, null));
				
			} else {
				
				// Property type
			
				String name = graph.getPossibleRelatedValue(res, L0.HasName, Bindings.STRING);
				if (name == null)
					continue;
				String label = graph.getPossibleRelatedValue2(res, L0.HasLabel, Bindings.STRING);

				if (label != null && !name.equals(label)) {
					name = label + " (" + name + ")"; //$NON-NLS-1$ //$NON-NLS-2$
				}

				Resource parent = graph.getPossibleObject(_res, L0.PartOf);
				if(parent == null) continue;

				String parentURI = graph.getURI(parent);
				if(parentURI.startsWith(modelURI)) {
					parentURI = parentURI.substring(modelURI.length());
					if(parentURI.startsWith("/")) parentURI = parentURI.substring(1); //$NON-NLS-1$
				}

				name = name + " - " + URIStringUtils.unescape(parentURI); //$NON-NLS-1$

				map.put(_res, new Pair<String, ImageDescriptor>(name, null));
			
			}
			
		}

	}
	
	public static void queryUserSelectedQueryType(
			final Map<Resource, Pair<String, ImageDescriptor>> map,
			final Consumer<Resource> selectionCallback)
	{
		Display.getDefault().asyncExec(new Runnable() {
			@Override
			public void run() {
				Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
				ResourceSelectionDialog3<Resource> dialog = new ResourceSelectionDialog3<Resource>(shell, map, Messages.ModelingUIUtils_SelectQueryType) {
					@Override
					protected IDialogSettings getBaseDialogSettings() {
						return Activator.getDefault().getDialogSettings();
					}
				};
				if (dialog.open() == Window.OK) {
					Object[] result = dialog.getResult();
					if (result != null && result.length == 1) {
						final Resource res = (Resource)result[0];
						selectionCallback.accept(res);
					}
				}
			}
		});
	}
	
	public static void newSCLQuery(final Resource parent, final Resource indexRoot) throws DatabaseException {
        if (indexRoot == null)
            return;
        Simantics.getSession().syncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                
                final Map<Resource, Pair<String, ImageDescriptor>> map = new HashMap<Resource, Pair<String,ImageDescriptor>>();
                findSCLQueryTypes(graph, indexRoot, map);
                queryUserSelectedQueryType(map, selected -> {
                    Simantics.getSession().async(new WriteRequest() {
                        @Override
                        public void perform(WriteGraph g) throws DatabaseException {
                            g.markUndoPoint();
                            Simantics.applySCL("Simantics/Query", "createSCLQueryDefault", g, parent, selected); //$NON-NLS-1$ //$NON-NLS-2$
                        }
                    });
                });
            }
        });
    }
    
}
