package org.simantics.modeling.utils;

import java.util.Collection;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.CommonDBUtils;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ModelingUtils;
import org.simantics.modeling.adapters.ChangeHistoryUpdated;
import org.simantics.modeling.adapters.ChangeInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class OntologicalRequirementEnforceRequest extends WriteRequest {

	private static final Logger LOGGER = LoggerFactory.getLogger(OntologicalRequirementEnforceRequest.class);

	private static final String PROP_WRITE_CHANGED_TAGS = "org.simantics.modeling.writeChangedTags"; //$NON-NLS-1$

	private Collection<Resource> creates;
	private Collection<Resource> modis;
	private Collection<Resource> ids;
	private String author;
	private long time;

	private static String getAuthor() {
		return System.getProperty("user.name", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	private static boolean writeChangedTags() {
		return !System.getProperty(PROP_WRITE_CHANGED_TAGS, "") //$NON-NLS-1$
				.equalsIgnoreCase("false"); //$NON-NLS-1$
	}

	public OntologicalRequirementEnforceRequest(Collection<Resource> creates, Collection<Resource> modis, Collection<Resource> ids) {
		this(creates,
				modis,
				ids,
				getAuthor(),
				System.currentTimeMillis());
	}

	public OntologicalRequirementEnforceRequest(Collection<Resource> creates, Collection<Resource> modis, Collection<Resource> ids, String author, long time) {
		this.creates = creates;
		this.modis = modis;
		this.ids = ids;
		this.author = author;
		this.time = time;
	}

	@Override
	public void perform(WriteGraph graph) throws DatabaseException {
		update(graph, creates, modis, ids, true, author, time, true, writeChangedTags(), true);
	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			boolean disableDependencyIndexing) throws DatabaseException
	{
		update(graph,
				creates,
				modis,
				ids,
				addComment,
				getAuthor(),
				System.currentTimeMillis(),
				disableDependencyIndexing,
				writeChangedTags(),
				true);
	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			boolean disableDependencyIndexing,
			boolean setChangeHistoryUpdated) throws DatabaseException
	{
		update(graph,
				creates,
				modis,
				ids,
				addComment,
				getAuthor(),
				System.currentTimeMillis(),
				disableDependencyIndexing,
				writeChangedTags(),
				setChangeHistoryUpdated);
	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			String author,
			long time,
			boolean disableDependencyIndexing) throws DatabaseException
	{
		update(graph, creates, modis, ids, addComment, author, time, disableDependencyIndexing, writeChangedTags(), true);
	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			String author,
			long time,
			boolean disableDependencyIndexing,
			boolean writeChangedTags) throws DatabaseException
	{
		update(graph, creates, modis, ids, addComment, author, time, disableDependencyIndexing, writeChangedTags, true);
	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			String author,
			long time,
			boolean disableDependencyIndexing,
			boolean writeChangedTags,
			boolean setChangeHistoryUpdated) throws DatabaseException
	{
		ModelingResources MOD = ModelingResources.getInstance(graph);
		Layer0 L0 = Layer0.getInstance(graph);

		boolean hasCreates = !creates.isEmpty();
		boolean hasModis = !modis.isEmpty();

		boolean trace = LOGGER.isTraceEnabled();
		if (trace)
			LOGGER.trace("update(disable dependency indexing={}, hasCreates={}, hasModis={})", disableDependencyIndexing, hasCreates, hasModis);

		int addedChangedInfos = 0;
		int updatedChangedInfos = 0;

		if (hasCreates) {
			var info = new ChangeInformation();
			info.createdAt = time;
			info.createdBy = author;
			info.modifiedAt = time;
			info.modifiedBy = author;

			for (Resource c : creates) {
				CommonDBUtils.selectClusterSet(graph, c);
				graph.claimLiteral(c, MOD.changeInformation, MOD.changeInformation_Inverse, MOD.ChangeInformation, info, ChangeInformation.BINDING);
				++addedChangedInfos;
			}
		}

		if (hasModis) {
			for (Resource m : modis) {
				ChangeInformation info = graph.getPossibleRelatedValue(m, MOD.changeInformation, ChangeInformation.BINDING);
				if (info == null) {
					// Should not be possible but lets handle this anyway
					info = new ChangeInformation();
					info.createdAt = time;
					info.createdBy = author;
					++addedChangedInfos;
				} else {
					++updatedChangedInfos;
				}
				info.modifiedAt = time;
				info.modifiedBy = author;
				CommonDBUtils.selectClusterSet(graph, m);
				graph.claimLiteral(m, MOD.changeInformation, MOD.changeInformation_Inverse, MOD.ChangeInformation, info, ChangeInformation.BINDING);
			}
		}

		if (writeChangedTags && (hasCreates || hasModis)) {
			ModelingUtils.markChanged(graph, creates, modis);
		}

		int idCount = 0;
		for (Resource r : ids) {
			if (!graph.hasStatement(r, L0.identifier)) {
				CommonDBUtils.selectClusterSet(graph, r);
				Layer0Utils.claimNewIdentifier(graph, r, true);
				if (trace)
					LOGGER.trace("added identifier for {}", NameUtils.getURIOrSafeNameInternal(graph, r));
				++idCount;
			}
		}

		// Only disable indexing if no identifiers were actually added.
		if (disableDependencyIndexing && idCount == 0) {
			Layer0Utils.setDependenciesIndexingDisabled(graph, true);
		}

		if (addComment) {
			var cm = graph.getMetadata(CommentMetadata.class);
			if (hasCreates || hasModis) {
				cm.add("Added " + addedChangedInfos + " and updated " + updatedChangedInfos + " change informations");
			}
			if (idCount > 0) {
				cm.add("Added " + idCount + " missing identifiers");
			}
			graph.addMetadata( cm ); //$NON-NLS-1$
		}
		if (setChangeHistoryUpdated)
			graph.addMetadata( graph.getMetadata(ChangeHistoryUpdated.class) );
	}

}