package org.simantics.modeling.utils;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.CommonDBUtils;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.scl.runtime.function.Function;
import org.simantics.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.list.array.TByteArrayList;

public class DumpOntologyStructure {

    private static final Logger LOGGER = LoggerFactory.getLogger(DumpOntologyStructure.class);

    private Resource ontology;

    private Map<Resource, String> names = new HashMap<>();
    private Map<Resource,Resource> parents = new TreeMap<>();
    private Map<Resource, File> libraryFolders = new HashMap<>();
    private Map<Resource, byte[]> contentDumps = new HashMap<>();

    private void readNameAndParent(ReadGraph graph, Resource container, Resource r) throws DatabaseException {
        String name = NameUtils.getSafeName(graph, r);
        parents.put(r, container);
        names.put(r, FileUtils.escapeFileName(name));
    }

    /*
     * This shall return containers sorted by full path.
     * This makes processing order stable and ensures that
     * directories are processed before their contents. 
     */
    private Collection<Resource> sortedContainers(File rootFolder) {
        Set<Resource> parentSet = new HashSet<Resource>(parents.values());
        TreeMap<String,Resource> result = new TreeMap<>();
        for(Resource r : parentSet) {
            File f = getFolder(rootFolder, r);
            result.put(f.getAbsolutePath(), r);
        }
        return result.values();
    }

    private Collection<Resource> sortedResources(File rootFolder) {
        TreeMap<String,Resource> result = new TreeMap<>();
        for(Resource r : parents.keySet()) {
            byte[] dump = contentDumps.get(r);
            if(dump == null)
                dump = "".getBytes(StandardCharsets.UTF_8);
            if(isParent(r)) {
                if(dump.length > 0) {
                    File f = new File(getFolder(rootFolder, r), "__contents__");
                    result.put(f.getAbsolutePath(), r);
                }
            } else {
                File f = getFile(rootFolder, r);
                result.put(f.getAbsolutePath(), r);
            }
        }
        return result.values();
    }

    private void readHierarchy(ReadGraph graph, Resource container) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        for(Resource r : CommonDBUtils.objectsWithType(graph, container, L0.ConsistsOf, L0.Entity)) {
            try {
                readNameAndParent(graph, container, r);
                readHierarchy(graph, r);
            } catch (DatabaseException e) {
                LOGGER.error("Error while reading content dump hierarchy for " + r, e);
            }
        }
    }

    private void readGeneric(ReadGraph graph) throws DatabaseException {
        ModelingResources MOD = ModelingResources.getInstance(graph);
        for(Resource r : parents.keySet()) {
            if(contentDumps.containsKey(r))
                continue;
            TByteArrayList result = new TByteArrayList();
            try {
                TreeMap<String,Resource> sortedTypes = new TreeMap<>();
                for(Resource type : graph.getTypes(r)) {
                    String uri = graph.getPossibleURI(type);
                    if(uri != null)
                        sortedTypes.put(uri, type);
                }
                for(Resource type : sortedTypes.values()) {
                    try {
                        Variable typeVariable = Variables.getVariable(graph, type);
                        @SuppressWarnings("rawtypes")
                        Function f = typeVariable.getPossiblePropertyValue(graph, MOD.contentDumpFunction);
                        if(f != null) {
                            @SuppressWarnings("unchecked")
                            byte[] dump = (byte[])Simantics.applySCLRead(graph, f, r);
                            if(dump != null) {
                                result.add(dump);
                            }
                        }
                    } catch (DatabaseException e) {
                        LOGGER.error("Error while computing content dump for " + r, e);
                    }
                }
                if(result.size() > 0)
                    contentDumps.put(r, result.toArray());
            } catch (DatabaseException e) {
                LOGGER.error("Error while computing content dump for " + r, e);
            }
        }
    }

    public DumpOntologyStructure read(ReadGraph graph, Resource ontology) throws DatabaseException {
        this.ontology = ontology;
        readHierarchy(graph, ontology);
        readGeneric(graph);
        return this;
    }

    private File escapeFile(File file) {
        if(file.exists())
            return file;
        return new File(escapeFile(file.getParentFile()), FileUtils.escapeFileName(file.getName()));
    }

    public void write(File unsafeFolder) throws IOException {
        File folder = escapeFile(unsafeFolder);
        FileUtils.delete(folder.toPath());
        folder.getParentFile().mkdirs();
        writeDirectories(folder);
        writeResources(folder);
    }

    Resource getParent(Resource r) {
        return parents.get(r);
    }
    
    private File getFolder(File root, Resource library) {
        if(ontology.equals(library))
            return root;
        Resource parent = getParent(library);
        if(parent == null)
            throw new IllegalStateException("null parent for " + library);
        File parentFolder = getFolder(root, parent);
        return new File(parentFolder, names.get(library)); 
    }

    private File getFile(File rootFolder, Resource r) {
        Resource parent = getParent(r);
        File folder = getFolder(rootFolder, parent);
        return new File(folder, names.get(r));
    }

    private File makeUnique(File original, Resource r) {
        int counter = 2;
        File file = new File(original.getParent(), original.getName());
        File test = file;
        while(test.exists()) {
            // Here we have a name clash with small and big letters! (windows)
            test = new File(file.getParent(), file.getName() + "____" + (counter++));
        }
        // Enforce this renaming in future operations also
        names.put(r, test.getName());
        return test;
    }
    
    private void writeDirectories(File rootFolder) {
        // Here stuff shall be returned in alphabetical order
        for(Resource library : sortedContainers(rootFolder)) {
            File folder = makeUnique(getFolder(rootFolder, library), library);
            folder.mkdirs();
            libraryFolders.put(library, folder);
        }
    }

    private void writeResources(File rootFolder) throws IOException {
        // Here stuff shall be returned in alphabetical order
        for(Resource r : sortedResources(rootFolder)) {
            writeResource(rootFolder, r);
        }
    }

    private boolean isParent(Resource r) {
        return parents.values().contains(r);
    }

    private void writeResource(File rootFolder, Resource resource) throws IOException {
        byte[] dump = contentDumps.get(resource);
        if(dump == null)
            dump = "".getBytes(StandardCharsets.UTF_8);
        if(isParent(resource)) {
            if(dump.length > 0)
                FileUtils.writeFile(new File(getFolder(rootFolder, resource), "__contents__"), dump);
        } else {
            write(rootFolder, resource, dump);
        }
    }

    private void write(File rootFolder, Resource resource, byte[] bytes) throws IOException {
        FileUtils.writeFile(makeUnique(getFile(rootFolder, resource), resource), bytes);
    }

}