/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.subscription;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableReference;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;

/**
 * @author Jani Simomaa <jani.simomaa@semantum.fi>
 */
public class SCLSubscription {
	
	/**
     * Creates new folder under given models subscription folder
     * 
     * @param model resource of the target model
     * @param name name of the folder
     * @return resource of the folder
     * @throws DatabaseException on database failures
     */
	public static Resource addSubscriptionFolder (WriteGraph graph, Resource model, String name) throws DatabaseException {
		
		NewSubscription folder = new NewSubscription(model, name);
		folder.perform(graph);
		return folder.subscription;
	}
	
	/**
     * Adds subscription item with default plotting values to the given subscription resource (folder). 
     * 
     * @param subscriptionresource resource of the subscription folder
     * @param variable variable of the item to be created
     * @return resource of the subscription item
     * @throws DatabaseException on database failures
     */
	public static Resource addSubscriptionItems (WriteGraph graph, Resource subscriptionresource, Variable variable) throws DatabaseException {
				
		List<VariableReference> references = (List<VariableReference>)variable.getPossiblePropertyValue(graph, "variableReferences");
		if(references == null) {
			VariableReference ref = new VariableReference(variable.getRVI(graph), variable.getDatatype(graph), null);
			references = new ArrayList<VariableReference>(1);
			references.add(ref);
		}
		AddSubscriptionItems subitems = new AddSubscriptionItems(subscriptionresource, references);
		
		subitems.perform(graph);
		return subitems.result.get(subitems.result.size() -1);
	}
	
	/**
     * Adds subscription item to the given subscription resource (folder) with advanced plotting values that user can define.
     * 
     * @param subscriptionresource resource of the subscription folder
     * @param variable variable of the item to be created
     * @param interval
     * @param deadband
     * @param gain
     * @param bias
     * @param unit
     * @param label
     * @return resource of the subscription item
     * @throws DatabaseException on database failures
     */
	public static Resource newSubscriptionItem (WriteGraph graph, Resource subscriptionresource, Variable variable, double interval, double deadband, double gain, double bias, String unit, String label) throws DatabaseException {
		return newSubscriptionItem(graph, subscriptionresource, null, variable, interval, deadband, gain, bias, unit, label);
	}

	public static Resource newSubscriptionItem (WriteGraph graph, Resource subscriptionresource, String itemId, Variable variable, double interval, double deadband, double gain, double bias, String unit, String label) throws DatabaseException {
		
		NewSubscriptionItem ns = new NewSubscriptionItem(
				subscriptionresource, 
				interval, 
				deadband, 
				gain, 
				bias, 
				unit, 
				label,
				itemId,
				variable.getRVI(graph),
				variable.getDatatype(graph)
				);
		
		ns.perform(graph);
		
		return ns.subscriptionItem;
	}
	
	/**
     * Returns the default subscription folder of the given model. If folder with label Default is not found, returns some other subscription folder
     * 
     * @param model resource of the target model
     * @return resource of the subscription folder
     * @throws DatabaseException on database failures
     */
	public static Resource defaultSubscriptionFolder (ReadGraph graph, Resource model) throws DatabaseException {
		
		Layer0 l0 = Layer0.getInstance(graph);
		ModelingResources MR = ModelingResources.getInstance(graph);
		Collection<Resource> childrens = graph.getObjects(model, l0.ConsistsOf);
		Resource folder = null;
		for (final Resource children : childrens) {
			
			if (graph.isInstanceOf(children, MR.Subscription)) {
				folder = children;
				String label = graph.getRelatedValue2(folder, l0.HasLabel);
				if (label.equals("Default")) {
					return folder;
				}
			}			
		}
		return folder;
	}
	
	public static String getLabel(ReadGraph graph, Resource subscription) throws DatabaseException {
		
		return SubscriptionItemLabel.resolveLabel(graph, subscription, true);		
	}
}
