package org.simantics.graph.store;

import gnu.trove.set.hash.THashSet;

import java.util.regex.Pattern;

import org.simantics.graph.query.Path;
import org.simantics.graph.query.PathChild;
import org.simantics.graph.query.Paths;

public class PathPattern {
	
	private static PathPattern EMPTY_PATTERN = new PathPattern(null, null);
	
	Path prefix;
	Pattern suffix;
	
	private PathPattern(Path prefix, Pattern suffix) {
		this.prefix = prefix;
		this.suffix = suffix;
	}

	public static PathPattern compile(String pattern) {
		pattern = stripPatternPrefix(pattern);
		if(pattern == null)
			return EMPTY_PATTERN;
		String[] parts = pattern.split("/");
		
		Path path = Paths.Root;
		for(int i=0;i<parts.length;++i) {
			String part = parts[i];
			if(containsWildcards(part)) {
				StringBuilder b = new StringBuilder(pattern.length());
				for(;i<parts.length;++i) {
					b.append('/');
					b.append(parts[i]);
				}
				return new PathPattern(path, compileGlobPattern(b.toString()));
			}
			else
				path = new PathChild(part, path);
		}
		
		return new PathPattern(path, null);
	}
	
	private static String patternStart = "http://";
	private static String stripPatternPrefix(String pattern) {
		for(int i=0;i<patternStart.length();++i) {
			if(pattern.length() <= i)
				return null;
			char c = pattern.charAt(i);
			if(c == '*')
				return pattern.substring(i);
			if(c != patternStart.charAt(i) && c != '?')
				return null;
		}
		return pattern.substring(patternStart.length());
	}
	
	private static boolean containsWildcards(String pattern) {
		return pattern.contains("*") || pattern.contains("?");
	}
	
	private static Pattern compileGlobPattern(String pattern) {
		int length = pattern.length();
		StringBuilder b = new StringBuilder(2*length);
		b.append("\\Q");
		for(int i=0;i<length;++i) {
			char c = pattern.charAt(i);
			switch(c) {
			case '*':
				b.append("\\E.*\\Q");
				break;
			case '?':
				b.append("\\E.\\Q");
				break;
			case '\\':
				++i;
				if(i < length) {
					c = pattern.charAt(i);
					if(c == '\\' && i+1 < length && pattern.charAt(i+1) == 'E') {
						++i;
						b.append("\\E\\\\E\\Q");
					}
					else if(c == '*' || c == '?')
						b.append(c);
					else {
						b.append('\\');
						b.append(c);
					}
				}
				break;
			default:
				b.append(c);
			}
		}
		b.append("\\E");
		return Pattern.compile(b.toString());
	}
	
	@Override
	public String toString() {
		return "(" + prefix + ", " + suffix + ")";
	}
	
	public void search(IdentityStore store, THashSet<Path> result) {
		int id = store.pathToId(prefix);
		if(id == -1)
			return;
		store.findChildren(id, prefix, "", suffix, result);
	}
	
	public static void main(String[] args) {
		System.out.println(compile("http://www.simantics.org/*/foo"));
	}
}
