package org.simantics.graph.query;

import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.hash.THashMap;
import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

import java.util.Collection;

import org.simantics.graph.representation.External;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.Internal;
import org.simantics.graph.representation.Optional;
import org.simantics.graph.representation.Root;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.representation.Value;
import org.simantics.graph.store.GraphStore;
import org.simantics.graph.store.IStore;
import org.simantics.graph.store.IdentityStore;
import org.simantics.graph.store.StatementStore;
import org.simantics.graph.store.ValueStore;


public class TransferableGraphConversion {
	private static StatementStore extractStatements(TransferableGraph1 tg) {
		StatementStore statements = new StatementStore();
		int[] array = tg.statements;
		int i=0;
		while(i<array.length) {
			int subject = array[i++];
			int predicate = array[i++];
			int inverse = array[i++];
			int object = array[i++];
			statements.add(subject, predicate, object);
			if(inverse >= 0)
				statements.add(object, inverse, subject);
		}
		return statements;
	}
	
	public static IdentityStore extractIdentities(TransferableGraph1 tg) {
	    return extractIdentities(tg.resourceCount, tg.identities);
	}
	
	private static IdentityStore extractIdentities(int resourceCount, Identity[] ids) {
	    IdentityStore identities = new IdentityStore();
        identities.setResourceCount(resourceCount);
        for(Identity identity : ids) {
            if(identity.definition instanceof Root) {
                Root def = (Root)identity.definition;
                identities.defineRoot(def.name, identity.resource);
            }
            else if(identity.definition instanceof External) {
                External def = (External)identity.definition;
                identities.defineChild(def.parent, def.name, identity.resource);
            }
            else if(identity.definition instanceof Optional) {
                Optional def = (Optional)identity.definition;
                identities.defineChild(def.parent, def.name, identity.resource);
            }
            else if(identity.definition instanceof Internal) {
                Internal def = (Internal)identity.definition;
                identities.defineChild(def.parent, def.name, identity.resource);
            }
        }
        return identities;	    
	}
	
	private static ValueStore extractValues(TransferableGraph1 tg) {
		ValueStore values = new ValueStore();
		for(Value value : tg.values)
			values.setValue(value.resource, value.value);
		return values;
	}
	
	public static GraphStore convert(TransferableGraph1 tg) {
		return new GraphStore(
				extractStatements(tg),
				extractIdentities(tg),
				extractValues(tg),
				new THashMap<Class<?>, IStore>()
				);
	}

	public static TransferableGraph1 convert(final IGraph cg, final GraphStore store) {
	
		// Create inverse map
		final TIntIntHashMap inverseMap = new TIntIntHashMap();
		final TIntHashSet withoutInverse = new TIntHashSet();
		final Paths paths = cg.getPaths();

		store.statements.getPredicates().forEach(new TIntProcedure() {
			@Override
			public boolean execute(int id) {
				for(Res inverse : cg.rawGetObjects(store.idToRes(id), paths.InverseOf)) {					
					int inv = store.createResToId(inverse);
					inverseMap.put(id, inv);
					inverseMap.put(inv, id);
					return true;
				}
				withoutInverse.add(id);
				return true;
			}
		});		
		
		if(!withoutInverse.isEmpty()) {
			IdentityStore identities = store.identities;
			StatementStore statements = store.statements;
			int inverseOfId = identities.pathToId(paths.InverseOf);
			if(inverseOfId >= 0)
				for(int id=0;id<store.identities.getResourceCount();++id) {
					TIntArrayList invs = statements.getObjects(id, inverseOfId);
					if(!invs.isEmpty()) {
						int inv = invs.getQuick(0);
						inverseMap.put(id, inv);
						inverseMap.put(inv, id);
					}
				}
		}
		
		// Create tg
		TransferableGraph1 tg = new TransferableGraph1(
				store.identities.getResourceCount(),
				store.identities.toArray(),
				store.statements.toArray(inverseMap),
				store.values.toArray()
				);
		
		return tg;
	}
	
	public static TransferableGraph1 convert(final GraphStore store) {		
		
		// Create tg
		TransferableGraph1 tg = new TransferableGraph1(
				store.identities.getResourceCount(),
				store.identities.toArray(),
				store.statements.toArray(),
				store.values.toArray()
				);
		
		return tg;
	}
	
	public static CompositeGraph convert(Paths paths, Collection<TransferableGraph1> tgs) {
		CompositeGraph graph = new CompositeGraph(paths);
		for(TransferableGraph1 tg : tgs)
			graph.addFragment(convert(tg));
		return graph;
	}
	
}
