/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbollibrary.ui;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.contexts.IContextService;
import org.eclipse.ui.part.IContributedContentsView;
import org.eclipse.ui.part.IPage;
import org.eclipse.ui.part.Page;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.part.PageBookView;
import org.simantics.Simantics;
import org.simantics.db.management.ISessionContextProvider;
import org.simantics.diagram.symbolcontribution.EmptySymbolProviderFactory;
import org.simantics.diagram.symbolcontribution.SymbolProviderFactory;
import org.simantics.utils.datastructures.BijectionMap;

/**
 * This paged view shows {@link SymbolLibraryPage}s for editors based on the
 * {@link ISymbolProviderFactory} the editors adapt to. ISymbolProvider equality is
 * used for deciding what pages to create/assign for activated workbench parts.
 * Only editors are considered in page creation.
 * 
 * <p>
 * To get a symbol palette for your editor, add the following to your
 * {@link IEditorPart#getAdapter(Class)} implementation:
 * 
 * <pre>
 *     if (c == ISymbolProviderFactory.class) {
 *         try {
 *             return DiagramTypeUtils.readSymbolProviderFactory(RequestProcessor, Resource diagram);
 *         } catch (DatabaseException e) {
 *             // TODO: Log error
 *             return null;
 *         }
 *     }
 * </pre>
 * 
 * @author Tuukka Lehtonen
 */
public class SymbolPageView extends PageBookView {

    private static final String                SYMBOL_VIEW_CONTEXT = "org.simantics.diagram.symbolLibrary";

    private ISessionContextProvider            contextProvider;

    private IMemento                           memento;

    BijectionMap<SymbolProviderFactory, IPage> providerPageMap     = new BijectionMap<SymbolProviderFactory, IPage>();

    @Override
    public void createPartControl(Composite parent) {
        super.createPartControl(parent);

        IContextService cs = (IContextService) getSite().getService(IContextService.class);
        cs.activateContext(SYMBOL_VIEW_CONTEXT);
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Object getViewAdapter(Class adapter) {
        if (adapter == IContributedContentsView.class) {
            // This makes it possible to duplicate a PropertyPageView with another
            // secondary ID and make it show the same property page that was showing
            // in the original property page view.
            return new IContributedContentsView() {
                @Override
                public IWorkbenchPart getContributingPart() {
                    return getContributingEditor();
                }
            };
        }
        if (adapter == ISessionContextProvider.class)
            return contextProvider;
        return super.getViewAdapter(adapter);
    }

    /**
     * Returns the editor which contributed the current
     * page to this view.
     *
     * @return the editor which contributed the current page
     * or <code>null</code> if no editor contributed the current page
     */
    private IWorkbenchPart getContributingEditor() {
        return getCurrentContributingPart();
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.part.ViewPart#init(org.eclipse.ui.IViewSite, org.eclipse.ui.IMemento)
     */
    @Override
    public void init(IViewSite site, IMemento memento) throws PartInitException {
        this.memento = memento;
        init(site);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.part.PageBookView#init(org.eclipse.ui.IViewSite)
     */
    @Override
    public void init(IViewSite site) throws PartInitException {
        super.init(site);

        // This prevents the view from providing a selection to other
        // workbench parts, thus making them lose their selections which is not
        // desirable.
        site.setSelectionProvider(null);

        contextProvider = Simantics.getSessionContextProvider();
    }

    @Override
    public void saveState(IMemento memento) {
        if (this.memento != null) {
            memento.putMemento(this.memento);
        }
    }

    /* (non-Javadoc)
     * Method declared on IWorkbenchPart.
     */
    @Override
    public void dispose() {
//        System.out.println("dispose: " + this);
        // Dispose of this before nullifying contextProvider because this
        // dispose may just need the context provider - at least PropertyPage
        // disposal will.
        super.dispose();

        contextProvider = null;
    }

    @Override
    protected IPage createDefaultPage(PageBook book) {
        Page page = new SymbolLibraryPage(EmptySymbolProviderFactory.INSTANCE);
        initPage(page);
        page.createControl(getPageBook());
        return page;
//        MessagePage page = new MessagePage();
//        initPage(page);
//        page.createControl(book);
//        page.setMessage(Messages.SymbolPageView_noSymbolsAvailable);
//        return page;
    }

    @Override
    protected PageRec doCreatePage(IWorkbenchPart part) {
        SymbolProviderFactory providerFactory = (SymbolProviderFactory) part.getAdapter(SymbolProviderFactory.class);
        if (providerFactory == null)
            return null;

        IPage page = providerPageMap.getRight(providerFactory);
        if (page == null) {
            //System.out.println("create new symbol library page for part " + part + " and symbol provider factory " + providerFactory);
            Page newPage = new SymbolLibraryPage(providerFactory);
            initPage(newPage);
            newPage.createControl(getPageBook());

            providerPageMap.map(providerFactory, newPage);
            page = newPage;
        }
        return new PageRec(part, page);
    }

    @Override
    protected void doDestroyPage(IWorkbenchPart part, PageRec pageRecord) {
        IPage page = pageRecord.page;
        @SuppressWarnings("unused")
        SymbolProviderFactory provider = providerPageMap.removeWithRight(page);
        //System.out.println("destroy page " + page + " for symbol provider " + provider + " by closing part " + part);
        page.dispose();
        pageRecord.dispose();
    }

    @Override
    protected IWorkbenchPart getBootstrapPart() {
        IWorkbenchPage page = getSite().getPage();
        if (page != null) {
            IEditorPart editor = page.getActiveEditor();
            if (editor != null)
                return editor;
            return page.getActivePart();
        }
        return null;
    }

    @Override
    protected boolean isImportant(IWorkbenchPart part) {
        // Ignore all but editors.
        return part instanceof IEditorPart;
    }

    /* (non-Javadoc)
     * Method declared on IViewPart.
     * Treat this the same as part activation.
     */
    @Override
    public void partBroughtToTop(IWorkbenchPart part) {
        partActivated(part);
    }

//    @Override
//    public void partActivated(IWorkbenchPart part) {
//        IWorkbenchPart source = getSourcePart(part);
//        //System.out.println("part activated: " + part + ", " + source);
//        super.partActivated(source);
//    }
//
//    private IWorkbenchPart getSourcePart(IWorkbenchPart part) {
//        IContributedContentsView view = (IContributedContentsView) part.getAdapter(IContributedContentsView.class);
//        if (view != null) {
//            IWorkbenchPart source = view.getContributingPart();
//            if (source != null)
//                return source;
//        }
//        return part;
//    }

}
