/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.handler;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Shape;
import java.awt.geom.Area;
import java.awt.geom.Path2D;
import java.util.Collection;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.SGDesignation;
import org.simantics.g2d.canvas.impl.AbstractCanvasParticipant;
import org.simantics.g2d.canvas.impl.SGNodeReflection.SGCleanup;
import org.simantics.g2d.canvas.impl.SGNodeReflection.SGInit;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.SelectionOutline;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.nodes.ShapeNode;
import org.simantics.scenegraph.g2d.nodes.SingleElementNode;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.threads.ThreadUtils;

/**
 * This participant highlights the specified DiagramSelection on its source
 * diagram as a stroked frame tightly surrounding the selected elements.
 * 
 * @author Tuukka Lehtonen
 */
public class HighlightMode extends AbstractCanvasParticipant implements IHintListener {

    private final Color  CUT_COLOR  = Color.ORANGE;
    private final Color  COPY_COLOR = new Color(128, 220, 150);

    private final BasicStroke DEFAULT_STROKE = new BasicStroke(3.0f);

    DiagramSelection     selection;
    int                  selectionId;
    int                  paintPriority;

    SingleElementNode    highlightNode;
    Collection<IElement> highlightElements;

    public HighlightMode(DiagramSelection selection, int selectionId, int paintPriority) {
        this.selection = selection;
        this.selectionId = selectionId;
        this.paintPriority = paintPriority;
    }

    @SGInit(designation = SGDesignation.CANVAS)
    public void init(G2DParentNode parent) {
        highlightNode = parent.addNode("cut/copy source highlight", SingleElementNode.class);
        highlightNode.setZIndex(paintPriority);
        highlightNode.setVisible(false);

        // This slows rendering down too much to use.
        //highlightNode.setComposite(AlphaComposite.SrcOver.derive(0.4f));

        this.highlightElements = selection.getOriginalElements();

        for (IElement e : highlightElements)
            e.addHintListener(this);

        paintSelectionFrames(highlightNode, selection, selectionId);
    }

    @SGCleanup
    public void cleanupSG() {
        for (IElement e : highlightElements)
            e.removeHintListener(this);

        highlightNode.remove();
    }

    void paintSelectionFrames(G2DParentNode parentNode, DiagramSelection selection, int selectionId) {
        Area cutArea = new Area();
        for (IElement e : selection.getOriginalElements()) {
            ElementClass ec = e.getElementClass();
            SelectionOutline so = ec.getAtMostOneItemOfClass(SelectionOutline.class);
            Shape shape = so != null ? so.getSelectionShape(e) : ElementUtils.getElementShapeOrBoundsOnDiagram(e);
            cutArea.add(new Area(shape));
        }
        if (!cutArea.isEmpty()) {
            // Is rendering of Area slower than Path2D?
            Path2D.Double path = new Path2D.Double(cutArea);

            ShapeNode shapeNode = parentNode.getOrCreateNode("highlight", ShapeNode.class);
            shapeNode.setShape(path);
            shapeNode.setScaleStroke(true);
            shapeNode.setStroke(DEFAULT_STROKE);
            shapeNode.setFill(false);
            shapeNode.setColor(selection.isCut() ? CUT_COLOR : COPY_COLOR);

            highlightNode.setVisible(true);
        }
    }

    @Override
    public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
        if (key == ElementHints.KEY_TRANSFORM) {
            //System.out.println("transform changed for " + sender + " to " + newValue);
            highlightNode.setVisible(false);
            deferredHighlightUpdate();
        }
    }

    @Override
    public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {
    }

    Future<?> updateTask;

    private synchronized void deferredHighlightUpdate() {
        if (updateTask == null) {
            updateTask = ThreadUtils.getNonBlockingWorkExecutor().schedule(
                    updateScheduler,
                    100,
                    TimeUnit.MILLISECONDS);
        }
    }

    Runnable updateScheduler = new Runnable() {
        @Override
        public void run() {
            ICanvasContext ctx = getContext();
            if (ctx != null)
                ThreadUtils.asyncExec(ctx.getThreadAccess(), painter);
        }
    };

    Runnable painter = new Runnable() {
        @Override
        public void run() {
            if (!isRemoved() && highlightNode.getRootNode() != null)
                paintSelectionFrames(highlightNode, selection, selectionId);
            updateTask = null;
        }
    };

}