package org.simantics.db;

import java.io.File;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Properties;
import java.util.function.Consumer;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.SDBException;
import org.simantics.db.service.ClusterUID;

public interface Database {
    static public enum Status {
        NoDatabase { @Override public String getString() { return "NoDatabase"; }},
        NotRunning { @Override public String getString() { return "NotRunning"; }},
        Standalone { @Override public String getString() { return "Standalone"; }},
        Local { @Override public String getString() { return "Local"; }},
        Remote { @Override public String getString() { return "Remote";}};
        public abstract String getString();
        @Override
        public String toString() {
            return message;
        }
        public String message = "";
    }
    public DatabaseUserAgent getUserAgent();
    public void setUserAgent(DatabaseUserAgent dbUserAgent);
    public Status getStatus();

    public File getFolder();
    public boolean isFolderOk();
    public boolean isFolderOk(File aFolder);
    public boolean isFolderEmpty();
    public boolean isFolderEmpty(File aFolder);
    public void initFolder(Properties properties) throws SDBException;
    public void deleteFiles() throws SDBException;

    public void start() throws SDBException;
    public boolean isRunning() throws SDBException;
    public boolean tryToStop() throws SDBException;

    public void connect() throws SDBException;
    public boolean isConnected() throws SDBException;
    public String execute(String command) throws SDBException;
    public void disconnect() throws SDBException;

    public void clone(File to, int revision, boolean saveHistory) throws SDBException;
    public Path createFromChangeSets(int revision) throws SDBException;
    public void deleteGuard() throws SDBException;
    public Path dumpChangeSets() throws SDBException;
    public void purgeDatabase(Consumer<Collection<ClusterUID>> callback) throws SDBException;
    public long serverGetTailChangeSetId() throws SDBException;

    public interface Session {
        public Database getDatabase(); // Convenience method.
        public void close() throws SDBException;
        public void open() throws SDBException;
        public boolean isClosed() throws SDBException;
        public interface ChangeSetData {
            public boolean isOk();
        }
        public interface ChangeSetIds {
            public long getFirstChangeSetId();
            public int getCount();
        }
        public interface Cluster {
            public int getInflateSize();
            public ByteBuffer getDeflated();
        }
        public interface ClusterChanges {
            public long getHeadChangeSetId();
            public int[] getResourceIndex();
            public int[] getPredicateIndex();
            public long[] getPredicateFirst();
            public long[] getPredicateSecond();
            public int[] getValueIndex();
        }
        public interface ClusterIds {
            public int getStatus();
            public long[] getFirst();
            public long[] getSecond();
        }
        public interface Information {
            public String getServerId();
            public String getProtocolId();
            public String getDatabaseId();
            public long getFirstChangeSetId();
        }
        public interface Refresh {
            public long getHeadChangeSetId();
            public long[] getFirst();
            public long[] getSecond();
        }
        public interface ResourceSegment {
            public byte[] getClusterId();
            public int getResourceIndex();
            public long getValueSize(); // Size of whole value.
            public byte[] getSegment(); // Bytes of asked segment.
            public long getOffset(); // Segment offset.
        }
        public interface Transaction {
            public long getHeadChangeSetId();
            public long getTransactionId();
        }
        interface OnChangeSetUpdate {
            public void onChangeSetUpdate(ChangeSetUpdate changeSetUpdate) throws SDBException;
        }
        public void acceptCommit(long transactionId, long changeSetId, byte[] metadata) throws SDBException;
        public long cancelCommit(long transactionId, long changeSetId, byte[] metadata, OnChangeSetUpdate onChangeSetUpdate) throws SDBException;
        public Transaction askReadTransaction() throws SDBException;
        public Transaction askWriteTransaction(long transactionId) throws SDBException;
        public long endTransaction(long transactionId) throws SDBException; // Return head change set id.
        public String execute(String command) throws SDBException;
        public byte[] getChangeSetMetadata(long changeSetId) throws SDBException;
        public ChangeSetData getChangeSetData(long minChangeSetId, long  maxChangeSetId, OnChangeSetUpdate onChangeSetupate) throws SDBException;
        public ChangeSetIds getChangeSetIds() throws SDBException;
        public Cluster getCluster(byte[] clusterId) throws SDBException;
        public ClusterChanges getClusterChanges(long changeSetId, byte[] clusterId) throws SDBException;
        public ClusterIds getClusterIds() throws SDBException;
        public Information getInformation() throws SDBException;
        public Refresh getRefresh(long changeSetId) throws SDBException;
        public ResourceSegment getResourceSegment(byte[] clusterUID, int resourceIndex, long offset, short size) throws SDBException;
        public long reserveIds(int count) throws SDBException;
        public void updateCluster(byte[] operations) throws SDBException;
        public interface ChangeSetUpdate {
            public long getChangeSetId();
            public int getChangeSetIndex();
            public int getNumberOfClusterChangeSets();
            public int getIndexOfClusterChangeSet();
            public byte[] getClusterId();
            public boolean getNewCluster();
            public byte[] getData();
        }
        public boolean undo(long[] changeSetIds, OnChangeSetUpdate onChangeSetUpdate) throws SDBException;
        
        public <T> T clone(ClusterUID clusterUID, ClusterCreator clusterCreator) throws DatabaseException;
        public boolean refreshEnabled();
        public boolean rolledback();
    }
    public Session newSession(ServiceLocator locator) throws SDBException;
    interface Journal {
        public class Line {
            public boolean status;
            public String request;
            public String comment;
        }
        public boolean canRead();
        public int count();
        public int read(int index, Line line) throws SDBException;
    }
    public Journal getJournal() throws SDBException;
    public String getCompression();
}
