package org.simantics.db.layer0.request;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead2;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.layer0.Layer0;
import org.simantics.simulation.ontology.SimulationResource;

/**
 * Returns a Name->Resource map for instances of given type in a given model.
 * 
 * Uses Dependencies-index for obtaining instances. Uses L0.HasName for
 * obtaining name.
 * 
 * @author Antti Villberg
 */
public class PossibleModelInstances extends ResourceRead2<Map<String, Resource>> {

    /**
     * @parameter modelPart is a Resource with URI under a model e.g. model can
     *            be found using L0.PartOf
     * @parameter type is the type to search instances for. The query returns
     *            properly all instances of all subtypes.
     */
    public PossibleModelInstances(Resource modelPart, Resource type) {
        super(modelPart, type);
    }

    private Resource getModel(ReadGraph graph) throws DatabaseException {
        SimulationResource SIMU = SimulationResource.getInstance(graph);
        if (graph.isInstanceOf(resource, SIMU.Model)) {
            return resource;
        } else {
            return graph.sync(new PossibleModel(resource));
        }
    }

    @Override
    public Map<String, Resource> perform(ReadGraph graph) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);

        Instances query = graph.getPossibleAdapter(resource2, Instances.class);
        if (query == null)
            return Collections.emptyMap();

        Resource model = getModel(graph);
        if (model == null)
            return Collections.emptyMap();

        Collection<Resource> queryResult = query.find(graph, model);
        if (queryResult.isEmpty())
            return Collections.emptyMap();

        Map<String, Resource> result = new HashMap<String, Resource>(queryResult.size());
        for(Resource instance : queryResult) {
            String name = graph.getPossibleRelatedValue(instance, L0.HasName, Bindings.STRING);
            if (name != null)
                result.put(name, instance);
        }
        return result;

    }

}
