/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.core.runtime.Platform;
import org.simantics.databoard.Bindings;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.DirectStatements;
import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.ReadGraph;
import org.simantics.db.RelationInfo;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.VirtualGraph;
import org.simantics.db.common.procedure.adapter.AsyncMultiProcedureAdapter;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.NoInverseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.graph.BarrierTracing;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.graph.ReadGraphSupport;
import org.simantics.db.impl.procedure.IntProcedureAdapter;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.impl.procedure.TripleIntProcedureAdapter;
import org.simantics.db.impl.support.ResourceSupport;
import org.simantics.db.procedure.AsyncMultiListener;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.AsyncSetListener;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.StatementProcedure;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.QueryFactoryKey;
import org.simantics.db.request.RequestFlags;
import org.simantics.layer0.Layer0;
import org.simantics.utils.DataContainer;
import org.simantics.utils.Development;
import org.simantics.utils.FileUtils;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.collections.CollectionUtils;
import org.simantics.utils.datastructures.disposable.AbstractDisposable;
import org.slf4j.LoggerFactory;

import gnu.trove.procedure.TIntProcedure;
import gnu.trove.procedure.TLongProcedure;
import gnu.trove.procedure.TObjectProcedure;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

@SuppressWarnings({"rawtypes", "unchecked"})
final public class QueryProcessor extends AbstractDisposable implements ReadGraphSupport {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(QueryProcessor.class);

	public static int                                       indent                = 0;

	final private int                                       functionalRelation;

	final private int                                       superrelationOf;

	final private int                                       instanceOf;

	final private int                                       inverseOf;

	final private int                                       asserts;

	final private int                                       hasPredicate;

	final private int                                       hasPredicateInverse;

	final private int                                       hasObject;

	final private int                                       inherits;

	final private int                                       subrelationOf;

	final private int                                       rootLibrary;

	/**
	 * A cache for the root library resource. Initialized in
	 * {@link #getRootLibraryResource()}.
	 */
	private volatile ResourceImpl                           rootLibraryResource;

	final private int                                       library;

	final private int                                       consistsOf;

	final private int                                       hasName;

	AtomicInteger                                       sleepers = new AtomicInteger(0);

	boolean                                         updating              = false;


	final public QueryCache                                 cache;
	final public QuerySupport                               querySupport;
	final public Session                                    session;
	final public ResourceSupport                            resourceSupport;
	
	final public Semaphore                                  requests = new Semaphore(1);
	
	final public QueryListening                            listening = new QueryListening(this);

	QueryThread[]                                   executors;

	enum ThreadState {

		INIT, RUN, SLEEP, DISPOSED

	}

	final Scheduling scheduling;
	
	public ThreadState[]									threadStates;
	
	final Object querySupportLock;
	
	public Long modificationCounter = 0L;

	Set<ExternalReadEntry<?>> updatedPrimitivesInCurrentWrite = new HashSet<>();

	public void close() {
	}


    /*
     * We are running errands while waiting for requests to complete.
     * We can only run work that is part of the current root request to avoid any deadlocks
     */
    public boolean performPending(ReadGraphImpl under) {
        SessionTask task = scheduling.getSubTask(under);
		if(task != null) {
			task.run(thread.get());
			return true;
		}
		return false;
	}
    
    final public void scheduleNow(SessionTask request) {
        SessionTask toExecute = scheduleOrReturnForExecution(request);
        if(toExecute != null)
            toExecute.run(thread.get());
    }

    final public SessionTask scheduleOrReturnForExecution(SessionTask request) {
        
        return scheduling.scheduleOrReturnForExecution(request);

    }


	final int THREADS;
	final public int  THREAD_MASK;

	final public static ThreadGroup QueryThreadGroup = new ThreadGroup("Query Thread Group");

	public static abstract class SessionTask {

	    final protected ReadGraphImpl rootGraph;
		private int counter = 0;
		protected int position = 1;
		private Exception trace;

		public SessionTask() {
		    this(null);
		}
		
        public SessionTask(ReadGraphImpl rootGraph) {
            this.rootGraph = rootGraph;
        }
        
        public boolean isSubtask(ReadGraphImpl graph) {
            return graph.isParent(rootGraph);
        }

        public abstract void run0(int thread);

		public final void run(int thread) {
		    if(counter++ > 0) {
		        if(BarrierTracing.BOOKKEEPING) {
		            trace.printStackTrace();
		            new Exception().printStackTrace();
		        }
		        throw new IllegalStateException("Multiple invocations of SessionTask!");
		    }
		    if(BarrierTracing.BOOKKEEPING) {
		        trace = new Exception();
		    }
		    run0(thread);
		}
		
		public boolean maybeReady() {
			return true;
		}

		@Override
		public String toString() {
			if(rootGraph == null)
				return "SessionTask[no graph]";
			else
				return "SessionTask[" + rootGraph.parent + "]";
		}

	}

	public static abstract class SessionRead extends SessionTask {

		final public Semaphore notify;
		final public DataContainer<Throwable> throwable; 

		public SessionRead(DataContainer<Throwable> throwable, Semaphore notify) {
			super(null);
			this.throwable = throwable;
			this.notify = notify;
		}

	}

	public QueryProcessor(final int threads, QuerySupport core, Set<Thread> threadSet)
			throws DatabaseException {

		THREADS = threads;
		THREAD_MASK = threads - 1;

		scheduling = new Scheduling(requests);
		
		querySupport = core;
		cache = new QueryCache(core, threads);
		session = querySupport.getSession();
		resourceSupport = querySupport.getSupport();
		querySupportLock = core.getLock();

		executors = new QueryThread[THREADS];
		threadStates = new ThreadState[THREADS];

		for (int i = 0; i < THREADS; i++) {
			threadStates[i] = ThreadState.INIT;
		}

		for (int i = 0; i < THREADS; i++) {

			final int index = i;

			executors[i] = new QueryThread(session, this, index, "Query Thread " + index);

			threadSet.add(executors[i]);

		}

		// Now start threads
		for (int i = 0; i < THREADS; i++) {
			executors[i].start();
		}

		// Make sure that query threads are up and running
		while(sleepers.get() != THREADS) {
			try {
				Thread.sleep(5);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}

		rootLibrary = core.getBuiltin("http:/");
		boolean builtinsInstalled = rootLibrary != 0;

		if (builtinsInstalled) {
			functionalRelation = core.getBuiltin(Layer0.URIs.FunctionalRelation);
			assert (functionalRelation != 0);
		} else
			functionalRelation = 0;

		if (builtinsInstalled) {
			instanceOf = core.getBuiltin(Layer0.URIs.InstanceOf);
			assert (instanceOf != 0);
		} else
			instanceOf = 0;

		if (builtinsInstalled) {
			inverseOf = core.getBuiltin(Layer0.URIs.InverseOf);
			assert (inverseOf != 0);
		} else
			inverseOf = 0;


		if (builtinsInstalled) {
			inherits = core.getBuiltin(Layer0.URIs.Inherits);
			assert (inherits != 0);
		} else
			inherits = 0;

		if (builtinsInstalled) {
			asserts = core.getBuiltin(Layer0.URIs.Asserts);
			assert (asserts != 0);
		} else
			asserts = 0;

		if (builtinsInstalled) {
			hasPredicate = core.getBuiltin(Layer0.URIs.HasPredicate);
			assert (hasPredicate != 0);
		} else
			hasPredicate = 0;

		if (builtinsInstalled) {
			hasPredicateInverse = core.getBuiltin(Layer0.URIs.HasPredicateInverse);
			assert (hasPredicateInverse != 0);
		} else
			hasPredicateInverse = 0;

		if (builtinsInstalled) {
			hasObject = core.getBuiltin(Layer0.URIs.HasObject);
			assert (hasObject != 0);
		} else
			hasObject = 0;

		if (builtinsInstalled) {
			subrelationOf = core.getBuiltin(Layer0.URIs.SubrelationOf);
			assert (subrelationOf != 0);
		} else
			subrelationOf = 0;

		if (builtinsInstalled) {
			superrelationOf = core.getBuiltin(Layer0.URIs.SuperrelationOf);
			assert (superrelationOf != 0);
		} else
			superrelationOf = 0;

		if (builtinsInstalled) {
			library = core.getBuiltin(Layer0.URIs.Library);
			assert (library != 0);
		} else
			library = 0;

		if (builtinsInstalled) {
			consistsOf = core.getBuiltin(Layer0.URIs.ConsistsOf);
			assert (consistsOf != 0);
		} else
			consistsOf = 0;

		if (builtinsInstalled) {
			hasName = core.getBuiltin(Layer0.URIs.HasName);
			assert (hasName != 0);
		} else
			hasName = 0;

	}

	final public void releaseWrite(ReadGraphImpl graph) {
		propagateChangesInQueryCache(graph);
		modificationCounter++;
	}

	final public int getId(final Resource r) {
		return querySupport.getId(r);
	}

	public QuerySupport getCore() {
		return querySupport;
	}

	public int getFunctionalRelation() {
		return functionalRelation;
	}

	public int getInherits() {
		return inherits;
	}

	public int getInstanceOf() {
		return instanceOf;
	}

	public int getInverseOf() {
		return inverseOf;
	}

	public int getSubrelationOf() {
		return subrelationOf;
	}

	public int getSuperrelationOf() {
		return superrelationOf;
	}

	public int getAsserts() {
		return asserts;
	}

	public int getHasPredicate() {
		return hasPredicate;
	}

	public int getHasPredicateInverse() {
		return hasPredicateInverse;
	}

	public int getHasObject() {
		return hasObject;
	}

	public int getRootLibrary() {
		return rootLibrary;
	}

	public Resource getRootLibraryResource() {
		if (rootLibraryResource == null) {
			// Synchronization is not needed here, it doesn't matter if multiple
			// threads simultaneously set rootLibraryResource once.
			int root = getRootLibrary();
			if (root == 0)
				throw new UnsupportedOperationException("database is not initialized, cannot get root library resource");
			this.rootLibraryResource = new ResourceImpl(querySupport.getSupport(), root);
		}
		return rootLibraryResource;
	}

	public int getLibrary() {
		return library;
	}

	public int getConsistsOf() {
		return consistsOf;
	}

	public int getHasName() {
		return hasName;
	}

	public void forResource(ReadGraphImpl graph, final String id, CacheEntry parent, final InternalProcedure<Integer> procedure) {

		try {
			
			QueryCache.runnerURIToResource(graph, id, parent, null, new InternalProcedure<Integer>() {

				@Override
				public void execute(ReadGraphImpl graph, Integer result) throws DatabaseException {

					if (result != null && result != 0) {
						procedure.execute(graph, result);
						return;
					}

					// Fall back to using the fixed builtins.
//					result = querySupport.getBuiltin(id);
//					if (result != 0) {
//						procedure.execute(graph, result);
//						return;
//					} 

//					try {
//						result = querySupport.getRandomAccessReference(id);
//					} catch (ResourceNotFoundException e) {
//						procedure.exception(graph, e);
//						return;
//					}

					if (result != 0) {
						procedure.execute(graph, result);
					} else {
						procedure.exception(graph, new ResourceNotFoundException(id));
					}

				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
					procedure.exception(graph, t);
				}

			});
		} catch (DatabaseException e) {
		    
		    try {
		        
                procedure.exception(graph, e);
                
            } catch (DatabaseException e1) {
                
                Logger.defaultLogError(e1);
                
            }
		    
		}

	}

	public void forBuiltin(ReadGraphImpl graph, final String id, CacheEntry parent, final InternalProcedure<Integer> procedure) throws DatabaseException {

		Integer result = querySupport.getBuiltin(id);
		if (result != 0) {
			procedure.execute(graph, result);
		} else {
			procedure.exception(graph, new ResourceNotFoundException(id));
		}

	}

	final <T> void runMultiRead(final ReadGraphImpl graph, MultiReadEntry cached, final MultiRead<T> query, final CacheEntry parent, final QueryProcessor provider, final ListenerBase listener, final SyncMultiProcedure<T> procedure) {

		try {
			QueryCache.runnerMultiReadEntry(graph, query, parent, listener, procedure);
		} catch (DatabaseException e) {
			throw new IllegalStateException(e);
		}

	}

	public final <T> void runAsyncMultiRead(final ReadGraphImpl graph, final AsyncMultiRead<T> query, final CacheEntry parent, final ListenerBase listener, final AsyncMultiProcedure<T> procedure) {

		
		try {
			QueryCache.runnerAsyncMultiReadEntry(graph, query, parent, listener, procedure);
		} catch (DatabaseException e) {
			throw new IllegalStateException(e);
		}

	}

	final <T> void runPrimitiveRead(ReadGraphImpl graph, ExternalReadEntry cached, final ExternalRead<T> query, final CacheEntry parent, final QueryProcessor provider, final ListenerBase listener, final AsyncProcedure<T> procedure) throws DatabaseException {
		QueryCache.runnerExternalReadEntry(graph, query, parent, listener, procedure);
	}

//    @Override
//	public <T> T query(final ReadGraphImpl graph, final Read<T> query, final CacheEntry parent, final AsyncProcedure<T> procedure, final ListenerBase listener) throws DatabaseException {
//    	
//    	return QueryCache.resultReadEntry(graph, query, parent, listener, procedure);
//
//	}

	public <T> void queryMultiRead(final ReadGraphImpl graph, final MultiRead<T> query, final CacheEntry parent, final ListenerBase listener, final SyncMultiProcedure<T> procedure) throws DatabaseException {

		QueryCache.runnerMultiReadEntry(graph, query, parent, listener, procedure);

	}

	public <T> void queryPrimitiveRead(final ReadGraphImpl graph, final ExternalRead<T> query, final CacheEntry parent, final ListenerBase listener, final AsyncProcedure<T> procedure) throws DatabaseException {

		QueryCache.runnerExternalReadEntry(graph, query, parent, listener, procedure);

	}

	boolean isBound(ExternalReadEntry<?> entry) {
		if(listening.hasParents(entry)) return true;
		else if(listening.hasListener(entry)) return true;
		else return false;
	}

	static class Dummy implements InternalProcedure<Object>, IntProcedure {

		@Override
		public void execute(ReadGraphImpl graph, int i) {
		}

		@Override
		public void finished(ReadGraphImpl graph) {
		}

		@Override
		public void execute(ReadGraphImpl graph, Object result) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}
		
	}
	
	private static final Dummy dummy = new Dummy();

	/*
    public <Procedure> Object performForEach2(ReadGraphImpl graph, UnaryQuery<Procedure> query, CacheEntry parent, ListenerBase listener, Procedure procedure) throws Throwable {

        if (DebugPolicy.PERFORM)
            System.out.println("PE[ " + (query.hashCode() &  THREAD_MASK) + "] " + query);

        assert (!dirty);
        assert (!collecting);

        assert(query.assertNotDiscarded());

        registerDependencies(graph, query, parent, listener, procedure, false);

        // FRESH, REFUTED, EXCEPTED go here 
        if (!query.isReady()) {

            size++;
            misses++;

            query.computeForEach(graph, this, (Procedure)dummy, true);
            return query.get(graph, this, null);

        } else {

            hits++;

            return query.get(graph, this, procedure);

        }

    }
	*/
	

	interface QueryCollectorSupport {
		public CacheCollectionResult allCaches();
		public Collection<CacheEntry<?>> getRootList();
		public int getCurrentSize();
		public long getActivity();
		public int calculateCurrentSize();
		public CacheEntryBase iterate(int level);
		public void remove();
		public void setLevel(CacheEntryBase entry, short level);
		public boolean start(boolean flush);
	}

	interface QueryCollector {

		public void collect(int youngTarget, int allowedTimeInMs);

	}

	class QueryCollectorSupportImpl implements QueryCollectorSupport {

		private static final boolean DEBUG = false;
		private static final double ITERATION_RATIO = 0.2;
		
		private CacheCollectionResult iteration = new CacheCollectionResult();
		private boolean fresh = true;
		private boolean needDataInStart = true;
		
		QueryCollectorSupportImpl() {
			iteration.restart();
		}

		public CacheCollectionResult allCaches() {
			CacheCollectionResult result = new CacheCollectionResult();
			QueryProcessor.this.allCaches(result);
			result.restart();
			cache.resetUpdates();
			return result;
		}
		
		public boolean start(boolean flush) {
			// We need new data from query maps
			fresh = true;
			if(needDataInStart || flush) {
				// Last run ended after processing all queries => refresh data
				restart(flush ? 0.0 : ITERATION_RATIO);
			} else {
				// continue with previous big data
			}
			// Notify caller about iteration situation
			return iteration.isAtStart();
		}

		/*
		 * Uses the targetRatio to determine whether to use existing data
		 */
		private void restart(double targetRatio) {
			
			needDataInStart = true;

			long start = System.nanoTime();
			if(fresh) {
				
				// We need new data from query maps
				
				int iterationSize = iteration.size()+1;
				int diff = calculateCurrentSize()-iterationSize;
				
				diff += getActivity();
				
				double ratio = (double)diff / (double)iterationSize;
				boolean dirty = Math.abs(ratio) >= targetRatio;
				
				if(dirty) {
					iteration = allCaches();
					if(DEBUG) {
						System.err.print("iterate: allCaches in " + 1e-9*(System.nanoTime()-start) + "s. (" + iteration.size() + ") ");
						for(int i=0;i<CacheCollectionResult.LEVELS;i++)
							System.err.print(" " + iteration.levels[i].size());
						System.err.println("");
					}
				} else {
					iteration.restart();
				}
				
				fresh = false;
				needDataInStart = false;
			} else {
				// We are returning here within the same GC round - reuse the cache table
				iteration.restart();
			}
			
			return;
			
		}
		
		@Override
		public CacheEntryBase iterate(int level) {
			
			CacheEntryBase entry = iteration.next(level);
			if(entry == null) {
				restart(ITERATION_RATIO);
				return null;
			}
			
			while(entry != null && entry.isDiscarded()) {
				entry = iteration.next(level);
			}
			
			return entry;
			
		}
		
		@Override
		public void remove() {
			iteration.remove();
		}
		
		@Override
		public void setLevel(CacheEntryBase entry, short level) {
			iteration.setLevel(entry, level);
		}

		public Collection<CacheEntry<?>> getRootList() {
			return cache.getRootList();
		}

		@Override
		public int calculateCurrentSize() {
			return cache.calculateCurrentSize();
		}

		@Override
		public int getCurrentSize() {
			return cache.size;
		}

		@Override
		public long getActivity() {
			return cache.updates;
		}
		
	}
	//    final private static int MINIMUM_SIZE = (int)(Runtime.getRuntime().maxMemory() / 600);

	private QueryCollectorSupport collectorSupport = new QueryCollectorSupportImpl();
	private QueryCollector collector = new QueryCollectorImpl(this, collectorSupport);

    public int querySize() {
        return cache.size;
    }

	public void gc(int youngTarget, int allowedTimeInMs) {

		collector.collect(youngTarget, allowedTimeInMs);

	}


	void processParentReport(CacheEntry entry, Map<CacheEntry, Set<CacheEntry>> workarea) {

		if(entry.isDiscarded()) return;
		if(workarea.containsKey(entry)) return;
		
		HashSet<CacheEntry> ps = new HashSet<CacheEntry>();
		listening.forParents(entry, parent -> {
			if(parent.isDiscarded())
				return;
			ps.add(parent);
			processParentReport(parent, workarea);
		});
		workarea.put(entry, ps);

	}

	public synchronized String reportQueryActivity(File file) throws IOException {
		
		System.err.println("reportQueries " + file.getAbsolutePath());

		if (!isAlive())
			return "Disposed!";

		PrintStream b = new PrintStream(new BufferedOutputStream(new FileOutputStream(file)));

		List<Pair<String,Integer>> entries = CollectionUtils.valueSortedEntries(Development.histogram);
		Collections.reverse(entries);
		
		for(Pair<String,Integer> entry : entries) {
			b.println(entry.first + ": " + entry.second);
		}

		b.close();
		
		Development.histogram.clear();

		return "OK";

	}
	
	public synchronized String reportQueries(File file) throws IOException {

		System.err.println("reportQueries " + file.getAbsolutePath());

		if (!isAlive())
			return "Disposed!";

		PrintStream b = new PrintStream(new BufferedOutputStream(new FileOutputStream(file)));

		long start = System.nanoTime();

//		ArrayList<CacheEntry> all = ;
		
		Map<CacheEntry, Set<CacheEntry>> workarea = new HashMap<CacheEntry, Set<CacheEntry>>();
		Collection<CacheEntryBase> caches = allCaches(new CacheCollectionResult()).toCollection();
		for(CacheEntryBase entry : caches) {
			processParentReport(entry, workarea);
		}
		
		//        for(CacheEntry e : all) System.err.println("entry: " + e);

		long duration = System.nanoTime() - start;
		System.err.println("Query root set in " + 1e-9*duration + "s.");

		start = System.nanoTime();

		HashMap<CacheEntry, Integer> flagMap = new HashMap<CacheEntry, Integer>(); 

		int listeners = 0;

		for(CacheEntry entry : workarea.keySet()) {
			boolean listener = listening.hasListenerAfterDisposing(entry);
			boolean hasParents = listening.hasParents(entry);
			if(listener) {
				// Bound
				flagMap.put(entry, 0);
			} else if (!hasParents) {
				// Unbound
				flagMap.put(entry, 1);
			} else {
				// Unknown
				flagMap.put(entry, 2);
			}
			//        	// Write leaf bit
			//        	entry.flags |= 4;
		}

		boolean done = true;
		int loops = 0;

		do {

			done = true;

			long start2 = System.nanoTime();

			int boundCounter = 0;
			int unboundCounter = 0;
			int unknownCounter = 0;

			for(CacheEntry<?> entry : workarea.keySet()) {

				//System.err.println("process " + entry);

				int flags = flagMap.get(entry);
				int bindStatus = flags & 3;

				if(bindStatus == 0) boundCounter++;
				else if(bindStatus == 1) unboundCounter++;
				else if(bindStatus == 2) unknownCounter++;

				if(bindStatus < 2) continue;

				int newStatus = 1;
				List<CacheEntry> parents = listening.getParents(entry);
				for(int i=0;i<parents.size();i++) {
					CacheEntry parent = parents.get(i);

					if(parent.isDiscarded()) flagMap.put(parent, 1);

					int flags2 = flagMap.get(parent);
					int bindStatus2 = flags2 & 3;
					// Parent is bound => child is bound
					if(bindStatus2 == 0) {
						newStatus = 0;
						break;
					}
					// Parent is unknown => child is unknown
					else if (bindStatus2 == 2) {
						newStatus = 2;
						done = false;
						break;
					}
				}

				flagMap.put(entry, newStatus);

			}

			duration = System.nanoTime() - start2;
			System.err.println("Query analysis pass (" + boundCounter + "/" + unboundCounter + "/" + unknownCounter + ") in "+ 1e-9*duration + "s.");
			b.println("Query analysis pass (" + boundCounter + "/" + unboundCounter + "/" + unknownCounter + ") in "+ 1e-9*duration + "s.");

		} while(!done && loops++ < 20);

		if(loops >= 20) {

			for(CacheEntry entry : workarea.keySet()) {

				int bindStatus = flagMap.get(entry);
				if(bindStatus == 2) System.err.println("Undefined bind status for " + entry);

			}

		}

		duration = System.nanoTime() - start;
		System.err.println("Query analysis in " + 1e-9*duration + "s.");

		Map<Class<?>, Integer> counts = new HashMap<Class<?>, Integer>();

		for(CacheEntry entry : workarea.keySet()) {
			Class<?> clazz = entry.getClass();
			if(entry instanceof ReadEntry) clazz = ((ReadEntry)entry).id.getClass(); 
			else if(entry instanceof MultiReadEntry) clazz = ((MultiReadEntry)entry).id.getClass(); 
			else if(entry instanceof AsyncReadEntry) clazz = ((AsyncReadEntry)entry).id.getClass(); 
			else if(entry instanceof AsyncMultiReadEntry) clazz = ((AsyncMultiReadEntry)entry).id.getClass(); 
			else if(entry instanceof ExternalReadEntry) clazz = ((ExternalReadEntry)entry).id.getClass(); 
			Integer c = counts.get(clazz);
			if(c == null) counts.put(clazz, -1);
			else counts.put(clazz, c-1);
		}

		b.print("// Simantics DB client query report file\n");
		b.print("// This file contains the following information\n");
		b.print("// -The amount of cached query instances per query class\n");
		b.print("// -The sizes of retained child sets\n");
		b.print("// -List of parents for each query (search for 'P <query name>')\n");
		b.print("//  -Followed by status, where\n");
		b.print("//   -0=bound\n");
		b.print("//   -1=free\n");
		b.print("//   -2=unknown\n");
		b.print("//   -L=has listener\n");
		b.print("// -List of children for each query (search for 'C <query name>')\n");

		b.print("----------------------------------------\n");

		b.print("// Queries by class\n");
		for(Pair<Class<?>, Integer> p : CollectionUtils.valueSortedEntries(counts)) {
			b.print(-p.second + " " + p.first.getName() + "\n");
		}

		Map<CacheEntry, Integer> hist = new HashMap<CacheEntry, Integer>();
		for(CacheEntry e : workarea.keySet())
			hist.put(e, -1);
		
		boolean changed = true;
		int iter = 0;
		while(changed && iter++<50) {
			
			changed = false;
			
			Map<CacheEntry, Integer> newHist = new HashMap<CacheEntry, Integer>();
			for(CacheEntry e : workarea.keySet())
				newHist.put(e, -1);

			for(Map.Entry<CacheEntry, Set<CacheEntry>> e : workarea.entrySet()) {
				Integer c = hist.get(e.getKey());
				for(CacheEntry p : e.getValue()) {
					Integer i = newHist.get(p);
					newHist.put(p, i+c);
				}
			}
			for(CacheEntry e : workarea.keySet()) {
				Integer value = newHist.get(e);
				Integer old = hist.get(e);
				if(!value.equals(old)) {
					hist.put(e, value);
//					System.err.println("hist " + e + ": " + old + " => " + value);
					changed = true;
				}
			}
			
			System.err.println("Retained set iteration " + iter);

		}

		b.print("// Queries by retained set\n");
		for(Pair<CacheEntry, Integer> p : CollectionUtils.valueSortedEntries(hist)) {
			b.print("" + -p.second + " " + p.first + "\n");
		}

		HashMap<CacheEntry, Collection<CacheEntry>> inverse = new HashMap<CacheEntry, Collection<CacheEntry>>();

		b.print("// Entry parent listing\n");
		for(CacheEntry entry : workarea.keySet()) {
			int status = flagMap.get(entry);
			boolean hasListener = listening.hasListenerAfterDisposing(entry);
			b.print("Q " + entry.toString());
			if(hasListener) {
				b.print(" (L" + status + ")");
				listeners++;
			} else {
				b.print(" (" + status + ")");
			}
			b.print("\n");
			for(CacheEntry parent : workarea.get(entry)) {
				Collection<CacheEntry> inv = inverse.get(parent);
				if(inv == null) {
					inv = new ArrayList<CacheEntry>();
					inverse.put(parent, inv);
				}
				inv.add(entry);
				b.print("  " + parent.toString());
				b.print("\n");
			}
		}

		b.print("// Entry child listing\n");
		for(Map.Entry<CacheEntry, Collection<CacheEntry>> entry : inverse.entrySet()) {
			b.print("C " + entry.getKey().toString());
			b.print("\n");
			for(CacheEntry child : entry.getValue()) {
				Integer h = hist.get(child);
				if(h != null) {
					b.print("  " + h);
				} else {
					b.print("  <no children>");
				}
				b.print("  " + child.toString());
				b.print("\n");
			}
		}

		b.print("#queries: " + workarea.keySet().size() + "\n");
		b.print("#listeners: " + listeners + "\n");

		b.close();

		return "Dumped " + workarea.keySet().size() + " queries.";

	}

	static class QueryCluster {
		
		final QuerySerializerImpl serializer;
		final List<CacheEntryBase> entries = new ArrayList<>();
		
		QueryCluster(QueryProcessor processor) {
			serializer = new QuerySerializerImpl(processor);
		}
		
	}
	
    public synchronized void save() throws IOException {

        long start = System.nanoTime();

		QuerySerializerImpl serializer_ = new QuerySerializerImpl(this);

        Collection<CacheEntryBase> caches = allCaches(new CacheCollectionResult()).toCollection();
        Map<Long,QueryCluster> cachesByCluster = new HashMap<>();
        for(CacheEntryBase entry : caches) {
        	QueryFactoryKey clazz = entry.classId();
            if(clazz == null)
                continue;
            long cluster = entry.cluster(serializer_);
            QueryCluster queries = cachesByCluster.get(cluster);
            if(queries == null) {
                queries = new QueryCluster(this);
                cachesByCluster.put(cluster, queries);
            }
            queries.entries.add(entry);
        }

        File workspace = Platform.getLocation().toFile();
        File dir = new File(workspace, "queryData");
        FileUtils.deleteAll(dir);

        dir.mkdir();
        
        int totalCount = 0;

        for(Long cluster : cachesByCluster.keySet()) {

            QueryCluster queries = cachesByCluster.get(cluster);
            QuerySerializerImpl serializer = queries.serializer;
            int count = 0;
            int pos = serializer.writeUnknownSize();
            for(CacheEntryBase entry : queries.entries) {
                QueryFactoryKey clazz = entry.classId();
                if(clazz == null)
                    continue;
                if(entry.isDiscarded())
                    continue;
                try {
                    entry.serialize(serializer);
                    count++;
                } catch (IllegalStateException e) {
                    System.err.println(e.getMessage());
                }
            }
            serializer.setUnknownSize(pos, count);

            totalCount += count;

            FileUtils.writeFile(new File(dir, "" + cluster + ".queryData"), serializer.bytes());

        }

        long end = System.nanoTime();

        LOGGER.info("Saved {} queries for {} clusters in {} ms.", totalCount, cachesByCluster.size(), 1e-6*(end-start));

    }

    public void restore() throws IOException {

        long start = System.nanoTime();

        File workspace = Platform.getLocation().toFile();
        File dir = new File(workspace, "queryData");
        dir.mkdir();

        int count = 0;

        try {

            for(File f : FileUtils.listFilesByExtension(dir, "queryData")) {
                byte[] bytes = FileUtils.readFile(f);
                QueryDeserializerImpl qd = new QueryDeserializerImpl(this, bytes);
                qd.readHeaders();
                count += qd.readQueries();
            }
            
        } catch (Exception e) {
            LOGGER.error("Unexpected exception while restoring queries", e);
        }

        long end = System.nanoTime();

        LOGGER.info("Restored {} queries in {} ms.", count, 1e-6*(end-start));

    }

	boolean removeQuery(CacheEntry entry) {

		// This entry has been removed before. No need to do anything here.
		if(entry.isDiscarded())
			return false;

		assert (!entry.isDiscarded());

		Query query = entry.getQuery();

		query.removeEntry(this);

		cache.updates++;
		cache.size--;

		entry.discard();
		listening.discarded(entry);

		return true;

	}

	/**
	 * 
	 * @return true if this entry is being listened
	 */
	private boolean updateQuery(UpdateEntry e, Deque<UpdateEntry> todo, IdentityHashMap<CacheEntry, CacheEntry> immediates) throws DatabaseException {

		assert (e != null);

		CacheEntry entry = e.entry;

		/*
		 * If the dependency graph forms a DAG, some entries are inserted in the
		 * todo list many times. They only need to be processed once though.
		 */
		if (entry.isDiscarded()) {
			if (Development.DEVELOPMENT) {
				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
					StringBuilder sb = new StringBuilder();
					sb.append("D");
					for (int i = 0; i < e.indent; i++)
						sb.append(" ");
					sb.append(entry.toString());
					LOGGER.info(sb.toString());
				}
			}
//			System.err.println(" => DISCARDED");
			return false;
		}

//		if (entry.isRefuted()) {
//			if (Development.DEVELOPMENT) {
//				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
//					System.err.print("R");
//					for (int i = 0; i < e.indent; i++)
//						System.err.print(" ");
//					System.err.println(entry.getQuery());
//				}
//			}
//			return false;
//		}

		if (entry.isExcepted()) {
			if (Development.DEVELOPMENT) {
				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
					System.err.print("E");
				}
			}
		}

		if (entry.isPending()) {
			if (Development.DEVELOPMENT) {
				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
					System.err.print("P");
				}
			}
		}

		cache.updates++;

		Query query = entry.getQuery();
		int type = query.type();

		boolean hasListener = listening.hasListener(entry); 

		if (Development.DEVELOPMENT) {
			StringBuilder sb = new StringBuilder();
			if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
				sb.append("U ");
				for (int i = 0; i < e.indent; i++)
					sb.append(" ");
				sb.append(entry.getQuery());
				if(listening.hasListener(entry)) {
					sb.append(" (L)");
				}
				LOGGER.info(sb.toString());
			}
		}
		
		List<CacheEntry> parentsToProcess = listening.getParents(entry);

		if(entry.isPending() || entry.isExcepted()) {

			// If updated
			if ((type & RequestFlags.UPDATE_MASK) == RequestFlags.IMMEDIATE_UPDATE) {

				immediates.put(entry, entry);

			} else {

				if(hasListener) {
					entry.refute();
				} else {
					removeQuery(entry);
				}

			}

		} else {

			boolean needImmediateProcessing = (type & RequestFlags.UPDATE_MASK) == RequestFlags.IMMEDIATE_UPDATE;
			if(needImmediateProcessing) {
				needImmediateProcessing = false;
				for(int i=0;i<parentsToProcess.size();i++) {
					CacheEntry parent = parentsToProcess.get(i);
					if(!parent.isDiscarded()) {
						needImmediateProcessing = true;
						continue;
					}
				}
			}
			
			// If updated
			if (needImmediateProcessing) {

				immediates.put(entry, entry);

			} else {

				if(hasListener) {
					entry.refute();
				} else {
					removeQuery(entry);
				}

			}

		}

//		System.err.println(" => FOO " + type);

		if (hasListener) {
			ArrayList<ListenerEntry> entries = listening.listeners.get(entry);
			if(entries != null) {
				for (ListenerEntry le : entries) {
					listening.scheduleListener(le);
				}
			}
		}

		// If invalid, update parents
		if (type == RequestFlags.INVALIDATE) {
			if (parentsToProcess != null) {
				for(int i=0;i<parentsToProcess.size();i++) {
					CacheEntry parent = parentsToProcess.get(i);
					listening.updateParent(e.indent, entry, parent, todo);
				}
			}
		}

		return hasListener;

	}

	/**
	 * @param av1 an array (guaranteed)
	 * @param av2 any object
	 * @return <code>true</code> if the two arrays are equal
	 */
	private final boolean arrayEquals(Object av1, Object av2) {
		if (av2 == null)
			return false;
		Class<?> c1 = av1.getClass().getComponentType();
		Class<?> c2 = av2.getClass().getComponentType();
		if (c2 == null || !c1.equals(c2))
			return false;
		boolean p1 = c1.isPrimitive();
		boolean p2 = c2.isPrimitive();
		if (p1 != p2)
			return false;
		if (!p1)
			return Arrays.equals((Object[]) av1, (Object[]) av2);
		if (boolean.class.equals(c1))
			return Arrays.equals((boolean[]) av1, (boolean[]) av2);
		else if (byte.class.equals(c1))
			return Arrays.equals((byte[]) av1, (byte[]) av2);
		else if (int.class.equals(c1))
			return Arrays.equals((int[]) av1, (int[]) av2);
		else if (long.class.equals(c1))
			return Arrays.equals((long[]) av1, (long[]) av2);
		else if (float.class.equals(c1))
			return Arrays.equals((float[]) av1, (float[]) av2);
		else if (double.class.equals(c1))
			return Arrays.equals((double[]) av1, (double[]) av2);
		throw new RuntimeException("??? Contact application querySupport.");
	}



	final Object compareTo(ReadGraphImpl graph, final CacheEntry entry, final Object oldValue) {

		try {

			Query query = entry.getQuery();

			if (Development.DEVELOPMENT) {
				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_RECOMPUTE, Bindings.BOOLEAN)) {
					LOGGER.info("R " + query);
				}
			}

			entry.prepareRecompute(querySupport);
			
			ReadGraphImpl parentGraph = graph.forRecompute(entry);
			query.recompute(parentGraph);
			parentGraph.asyncBarrier.dec();
			parentGraph.asyncBarrier.waitBarrier(query, parentGraph);

			if(entry.isExcepted()) return ListenerEntry.NO_VALUE;

			Object newValue = entry.getResult();

			if (ListenerEntry.NO_VALUE == oldValue) {
				if (Development.DEVELOPMENT) {
					if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_CHANGES, Bindings.BOOLEAN)) {
						LOGGER.info("C1 " + query);
						LOGGER.info("- " + newValue);
					}
				}
				return newValue;
			}

			boolean changed = false;

			if (newValue != null) {
				if (newValue.getClass().isArray()) {
					changed = !arrayEquals(newValue, oldValue);
				} else {
					changed = !newValue.equals(oldValue);
				}
			} else
				changed = (oldValue != null);

			if(changed) {
				if (Development.DEVELOPMENT) {
					if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_CHANGES, Bindings.BOOLEAN)) {
						LOGGER.info("C2 " + query);
						LOGGER.info("- " + oldValue);
						LOGGER.info("- " + newValue);
					}
				}
			}

			return changed ? newValue : ListenerEntry.NOT_CHANGED;

		} catch (Throwable t) {

			Logger.defaultLogError(t);
			entry.except(t);
			return ListenerEntry.NO_VALUE;

		}

	}


	/**
	 * @return true if this entry still has listeners
	 */
	public boolean update(final ReadGraphImpl graph, final CacheEntry entry) {
		assert(entry != null);
		Deque<UpdateEntry> todo = new ArrayDeque<>(2);
		todo.add(new UpdateEntry(null, entry, 0));
		return update(graph, todo);
	}

	/**
	 * @return true if this entry still has listeners
	 */
	public boolean update(final ReadGraphImpl graph, final Deque<UpdateEntry> todo) {

		assert (!cache.collecting);
		assert (!updating);
		updating = true;

		boolean hadListeners = false;
		boolean listenersUnknown = false;

		try {

			assert(todo != null);
			IdentityHashMap<CacheEntry, CacheEntry> immediates = new IdentityHashMap<>();

			while(true) {

				// Walk the tree and collect immediate updates
				while (!todo.isEmpty()) {
					UpdateEntry e = todo.pop();
					hadListeners |= updateQuery(e, todo, immediates);
				}

				if(immediates.isEmpty()) break;

				// Evaluate all immediate updates and collect parents to update
				for(CacheEntry immediate : immediates.values()) {

					if(immediate.isDiscarded()) {
						continue;
					}

					if(immediate.isExcepted()) {

						Object newValue = compareTo(graph, immediate, ListenerEntry.NO_VALUE);
						if (newValue != ListenerEntry.NOT_CHANGED)
							listening.updateParents(0, immediate, todo);

					} else {

						Object oldValue = immediate.getResult();
						Object newValue = compareTo(graph, immediate, oldValue);

						if (newValue != ListenerEntry.NOT_CHANGED) {
							listening.updateParents(0, immediate, todo);
						} else {
							// If not changed, keep the old value
							immediate.setResult(oldValue);
							immediate.setReady();
							listenersUnknown = true;
						}

					}

				}
				immediates.clear();

			}

		} catch (Throwable t) {
			LOGGER.error("Query update propagation failure", t);
		}

		assert (updating);
		updating = false;

		return hadListeners | listenersUnknown;

	}

	private ObjectUpdateSet scheduledObjectUpdates = new ObjectUpdateSet();
	private ValueUpdateSet scheduledValueUpdates = new ValueUpdateSet();
	private ValueUpdateSet scheduledInvalidates = new ValueUpdateSet();
	// Maybe use a mutex from util.concurrent?
	private Object primitiveUpdateLock = new Object();
	private THashSet scheduledPrimitiveUpdates = new THashSet();

	private ArrayList<CacheEntry> refutations = new ArrayList<>();
	
	private void markForUpdate(ReadGraphImpl graph, CacheEntry e) {
		e.refute();
		refutations.add(e);
	}

	private void updateRefutations(ReadGraphImpl graph) {
		if (refutations.isEmpty())
			return;

		Deque<UpdateEntry> todo = new ArrayDeque<>(refutations.size()*2);
		for(CacheEntry e : refutations)
			todo.add(new UpdateEntry(null, e, 0));

		refutations.clear();

		update(graph, todo);
	}

	public void propagateChangesInQueryCache(final ReadGraphImpl graph) {
		
		ReadGraphImpl syncGraph = graph.forSyncExecute();
		propagateChangesInQueryCache_(syncGraph);
		syncGraph.asyncBarrier.dec();
		syncGraph.asyncBarrier.waitBarrier(this, syncGraph);

	}

	private void propagateChangesInQueryCache_(final ReadGraphImpl graph) {
		// Synchronize all pending listener tasks & enter synchronous operation
		listening.stopThreading();
		propagateChangesInQueryCache__(graph);
		listening.startThreading();
	}

	private void propagateChangesInQueryCache__(final ReadGraphImpl graph) {

		cache.dirty = false;
		lastInvalidate = 0;

		// Special case - one statement
		if(scheduledObjectUpdates.size() == 1 && scheduledValueUpdates.size() == 0 && scheduledPrimitiveUpdates.size() == 0 && scheduledInvalidates.size() == 0) {

			long arg0 = scheduledObjectUpdates.getFirst();

			final int subject = (int)(arg0 >>> 32);
			final int predicate = (int)(arg0 & 0xffffffff);

			QueryCache.entriesObjects(QueryProcessor.this, subject, o -> markForUpdate(graph,o));
			QueryCache.entriesDirectObjects(QueryProcessor.this, subject, o -> markForUpdate(graph,o));
			QueryCache.entriesStatements(QueryProcessor.this, subject, s -> markForUpdate(graph,s));

			if(predicate == instanceOf || predicate == inherits || predicate == subrelationOf) {
				PrincipalTypes principalTypes = QueryCache.entryPrincipalTypes(QueryProcessor.this, subject);
				if(principalTypes != null) markForUpdate(graph, principalTypes);
				Types types = QueryCache.entryTypes(QueryProcessor.this, subject);
				if(types != null) markForUpdate(graph, types);
			}

			if(predicate == subrelationOf) {
				SuperRelations superRelations = SuperRelations.entry(QueryProcessor.this, subject);
				if(superRelations != null) markForUpdate(graph, superRelations);
			}

			DirectPredicates dp = QueryCache.entryDirectPredicates(QueryProcessor.this, subject);
			if(dp != null) markForUpdate(graph, dp);
			OrderedSet os = QueryCache.entryOrderedSet(QueryProcessor.this, predicate);
			if(os != null) markForUpdate(graph, os);

			updateRefutations(graph);
			
			scheduledObjectUpdates.clear();

			if (Development.DEVELOPMENT) {
				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
					LOGGER.info("== Query update ends ==");
				}
			}

			return;

		}

		// Special case - one value
		if(scheduledObjectUpdates.size() == 0 && scheduledValueUpdates.size() == 1 && scheduledPrimitiveUpdates.size() == 0 && scheduledInvalidates.size() == 0) {

			int arg0 = scheduledValueUpdates.getFirst();

			ValueQuery valueQuery = QueryCache.entryValueQuery(QueryProcessor.this, arg0);
			if(valueQuery != null) markForUpdate(graph, valueQuery);

			updateRefutations(graph);

			scheduledValueUpdates.clear();

			if (Development.DEVELOPMENT) {
				if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
					LOGGER.info("== Query update ends ==");
				}
			}
			
			return;

		}

		final TIntHashSet predicates = new TIntHashSet();
		final TIntHashSet orderedSets = new TIntHashSet();

		THashSet primitiveUpdates;
		synchronized (primitiveUpdateLock) {
			primitiveUpdates = scheduledPrimitiveUpdates;
			scheduledPrimitiveUpdates = new THashSet();
		}

		scheduledValueUpdates.forEach(new TIntProcedure() {

			@Override
			public boolean execute(int arg0) {
				ValueQuery valueQuery = QueryCache.entryValueQuery(QueryProcessor.this, arg0);
				if(valueQuery != null) markForUpdate(graph, valueQuery);
				return true;
			}

		});

		scheduledInvalidates.forEach(new TIntProcedure() {

			@Override
			public boolean execute(int resource) {
				
				ValueQuery valueQuery = QueryCache.entryValueQuery(QueryProcessor.this, resource);
				if(valueQuery != null) markForUpdate(graph, valueQuery);
				
				PrincipalTypes principalTypes = QueryCache.entryPrincipalTypes(QueryProcessor.this, resource);
				if(principalTypes != null) markForUpdate(graph, principalTypes);
				Types types = QueryCache.entryTypes(QueryProcessor.this, resource);
				if(types != null) markForUpdate(graph, types);

				SuperRelations superRelations = SuperRelations.entry(QueryProcessor.this, resource);
				if(superRelations != null) markForUpdate(graph, superRelations);

				ChildMap childMap = ChildMap.entry(QueryProcessor.this, resource);
				if(childMap != null) markForUpdate(graph, childMap);
				
				QueryCache.entriesDirectObjects(QueryProcessor.this, resource, entry -> {
					DirectObjects directObjects = (DirectObjects) entry.getQuery();
					markForUpdate(graph, directObjects);
				});
				
				QueryCache.entriesObjects(QueryProcessor.this, resource, entry -> {
					Objects objects = (Objects) entry.getQuery();
					markForUpdate(graph, objects);
				});

				predicates.add(resource);
				
				return true;
			}

		});

		scheduledObjectUpdates.forEach(new TLongProcedure() {

			@Override
			public boolean execute(long arg0) {

				final int subject = (int)(arg0 >>> 32);
				final int predicate = (int)(arg0 & 0xffffffff);

				if(predicate == instanceOf || predicate == inherits || predicate == subrelationOf) {
					PrincipalTypes principalTypes = QueryCache.entryPrincipalTypes(QueryProcessor.this, subject);
					if(principalTypes != null) markForUpdate(graph, principalTypes);
					Types types = QueryCache.entryTypes(QueryProcessor.this, subject);
					if(types != null) markForUpdate(graph, types);
				}

				if(predicate == subrelationOf) {
					SuperRelations superRelations = SuperRelations.entry(QueryProcessor.this, subject);
					if(superRelations != null) markForUpdate(graph, superRelations);
				}

				predicates.add(subject);
				orderedSets.add(predicate);

				return true;

			}

		});

		predicates.forEach(new TIntProcedure() {

			@Override
			public boolean execute(final int predicate) {

				QueryCache.entriesObjects(QueryProcessor.this, predicate, o -> markForUpdate(graph,o));
				QueryCache.entriesDirectObjects(QueryProcessor.this, predicate, o -> markForUpdate(graph,o));
				QueryCache.entriesStatements(QueryProcessor.this, predicate, s -> markForUpdate(graph,s));

				DirectPredicates entry = QueryCache.entryDirectPredicates(QueryProcessor.this, predicate);
				if(entry != null) markForUpdate(graph, entry);

				return true;

			}

		});

		orderedSets.forEach(new TIntProcedure() {

			@Override
			public boolean execute(int orderedSet) {

				OrderedSet entry = QueryCache.entryOrderedSet(QueryProcessor.this, orderedSet);
				if(entry != null) markForUpdate(graph, entry);

				return true;

			}

		});

		primitiveUpdates.forEach(new TObjectProcedure() {

			@Override
			public boolean execute(Object arg0) {

				ExternalReadEntry query = (ExternalReadEntry)cache.externalReadEntryMap.get(arg0);
				if (query != null) {
					markForUpdate(graph, query);
					updatedPrimitivesInCurrentWrite.add(query);
				}

				return true;

			}

		});

		updateRefutations(graph);

		scheduledValueUpdates.clear();
		scheduledObjectUpdates.clear();
		scheduledInvalidates.clear();
		
		if (Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_UPDATE, Bindings.BOOLEAN)) {
				LOGGER.info("== Query update ends ==");
			}
		}

	}

	public void updateValue(final int resource) {
		scheduledValueUpdates.add(resource);
		cache.dirty = true;
	}

	public void updateStatements(final int resource, final int predicate) {
		scheduledObjectUpdates.add((((long)resource) << 32) + predicate);
		cache.dirty = true;
	}
	
	private int lastInvalidate = 0;
	
	public void invalidateResource(final int resource) {
		if(lastInvalidate == resource) return;
		scheduledInvalidates.add(resource);
		lastInvalidate = resource;
		cache.dirty = true;
	}

	public void updatePrimitive(final ExternalRead primitive) {

		// External reads may be updated from arbitrary threads.
		// Synchronize to prevent race-conditions.
		synchronized (primitiveUpdateLock) {
			scheduledPrimitiveUpdates.add(primitive);
		}
		querySupport.dirtyPrimitives();

	}

	@Override
	public synchronized String toString() {
		return "QueryProvider [size = " + cache.size + ", hits = " + cache.hits + " misses = " + cache.misses + ", updates = " + cache.updates + "]";
	}

	@Override
	protected void doDispose() {

		requests.release(Integer.MAX_VALUE / 2);
		
		for(int index = 0; index < THREADS; index++) { 
			executors[index].dispose();
		}

		// First just wait
		for(int i=0;i<100;i++) {

			boolean alive = false;
			for(int index = 0; index < THREADS; index++) { 
				alive |= executors[index].isAlive();
			}
			if(!alive) return;
			try {
				Thread.sleep(5);
			} catch (InterruptedException e) {
				Logger.defaultLogError(e);
			}

		}

		// Then start interrupting
		for(int i=0;i<100;i++) {

			boolean alive = false;
			for(int index = 0; index < THREADS; index++) { 
				alive |= executors[index].isAlive();
			}
			if(!alive) return;
			for(int index = 0; index < THREADS; index++) {
				executors[index].interrupt();
			}
		}

		//		// Then just destroy
		//		for(int index = 0; index < THREADS; index++) {
		//			executors[index].destroy();
		//		}

		for(int index = 0; index < THREADS; index++) {
			try {
				executors[index].join(5000);
			} catch (InterruptedException e) {
				Logger.defaultLogError("QueryThread " + index + " will not die.", e);
			}
			executors[index] = null;
		}

	}

	public int getHits() {
		return cache.hits;
	}

	public int getMisses() {
		return cache.misses;
	}

	public int getSize() {
		return cache.size;
	}

	public Set<Long> getReferencedClusters() {

		HashSet<Long> result = new HashSet<>();
		QueryCache.entriesObjects(this, entry -> {
			Objects query = (Objects) entry.getQuery();
			result.add(querySupport.getClusterId(query.r1()));
		});
		QueryCache.entriesDirectPredicates(this, entry -> {
			DirectPredicates query = (DirectPredicates) entry.getQuery();
			result.add(querySupport.getClusterId(query.id));
		});
		cache.valueQueryMap.values(entry -> {
			result.add(querySupport.getClusterId(entry.id));
		});
		return result;
	}
	
	public long cluster(int resource) {
	    if(resource <= 0)
	        return 0;
	    return querySupport.getClusterId(resource);
	}

	public void assertDone() {
	}

	CacheCollectionResult allCaches(CacheCollectionResult result) {
		
		return cache.allCaches(result);

	}

	public void printDiagnostics() {
	}

	public void requestCluster(ReadGraphImpl graph, long clusterId, Runnable runnable) {
		querySupport.requestCluster(graph, clusterId, runnable);
	}

	public int clean() {
		collector.collect(0, Integer.MAX_VALUE);
		return cache.size;
	}

	public void clean(final Collection<ExternalRead<?>> requests) {
		QueryCollectorSupport collectorSupport = new QueryCollectorSupport() {
			Iterator<ExternalRead<?>> iterator = requests.iterator();
			@Override
			public CacheCollectionResult allCaches() {
				throw new UnsupportedOperationException();
			}
			@Override
			public CacheEntryBase iterate(int level) {
				if(iterator.hasNext()) {
					ExternalRead<?> request = iterator.next();
					ExternalReadEntry entry = cache.externalReadEntryMap.get(request);
					if (entry != null) return entry;
					else return iterate(level);
				} else {
					iterator = requests.iterator();
					return null;
				}
			}
			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}
			@Override
			public void setLevel(CacheEntryBase entry, short level) {
				throw new UnsupportedOperationException();
			}
			@Override
			public Collection<CacheEntry<?>> getRootList() {
				ArrayList<CacheEntry<?>> result = new ArrayList<CacheEntry<?>>(requests.size());
				for (ExternalRead<?> request : requests) {
					ExternalReadEntry entry = cache.externalReadEntryMap.get(request);
					if (entry != null)
						result.add(entry);
				}
				return result;
			}
			public long getActivity() {
				return cache.updates;
			}
			@Override
			public int getCurrentSize() {
				return cache.size;
			}
			@Override
			public int calculateCurrentSize() {
				// This tells the collector to attempt collecting everything.
				return Integer.MAX_VALUE;
			}
			@Override
			public boolean start(boolean flush) {
				return true;
			}
		};
		new QueryCollectorImpl2(this, collectorSupport).collect(0, Integer.MAX_VALUE);
	}

	public void scanPending() {
		
		cache.scanPending();

	}

	public ReadGraphImpl graphForVirtualRequest() {
		return ReadGraphImpl.createAsync(this);
	}

	
	private HashMap<Resource, Class<?>> builtinValues;
	
	public Class<?> getBuiltinValue(Resource r) {
		if(builtinValues == null) initBuiltinValues();
		return builtinValues.get(r);
	}

	Exception callerException = null;

    public interface AsyncBarrier {
        public void inc(); 
        public void dec();
        public void waitBarrier(Object request, ReadGraphImpl impl);
        public boolean isBlocking();
    }

//	final public QueryProcessor processor;
//	final public QuerySupport support;

	//    boolean disposed = false;

	private void initBuiltinValues() {

		Layer0 b = getSession().peekService(Layer0.class);
		if(b == null) return;

		builtinValues = new HashMap<Resource, Class<?>>();

		builtinValues.put(b.String, String.class);
		builtinValues.put(b.Double, Double.class);
		builtinValues.put(b.Float, Float.class);
		builtinValues.put(b.Long, Long.class);
		builtinValues.put(b.Integer, Integer.class);
		builtinValues.put(b.Byte, Byte.class);
		builtinValues.put(b.Boolean, Boolean.class);

		builtinValues.put(b.StringArray, String[].class);
		builtinValues.put(b.DoubleArray, double[].class);
		builtinValues.put(b.FloatArray, float[].class);
		builtinValues.put(b.LongArray, long[].class);
		builtinValues.put(b.IntegerArray, int[].class);
		builtinValues.put(b.ByteArray, byte[].class);
		builtinValues.put(b.BooleanArray, boolean[].class);

	}

//	public ReadGraphSupportImpl(final QueryProcessor provider2) {
//
//		if (null == provider2) {
//			this.processor = null;
//			support = null;
//			return;
//		}
//		this.processor = provider2;
//		support = provider2.getCore();
//		initBuiltinValues();
//
//	}

//	final static public ReadGraphSupportImpl basedOn(ReadGraphSupportImpl impl) {
//		return new ReadGraphSupportImpl(impl.processor);
//	}

	@Override
	final public Session getSession() {
		return session;
	}
	
	final public ResourceSupport getResourceSupport() {
		return resourceSupport;
	}

	@Override
	final public void forEachPredicate(final ReadGraphImpl impl, final Resource subject, final AsyncMultiProcedure<Resource> procedure) {

        try {

	        for(Resource predicate : getPredicates(impl, subject))
	            procedure.execute(impl, predicate);

	        procedure.finished(impl);

	    } catch (Throwable e) {
	        procedure.exception(impl, e);
	    }

	}

	@Override
	final public void forEachPredicate(final ReadGraphImpl impl, final Resource subject, final MultiProcedure<Resource> procedure) {
		
		throw new UnsupportedOperationException();

//		assert(subject != null);
//		assert(procedure != null);
//
//		final ListenerBase listener = getListenerBase(procedure);
//
//		try {
//			QueryCache.runnerPredicates(impl, querySupport.getId(subject), impl.parent, listener, new IntProcedure() {
//
//				@Override
//				public void execute(ReadGraphImpl graph, int i) {
//					try {
//						procedure.execute(querySupport.getResource(i));
//					} catch (Throwable t2) {
//						Logger.defaultLogError(t2);
//					}
//				}
//
//				@Override
//				public void finished(ReadGraphImpl graph) {
//					try {
//						procedure.finished();
//					} catch (Throwable t2) {
//						Logger.defaultLogError(t2);
//					}
////				impl.state.barrier.dec();
//				}
//
//				@Override
//				public void exception(ReadGraphImpl graph, Throwable t) {
//					try {
//						procedure.exception(t);
//					} catch (Throwable t2) {
//						Logger.defaultLogError(t2);
//					}
////				impl.state.barrier.dec();
//				}
//
//			});
//		} catch (DatabaseException e) {
//			Logger.defaultLogError(e);
//		}

	}
	
	@Override
	final public IntSet getPredicates(final ReadGraphImpl impl, final Resource subject) throws Throwable {
		return QueryCacheBase.resultPredicates(impl, querySupport.getId(subject), impl.parent, null); 
	}

	final public IntSet getDirectPredicates(final ReadGraphImpl impl, final Resource subject) throws Throwable {
		return QueryCacheBase.resultDirectPredicates(impl, querySupport.getId(subject), impl.parent, null); 
	}

	@Override
	final public void forEachStatement(final ReadGraphImpl impl, final Resource subject,
			final Resource predicate, final MultiProcedure<Statement> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

//		impl.state.barrier.inc();

		try {
			Statements.queryEach(impl, querySupport.getId(subject), querySupport.getId(predicate), this, impl.parent, listener, new TripleIntProcedureAdapter() {

				@Override
				public void execute(ReadGraphImpl graph, int s, int p, int o) {
					try {
						procedure.execute(querySupport.getStatement(s, p, o));
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {
						procedure.finished();
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forEachStatement(final ReadGraphImpl impl, final Resource subject,
			final Resource predicate, final AsyncMultiProcedure<Statement> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		TripleIntProcedureAdapter proc = new TripleIntProcedureAdapter() {

			boolean first = true;

			@Override
			public void execute(ReadGraphImpl graph, int s, int p, int o) {
				try {
					if(first) {
						procedure.execute(graph, querySupport.getStatement(s, p, o));
					} else {
						procedure.execute(impl.newRestart(graph), querySupport.getStatement(s, p, o));
					}
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
			}

			@Override
			public void finished(ReadGraphImpl graph) {

				try {
					if(first) {
						first = false;
						procedure.finished(graph);
//						impl.state.barrier.dec(this);
					} else {
						procedure.finished(impl.newRestart(graph));
					}
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}

			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {

				try {
					if(first) {
						first = false;
						procedure.exception(graph, t);
//						impl.state.barrier.dec(this);
					} else {
						procedure.exception(impl.newRestart(graph), t);
					}
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}

			}

		};

		int sId = querySupport.getId(subject);
		int pId = querySupport.getId(predicate);

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(proc, "#Statements" + sId + "#" + pId);
//		else impl.state.barrier.inc(null, null);

		try {
			Statements.queryEach(impl, sId, pId, this, impl.parent, listener, proc);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forEachStatement(final ReadGraphImpl impl, final Resource subject,
			final Resource predicate, final StatementProcedure procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		TripleIntProcedureAdapter proc = new TripleIntProcedureAdapter() {

			boolean first = true;

			@Override
			public void execute(ReadGraphImpl graph, int s, int p, int o) {
				try {
					if(first) {
						procedure.execute(graph, s, p, o);
					} else {
						procedure.execute(impl.newRestart(graph), s, p, o);
					}
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
			}

			@Override
			public void finished(ReadGraphImpl graph) {

				try {
					if(first) {
						first = false;
						procedure.finished(graph);
//						impl.state.barrier.dec(this);
					} else {
						procedure.finished(impl.newRestart(graph));
					}
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}

			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {

				try {
					if(first) {
						first = false;
						procedure.exception(graph, t);
//						impl.state.barrier.dec(this);
					} else {
						procedure.exception(impl.newRestart(graph), t);
					}
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}

			}

		};

		int sId = querySupport.getId(subject);
		int pId = querySupport.getId(predicate);

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(proc, "#Statements" + sId + "#" + pId);
//		else impl.state.barrier.inc(null, null);

		try {
			Statements.queryEach(impl, sId, pId, this, impl.parent, listener, proc);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}
	
	@Override
	final public void forStatementSet(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncSetListener<Statement> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		forEachStatement(impl, subject, predicate, new AsyncMultiListener<Statement>() {

			private Set<Statement> current = null;
			private Set<Statement> run = new HashSet<Statement>();

			@Override
			public void execute(AsyncReadGraph graph, Statement result) {

				boolean found = false;

				if(current != null) {

					found = current.remove(result);

				}

				if(!found) procedure.add(graph, result);

				run.add(result);

			}

			@Override
			public void finished(AsyncReadGraph graph) {

				if(current != null) { 
					for(Statement r : current) procedure.remove(graph, r);
				}

				current = run;

				run = new HashSet<Statement>();

			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public boolean isDisposed() {
				return procedure.isDisposed();
			}

		});

	}

	@Override
	final public void forEachAssertedStatement(final ReadGraphImpl impl, final Resource subject,
			final Resource predicate, final AsyncMultiProcedure<Statement> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

//		impl.state.barrier.inc();

		try {
			QueryCache.runnerAssertedStatements(impl, querySupport.getId(subject), querySupport.getId(predicate), impl.parent, listener, new TripleIntProcedureAdapter() {

				@Override
				public void execute(ReadGraphImpl graph, int s, int p, int o) {
					try {
						procedure.execute(graph, querySupport.getStatement(s, p, o));
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {
						procedure.finished(graph);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	private static ListenerBase getListenerBase(Object procedure) {
		if(procedure instanceof ListenerBase) return (ListenerBase)procedure;
		else return null;
	}

	@Override
	final public void forEachObject(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final MultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

//		impl.state.barrier.inc();

		try {
			QueryCache.runnerObjects(impl, querySupport.getId(subject), querySupport.getId(predicate), impl.parent, listener, new IntProcedure() {

				@Override
				public void execute(ReadGraphImpl graph, int i) {
					try {
						procedure.execute(querySupport.getResource(i));
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {
						procedure.finished();
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					System.out.println("forEachObject exception " + t);
					try {
						procedure.exception(t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forEachDirectPredicate(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Set<Resource>> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		int sId = querySupport.getId(subject);

		try {
			QueryCache.runnerDirectPredicates(impl, sId, impl.parent, listener, new InternalProcedure<IntSet>() {

				@Override
				public void execute(ReadGraphImpl graph, IntSet result) throws DatabaseException {
					procedure.execute(graph, result);
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
					procedure.exception(graph, throwable);
				}
				
			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	final public DirectStatements getDirectStatements(final ReadGraphImpl impl, final Resource subject, final boolean ignoreVirtual) {

//		assert(subject != null);
//		assert(procedure != null);
//
//		final ListenerBase listener = getListenerBase(procedure);
//
//		org.simantics.db.impl.query.DirectStatements.queryEach(impl, querySupport.getId(subject), this, impl.parent, listener, procedure);
		
		return querySupport.getStatements(impl, querySupport.getId(subject), this, ignoreVirtual);

	}

//	@Override
//	final public void forEachDirectStatement(final ReadGraphImpl impl, final Resource subject, final SyncProcedure<DirectStatements> procedure, boolean ignoreVirtual) {
//
//		assert(subject != null);
//		assert(procedure != null);
//
//		final ListenerBase listener = getListenerBase(procedure);
//
//		org.simantics.db.impl.query.DirectStatements.queryEach(impl, querySupport.getId(subject), this, impl.parent, listener, procedure, ignoreVirtual);
//
//	}
	
	private static final Resource INVALID_RESOURCE = new ResourceImpl(null, Integer.MIN_VALUE);

	@Override
	final public void forPossibleObject(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncProcedure<Resource> procedure) {

		forEachObject(impl, subject, predicate, new AsyncMultiProcedure<Resource>() {

			private Resource single = null;

			@Override
			public synchronized void execute(AsyncReadGraph graph, Resource result) {
				if(single == null) {
					single = result;
				} else {
					single = INVALID_RESOURCE;
				}
			}

			@Override
			public synchronized void finished(AsyncReadGraph graph) {
				if(single == null || single == INVALID_RESOURCE) procedure.execute(graph, null);
				else procedure.execute(graph, single);
			}

			@Override
			public synchronized void exception(AsyncReadGraph graph, Throwable throwable) {
				procedure.exception(graph, throwable);
			}

		});

	}

	final public void forEachObject(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final ListenerBase listener, final IntProcedure procedure) {
		
		final int sId = querySupport.getId(subject);
		final int pId = querySupport.getId(predicate);

		try {
			QueryCache.runnerObjects(impl, sId, pId, impl.parent, listener, procedure);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}
		
	}
	
	static class Runner2Procedure implements IntProcedure {
	    
	    public int single = 0;
	    public Throwable t = null;

	    public void clear() {
	        single = 0;
	        t = null;
	    }
	    
        @Override
        public void execute(ReadGraphImpl graph, int i) {
            if(single == 0) single = i;
            else single = -1;
        }

        @Override
        public void finished(ReadGraphImpl graph) {
            if(single == -1) single = 0;
        }

        @Override
        public void exception(ReadGraphImpl graph, Throwable throwable) {
            single = 0;
            this.t = throwable;
        }
        
        public int get() throws DatabaseException {
            if(t != null) {
                if(t instanceof DatabaseException) throw (DatabaseException)t;
                else throw new DatabaseException(t);
            }
            return single;
        }
	    
	}
	
	final public int getSingleObject(final ReadGraphImpl impl, final Resource subject, final Resource predicate) throws DatabaseException {
		
		final int sId = querySupport.getId(subject);
		final int pId = querySupport.getId(predicate);

		Runner2Procedure proc = new Runner2Procedure();
		QueryCache.runnerObjects(impl, sId, pId, impl.parent, null, proc);
		return proc.get();
	    
	}

	final public void forEachObject(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncMultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(predicate != null);

		final ListenerBase listener = getListenerBase(procedure);

		if(impl.parent != null || listener != null) {

			IntProcedure ip = new IntProcedure() {

				AtomicBoolean first = new AtomicBoolean(true);

				@Override
				public void execute(ReadGraphImpl graph, int i) {
					try {
						if(first.get()) {
							procedure.execute(impl, querySupport.getResource(i));
						} else {
							procedure.execute(impl.newRestart(graph), querySupport.getResource(i));
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}

				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {
						if(first.compareAndSet(true, false)) {
							procedure.finished(impl);
//							impl.state.barrier.dec(this);
						} else {
							procedure.finished(impl.newRestart(graph));
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//					impl.state.barrier.dec(this);
				}

				@Override
				public String toString() {
					return "forEachObject with " + procedure;
				}

			};

//			if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#Objects" + subject + "#" + predicate);
//			else impl.state.barrier.inc(null, null);

			forEachObject(impl, subject, predicate, listener, ip);

		} else {

			IntProcedure ip = new IntProcedure() {

				@Override
				public void execute(ReadGraphImpl graph, int i) {
					procedure.execute(graph, querySupport.getResource(i));
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					procedure.finished(graph);
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					procedure.exception(graph, t);
				}

				@Override
				public String toString() {
					return "forEachObject with " + procedure;
				}

			};

			forEachObject(impl, subject, predicate, listener, ip);

		}

	}

	@Override
	final public void forObjectSet(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncSetListener<Resource> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		forEachObject(impl, subject, predicate, new AsyncMultiListener<Resource>() {

			private Set<Resource> current = null;
			private Set<Resource> run = new HashSet<Resource>();

			@Override
			public void execute(AsyncReadGraph graph, Resource result) {

				boolean found = false;

				if(current != null) {

					found = current.remove(result);

				}

				if(!found) procedure.add(graph, result);

				run.add(result);

			}

			@Override
			public void finished(AsyncReadGraph graph) {

				if(current != null) { 
					for(Resource r : current) procedure.remove(graph, r);
				}

				current = run;

				run = new HashSet<Resource>();

			}

			@Override
			public boolean isDisposed() {
				return procedure.isDisposed();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forObjectSet " + procedure;
			}

		});

	}

	@Override
	final public void forPredicateSet(final ReadGraphImpl impl, final Resource subject, final AsyncSetListener<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		forEachPredicate(impl, subject, new AsyncMultiListener<Resource>() {

			private Set<Resource> current = null;
			private Set<Resource> run = new HashSet<Resource>();

			@Override
			public void execute(AsyncReadGraph graph, Resource result) {

				boolean found = false;

				if(current != null) {

					found = current.remove(result);

				}

				if(!found) procedure.add(graph, result);

				run.add(result);

			}

			@Override
			public void finished(AsyncReadGraph graph) {

				if(current != null) { 
					for(Resource r : current) procedure.remove(graph, r);
				}

				current = run;

				run = new HashSet<Resource>();

			}

			@Override
			public boolean isDisposed() {
				return procedure.isDisposed();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forPredicateSet " + procedure;
			}

		});

	}

	@Override
	final public void forPrincipalTypeSet(final ReadGraphImpl impl, final Resource subject, final AsyncSetListener<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		forEachPrincipalType(impl, subject, new AsyncMultiListener<Resource>() {

			private Set<Resource> current = null;
			private Set<Resource> run = new HashSet<Resource>();

			@Override
			public void execute(AsyncReadGraph graph, Resource result) {

				boolean found = false;

				if(current != null) {

					found = current.remove(result);

				}

				if(!found) procedure.add(graph, result);

				run.add(result);

			}

			@Override
			public void finished(AsyncReadGraph graph) {

				if(current != null) { 
					for(Resource r : current) procedure.remove(graph, r);
				}

				current = run;

				run = new HashSet<Resource>();

			}

			@Override
			public boolean isDisposed() {
				return procedure.isDisposed();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forPrincipalTypeSet " + procedure;
			}

		});

	}

	@Override
	final public void forAssertedObjectSet(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncSetListener<Resource> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		forEachAssertedObject(impl, subject, predicate, new AsyncMultiListener<Resource>() {

			private Set<Resource> current = null;
			private Set<Resource> run = new HashSet<Resource>();

			@Override
			public void execute(AsyncReadGraph graph, Resource result) {

				boolean found = false;

				if(current != null) {

					found = current.remove(result);

				}

				if(!found) procedure.add(graph, result);

				run.add(result);

			}

			@Override
			public void finished(AsyncReadGraph graph) {

				if(current != null) { 
					for(Resource r : current) procedure.remove(graph, r);
				}

				current = run;

				run = new HashSet<Resource>();

			}

			@Override
			public boolean isDisposed() {
				return procedure.isDisposed();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forObjectSet " + procedure;
			}

		});

	}

	@Override
	final public void forAssertedStatementSet(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncSetListener<Statement> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		forEachAssertedStatement(impl, subject, predicate, new AsyncMultiListener<Statement>() {

			private Set<Statement> current = null;
			private Set<Statement> run = new HashSet<Statement>();

			@Override
			public void execute(AsyncReadGraph graph, Statement result) {

				boolean found = false;

				if(current != null) {

					found = current.remove(result);

				}

				if(!found) procedure.add(graph, result);

				run.add(result);

			}

			@Override
			public void finished(AsyncReadGraph graph) {

				if(current != null) { 
					for(Statement s : current) procedure.remove(graph, s);
				}

				current = run;

				run = new HashSet<Statement>();

			}

			@Override
			public boolean isDisposed() {
				return procedure.isDisposed();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forStatementSet " + procedure;
			}

		});

	}

	@Override
	final public void forEachAssertedObject(final ReadGraphImpl impl, final Resource subject,
			final Resource predicate, final AsyncMultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		try {
			QueryCache.runnerAssertedStatements(impl, querySupport.getId(subject), querySupport.getId(predicate), impl.parent, listener, new TripleIntProcedure() {

				@Override
				public void execute(ReadGraphImpl graph, int s, int p, int o) {
					try {
						procedure.execute(graph, querySupport.getResource(o));
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {        	    
						procedure.finished(graph);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forEachPrincipalType(final ReadGraphImpl impl, final Resource subject, final AsyncMultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		IntProcedure ip = new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				try {
					procedure.execute(graph, querySupport.getResource(i));
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
			}

			@Override
			public void finished(ReadGraphImpl graph) {
				try {
					procedure.finished(graph);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

		};

		int sId = querySupport.getId(subject);

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#PrincipalTypes#" + sId);
//		else impl.state.barrier.inc(null, null);

		try {
			QueryCache.runnerPrincipalTypes(impl, sId, impl.parent, listener, ip);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forEachPrincipalType(final ReadGraphImpl impl, final Resource subject, final MultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

//		impl.state.barrier.inc();

		try {
			QueryCache.runnerPrincipalTypes(impl, querySupport.getId(subject), impl.parent, listener, new IntProcedure() {

				@Override
				public void execute(ReadGraphImpl graph, int i) {
					try {
						procedure.execute(querySupport.getResource(i));
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {
						procedure.finished();
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}
	}

    final public void forTypes(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Set<Resource>> procedure) {

        assert(subject != null);
        assert(procedure != null);

        final ListenerBase listener = getListenerBase(procedure);
        assert(listener == null);

        InternalProcedure<IntSet> ip = new InternalProcedure<IntSet>() {

            @Override
            public void execute(final ReadGraphImpl graph, IntSet set) {
            	procedure.execute(graph, set);
            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
            	procedure.exception(graph, t);
            }

        };

        int sId = querySupport.getId(subject);

        try {
			QueryCache.runnerTypes(impl, sId, impl.parent, listener, ip);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

    }
    
	@Override
	final public IntSet getTypes(final ReadGraphImpl impl, final Resource subject) throws Throwable {

		assert(subject != null);
		
		return QueryCacheBase.resultTypes(impl, querySupport.getId(subject), impl.parent, null);

	}

	@Override
	final public RelationInfo getRelationInfo(final ReadGraphImpl impl, final Resource subject) throws DatabaseException {
		
		assert(subject != null);

		return QueryCache.resultRelationInfoQuery(impl, querySupport.getId(subject), impl.parent, null);

	}

	@Override
	final public void forSupertypes(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Set<Resource>> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		try {
			QueryCache.runnerSuperTypes(impl, querySupport.getId(subject), impl.parent, listener, new InternalProcedure<IntSet>() {

				AtomicBoolean first = new AtomicBoolean(true);

				@Override
				public void execute(final ReadGraphImpl graph, IntSet set) {
//				final HashSet<Resource> result = new HashSet<Resource>();
//				set.forEach(new TIntProcedure() {
//
//					@Override
//					public boolean execute(int type) {
//						result.add(querySupport.getResource(type));
//						return true;
//					}
//
//				});
					try {
						if(first.compareAndSet(true, false)) {
							procedure.execute(graph, set);
//						impl.state.barrier.dec();
						} else {
							procedure.execute(impl.newRestart(graph), set);
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						if(first.compareAndSet(true, false)) {
							procedure.exception(graph, t);
//						impl.state.barrier.dec();
						} else {
							procedure.exception(impl.newRestart(graph), t);
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forDirectSuperrelations(final ReadGraphImpl impl, final Resource subject, final AsyncMultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		IntProcedure ip = new IntProcedureAdapter() {

			@Override
			public void execute(final ReadGraphImpl graph, int superRelation) {
				try {
					procedure.execute(graph, querySupport.getResource(superRelation));
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
			}

			@Override
			public void finished(final ReadGraphImpl graph) {
				try {
					procedure.finished(graph);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}


			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

		};

		int sId = querySupport.getId(subject); 

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#DirectSuperRelations#" + sId);
//		else impl.state.barrier.inc(null, null);

		try {
			QueryCache.runnerDirectSuperRelations(impl, sId, impl.parent, listener, ip);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}
		
//		DirectSuperRelations.queryEach(impl, sId, this, impl.parent, listener, ip);

	}

	@Override
	final public void forPossibleSuperrelation(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

//		impl.state.barrier.inc();

		PossibleSuperRelation.queryEach(impl, querySupport.getId(subject), this, impl.parent, listener, procedure);

	}

	@Override
	final public void forSuperrelations(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Set<Resource>> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		InternalProcedure<IntSet> ip = new InternalProcedure<IntSet>() {

			@Override
			public void execute(final ReadGraphImpl graph, IntSet set) {
//				final HashSet<Resource> result = new HashSet<Resource>();
//				set.forEach(new TIntProcedure() {
//
//					@Override
//					public boolean execute(int type) {
//						result.add(querySupport.getResource(type));
//						return true;
//					}
//
//				});
				try {
					procedure.execute(graph, set);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

		};

		int sId = querySupport.getId(subject);

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#SuperRelations#" + sId);
//		else impl.state.barrier.inc(null, null);

		try {
			QueryCache.runnerSuperRelations(impl, sId, impl.parent, listener, ip);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	final public byte[] getValue(final ReadGraphImpl impl, final Resource subject) throws DatabaseException {
	  return getValue(impl, querySupport.getId(subject));
	}

	final public byte[] getValue(final ReadGraphImpl impl, final int subject) throws DatabaseException {
		return QueryCache.resultValueQuery(impl, subject, impl.parent, null); 
	}

	@Override
	final public void forValue(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<byte[]> procedure) {

		assert(subject != null);
		assert(procedure != null);

		int sId = querySupport.getId(subject);

//		if(procedure != null) {
		
			final ListenerBase listener = getListenerBase(procedure);

			InternalProcedure<byte[]> ip = new InternalProcedure<byte[]>() {

				AtomicBoolean first = new AtomicBoolean(true);

				@Override
				public void execute(ReadGraphImpl graph, byte[] result) {
					try {
						if(first.compareAndSet(true, false)) {
							procedure.execute(graph, result);
//							impl.state.barrier.dec(this);
						} else {
							procedure.execute(impl.newRestart(graph), result);
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						if(first.compareAndSet(true, false)) {
							procedure.exception(graph, t);
//							impl.state.barrier.dec(this);
						} else {
							procedure.exception(impl.newRestart(graph), t);
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

			};

//			if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#Value" + sId);
//			else impl.state.barrier.inc(null, null);

			try {
				QueryCache.runnerValueQuery(impl, sId, impl.parent, listener, ip);
			} catch (DatabaseException e) {
				throw new IllegalStateException("Internal error");
			}

//		} else {
//
//			return QueryCacheBase.runnerValueQuery(impl, sId, impl.parent, null, null);
// 			
// 		}
//		
//		throw new IllegalStateException("Internal error");

	}

	@Override
	final public void forPossibleValue(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<byte[]> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		if(impl.parent != null || listener != null) {

			InternalProcedure<byte[]> ip = new InternalProcedure<byte[]>() {

				AtomicBoolean first = new AtomicBoolean(true);

				@Override
				public void execute(ReadGraphImpl graph, byte[] result) {
					try {
						if(first.compareAndSet(true, false)) {
							procedure.execute(graph, result);
//							impl.state.barrier.dec(this);
						} else {
							procedure.execute(impl.newRestart(graph), result);
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						if(first.compareAndSet(true, false)) {
							procedure.exception(graph, t);
//							impl.state.barrier.dec(this);
						} else {
							procedure.exception(impl.newRestart(graph), t);
						}
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

			};

			int sId = querySupport.getId(subject);

//			if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#Value" + sId);
//			else impl.state.barrier.inc(null, null);

			try {
				QueryCache.runnerValueQuery(impl, sId, impl.parent, listener, ip);
			} catch (DatabaseException e) {
				Logger.defaultLogError(e);
			}

		} else {

			InternalProcedure<byte[]> ip = new InternalProcedure<byte[]>() {

				@Override
				public void execute(ReadGraphImpl graph, byte[] result) {

					procedure.execute(graph, result);

				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {

					procedure.exception(graph, t);

				}

			};

			int sId = querySupport.getId(subject);

			try {
				QueryCache.runnerValueQuery(impl, sId, impl.parent, listener, ip);
			} catch (DatabaseException e) {
				Logger.defaultLogError(e);
			}

		}

	}

	@Override
	final public void forInverse(final ReadGraphImpl impl, final Resource relation, final AsyncProcedure<Resource> procedure) {

		assert(relation != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		IntProcedure ip = new IntProcedure() {

			private int result = 0;
			
		    final AtomicBoolean found = new AtomicBoolean(false);
		    final AtomicBoolean done = new AtomicBoolean(false);

			@Override
			public void finished(ReadGraphImpl graph) {
				
		    	// Shall fire exactly once!
		    	if(done.compareAndSet(false, true)) {
		    		try {
		    			if(result == 0) {
							procedure.exception(graph, new NoInverseException(""));
//		    				impl.state.barrier.dec(this);
		    			} else {
		        			procedure.execute(graph, querySupport.getResource(result));
//		    				impl.state.barrier.dec(this);
		    			}
		    		} catch (Throwable t) {
		        		Logger.defaultLogError(t);
		    		}
		    	}
		    	
			}

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				
				if(found.compareAndSet(false, true)) {
					this.result = i;
				} else {
					// Shall fire exactly once!
					if(done.compareAndSet(false, true)) {
						try {
							procedure.exception(graph, new ManyObjectsForFunctionalRelationException("Multiple items e.g. " + this.result + " and " + result));
//							impl.state.barrier.dec(this);
						} catch (Throwable t) {
				    		Logger.defaultLogError(t);
						}
					}
				}
				
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				// Shall fire exactly once!
				if(done.compareAndSet(false, true)) {
					try {
						procedure.exception(graph, t);
//						impl.state.barrier.dec(this);
					} catch (Throwable t2) {
			    		Logger.defaultLogError(t2);
					}
				}
			}

		};

		int sId = querySupport.getId(relation);

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#DirectObjects#" + sId);
//		else impl.state.barrier.inc(null, null);

		try {
			QueryCache.runnerObjects(impl, sId, getInverseOf(), impl.parent, listener, ip);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forResource(final ReadGraphImpl impl, final String id, final AsyncProcedure<Resource> procedure) {

		assert(id != null);
		assert(procedure != null);

		InternalProcedure<Integer> ip = new InternalProcedure<Integer>() {

			@Override
			public void execute(ReadGraphImpl graph, Integer result) {
				try {
					procedure.execute(graph, querySupport.getResource(result));
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}   

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {

				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

		};

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "Resource");
//		else impl.state.barrier.inc(null, null);

		forResource(impl, id, impl.parent, ip);

	}

	@Override
	final public void forBuiltin(final ReadGraphImpl impl, final String id, final AsyncProcedure<Resource> procedure) {

		assert(id != null);
		assert(procedure != null);

//		impl.state.barrier.inc();

		try {
			forBuiltin(impl, id, impl.parent, new InternalProcedure<Integer>() {

				@Override
				public void execute(ReadGraphImpl graph, Integer result) {
					try {
						procedure.execute(graph, querySupport.getResource(result)); 
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}   

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forHasStatement(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Boolean> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		try {
			IntSet result = QueryCache.resultDirectPredicates(impl, querySupport.getId(subject), impl.parent, listener);
			procedure.execute(impl, !result.isEmpty());
		} catch (DatabaseException e) {
			procedure.exception(impl, e);
		}
			
	}

	@Override
	final public void forHasStatement(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final AsyncProcedure<Boolean> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

		AsyncMultiProcedure<Resource> ip = new AsyncMultiProcedureAdapter<Resource>() {

			boolean found = false;

			@Override
			synchronized public void execute(AsyncReadGraph graph, Resource resource) {
				found = true;
			}

			@Override
			synchronized public void finished(AsyncReadGraph graph) {
				try {
					procedure.execute(graph, found);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec(this);
			}

		};

//		if(AsyncBarrierImpl.BOOKKEEPING) impl.state.barrier.inc(ip, "#ForEachObject#" + subject + "#" + predicate);
//		else impl.state.barrier.inc(null, null);

		forEachObject(impl, subject, predicate, ip);

	}

	@Override
	final public void forHasStatement(final ReadGraphImpl impl, final Resource subject, final Resource predicate, final Resource object, final AsyncProcedure<Boolean> procedure) {

		assert(subject != null);
		assert(predicate != null);
		assert(procedure != null);

//		impl.state.barrier.inc();

		forEachObject(impl, subject, predicate, new AsyncMultiProcedureAdapter<Resource>() {

			boolean found = false;

			@Override
			synchronized public void execute(AsyncReadGraph graph, Resource resource) {
				if(resource.equals(object)) found = true;
			}

			@Override
			synchronized public void finished(AsyncReadGraph graph) {
				try {
					procedure.execute(graph, found);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
//				impl.state.barrier.dec();
			}

		});

	}

	@Override
	final public void forHasValue(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Boolean> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

//		impl.state.barrier.inc();

		try {
			QueryCache.runnerValueQuery(impl, querySupport.getId(subject), impl.parent, listener, new InternalProcedure<byte[]>() {

				@Override
				public void execute(ReadGraphImpl graph, byte[] object) {
					boolean result = object != null;
					try {
						procedure.execute(graph, result);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forOrderedSet(final ReadGraphImpl impl, final Resource subject, final AsyncMultiProcedure<Resource> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		try {
			
			QueryCache.runnerOrderedSet(impl, querySupport.getId(subject), impl.parent, listener, new IntProcedure() {

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

				@Override
				public void execute(ReadGraphImpl graph, int i) {
					try {
						procedure.execute(graph, querySupport.getResource(i));
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					try {
						procedure.finished(graph);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
//				impl.state.barrier.dec();
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

	@Override
	final public void forChildMap(final ReadGraphImpl impl, final Resource subject, final AsyncProcedure<Map<String,Resource>> procedure) {

		assert(subject != null);
		assert(procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		try {
			
			QueryCache.runnerChildMap(impl, querySupport.getId(subject), impl.parent, listener, new InternalProcedure<ObjectResourceIdMap<String>>() {

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					try {
						procedure.exception(graph, t);
					} catch (Throwable t2) {
						Logger.defaultLogError(t2);
					}
				}

				@Override
				public void execute(ReadGraphImpl graph, ObjectResourceIdMap<String> result) throws DatabaseException {
					procedure.execute(graph, result);
				}

			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}

	}

//	@Override
//	final public <T> void query(final ReadGraphImpl impl, final AsyncRead<T> request, final CacheEntry parent, final AsyncProcedure<T> procedure, ListenerBase listener) throws DatabaseException {
//
//		assert(request != null);
//		assert(procedure != null);
//
//		QueryCache.runnerAsyncReadEntry(impl, request, parent, listener, procedure);
//
//	}

//	@Override
//	final public <T> T tryQuery(final ReadGraphImpl graph, final Read<T> request) throws DatabaseException {
//
//		assert(graph != null);
//		assert(request != null);
//
//		final ReadEntry entry = (ReadEntry)cache.getCached(request);
//		if(entry != null && entry.isReady()) {
//		    return (T)entry.get(graph, this, null);
//		} else {
//			return request.perform(graph);
//		}
//
//	}

//    final public <T> T tryQuery(final ReadGraphImpl graph, final ExternalRead<T> request) throws DatabaseException {
//
//        assert(graph != null);
//        assert(request != null);
//
//        final ExternalReadEntry<T> entry = cache.externalReadMap.get(request);
//        if(entry != null && entry.isReady()) {
//            if(entry.isExcepted()) {
//                Throwable t = (Throwable)entry.getResult();
//                if(t instanceof DatabaseException) throw (DatabaseException)t;
//                else throw new DatabaseException(t);
//            } else {
//                return (T)entry.getResult();
//            }            
//        } else {
//
//            final DataContainer<T> result = new DataContainer<T>();
//            final DataContainer<Throwable> exception = new DataContainer<Throwable>();
//            
//            request.register(graph, new Listener<T>() {
//                
//                @Override
//                public void exception(Throwable t) {
//                    exception.set(t);
//                }
//
//                @Override
//                public void execute(T t) {
//                    result.set(t);
//                }
//
//                @Override
//                public boolean isDisposed() {
//                    return true;
//                }
//            
//            });
//            
//            Throwable t = exception.get();
//            if(t != null) {
//                if(t instanceof DatabaseException) throw (DatabaseException)t;
//                else throw new DatabaseException(t);
//            }
//            
//            return result.get();
//
//        }
//
//    }
	
//	@Override
//	final public <T> void tryQuery(final ReadGraphImpl graph, final AsyncRead<T> request, AsyncProcedure<T> procedure) {
//
//		assert(graph != null);
//		assert(request != null);
//
//		final AsyncReadEntry entry = cache.asyncReadMap.get(request);
//		if(entry != null && entry.isReady()) {
//			if(entry.isExcepted()) {
//				procedure.exception(graph, (Throwable)entry.getResult());
//			} else {
//				procedure.execute(graph, (T)entry.getResult());
//			}
//		} else {
//			request.perform(graph, procedure);
//		}
//
//	}

	@Override
	final public <T> void query(final ReadGraphImpl impl, final MultiRead<T> request, final CacheEntry parent, final SyncMultiProcedure<T> procedure, ListenerBase listener) {

		assert(request != null);
		assert(procedure != null);

		try {

			queryMultiRead(impl, request, parent, listener, procedure);
			
		} catch (DatabaseException e) {
			
			throw new IllegalStateException(e);
			
		}

	}

	@Override
	final public <T> void query(final ReadGraphImpl impl, final AsyncMultiRead<T> request, final CacheEntry parent, final AsyncMultiProcedure<T> procedure, ListenerBase listener) {

		assert(request != null);
		assert(procedure != null);

//		impl.state.barrier.inc();

		runAsyncMultiRead(impl, request, parent, listener, new AsyncMultiProcedure<T>() {

			public void execute(AsyncReadGraph graph, T result) {

				try {
					procedure.execute(graph, result);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}
			}

			@Override
			public void finished(AsyncReadGraph graph) {

				try {
					procedure.finished(graph);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}

//				impl.state.barrier.dec();

			}

			@Override
			public String toString() {
				return procedure.toString();
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {

				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
					Logger.defaultLogError(t2);
				}

//				impl.state.barrier.dec();

			}

		});

	}

//	@Override
//	final public <T> void query(final ReadGraphImpl impl, final ExternalRead<T> request, final CacheEntry parent, final Procedure<T> procedure, ListenerBase listener) throws DatabaseException {
//
//		assert(request != null);
//		assert(procedure != null);
//
//		try {
//		
//			queryPrimitiveRead(impl, request, parent, listener, new AsyncProcedure<T>() {
//	
//				@Override
//				public String toString() {
//					return procedure.toString();
//				}
//	
//				@Override
//				public void execute(AsyncReadGraph graph, T result) {
//					try {
//						procedure.execute(result);
//					} catch (Throwable t2) {
//						Logger.defaultLogError(t2);
//					}
//				}
//
//				@Override
//				public void exception(AsyncReadGraph graph, Throwable throwable) {
//					try {
//						procedure.exception(throwable);
//					} catch (Throwable t2) {
//						Logger.defaultLogError(t2);
//					}
//				}
//	
//			});
//			
//		} catch (DatabaseException e) {
//			
//			throw new IllegalStateException(e);
//			
//		}
//
//	}

	@Override
	public VirtualGraph getProvider(Resource subject, Resource predicate, Resource object) {
		
		return querySupport.getProvider(querySupport.getId(subject), querySupport.getId(predicate), querySupport.getId(object));
		
	}
	
	@Override
	public VirtualGraph getProvider(Resource subject, Resource predicate) {
		
		return querySupport.getProvider(querySupport.getId(subject), querySupport.getId(predicate));
		
	}

	@Override
	public VirtualGraph getValueProvider(Resource subject) {
		
		return querySupport.getValueProvider(querySupport.getId(subject));
		
	}

	public boolean isImmutableForReading(int resourceId) {
		return querySupport.isImmutable(resourceId, false);
	}

	public boolean isImmutableForWriting(Resource resource) {
		ResourceImpl impl = (ResourceImpl)resource;
		return querySupport.isImmutable(impl.id, true);
	}
	
	private Layer0 L0;
	
	public Layer0 getL0(ReadGraph graph) {
		if(L0 == null) {
			L0 = Layer0.getInstance(graph);
		}
		return L0;
	}

	public Layer0 getL0() {
		return L0;
	}

    public static ThreadLocal<Integer> thread = new ThreadLocal<Integer>() {
        protected Integer initialValue() {
            return -1;
        }
    };

    static class Barrier {
        Semaphore semaphore = new Semaphore(0);
        Runnable task;
        Barrier(Runnable task) {
            this.task = task;
        }
    }

    private LinkedList<Barrier> barriers = new LinkedList<>();
    
    public Barrier popBarrier() {
        synchronized(barriers) {
            if(barriers.isEmpty())
                return null;
            return barriers.removeFirst();
        }
    }

    public void executeBarrier() {

        Barrier b = popBarrier();
        if(b != null) {
            b.task.run();
            b.semaphore.release();
        }

    }
    
    public void waitBarrier(Runnable r) {

        Barrier b = new Barrier(r);
        synchronized(barriers) {
            barriers.add(b);
        }

        try {

            boolean executed = false;

            synchronized (querySupportLock) {

                //Indicate that this thread is done
                if(sleepers.addAndGet(1) == THREADS) {
                    // We are the last one - just execute!
                    executeBarrier();
                    executed = true;
                }

            }

            try {
                b.semaphore.acquire();
            } catch (InterruptedException e) {
                LOGGER.error("Error while waiting for a barrier", e);
            }

        } finally {
            sleepers.decrementAndGet();
        }
    }

}
