/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.Deque;
import java.util.LinkedList;
import java.util.Objects;

import org.simantics.databoard.Bindings;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.QueryFactoryKey;
import org.simantics.db.request.RequestFlags;
import org.simantics.utils.Development;

final public class ExternalReadEntry<T> extends CacheEntryBase<AsyncProcedure<T>> implements Listener<T> {

    final Deque<T> items = new LinkedList<T>();

    protected ExternalRead<T> id;
    protected QueryProcessor processor;
    protected boolean registered = false;

    @Override
    int makeHash() {
    	return id.hashCode();
    }
    
    @Override
    public Object getOriginalRequest() {
        return id;
    }
    
    @Override
    public void clearResult(QuerySupport support) {
    }
    
    @Override
    public void discard() {
        id.unregistered();
        id = null;
        processor = null;
        super.discard();
    }

    @Override
    public final void refute() {
        // No point in refuting external read results since the query system does not have any idea of how external read values get computed and recomputed
        // Don't call super to ignore internal refutes.
    }

    @Override
    public void setPending(QuerySupport querySupport) {
        //if(result != NO_RESULT) {
            //new Exception("result = " + result).printStackTrace();
        //}
        statusOrException = PENDING;
        result = REQUIRES_COMPUTATION;
    }
    
    public ExternalReadEntry(ExternalRead<T> request, ReadGraphImpl graph) {
        assert request != null;
        this.id = request;
        this.processor = graph.processor;
    }
    
    @Override
    public void except(Throwable t) {
    	
		if (Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.CACHE_ENTRY_STATE, Bindings.BOOLEAN)) {
				System.err.println("[QUERY STATE]: excepted " + this);
			}
		}

        if(statusOrException != DISCARDED) {
            statusOrException = EXCEPTED;
            result = t;
        } else {
            result = t;
        }
        
        assert(isExcepted());
        
    }
    
    @Override
    public void setResult(Object result) {

        super.setResult(result);
        assert(!(result instanceof Throwable));
        assert(!isExcepted());
        
    }
    
    @Override
    public void setReady() {
    	super.setReady();
    }

    @Override
    final public Query getQuery() {
    	
        return new Query() {

			@Override
			public void recompute(ReadGraphImpl graph) {

				synchronized(items) {

					// Update
					if(!items.isEmpty()) {
					    setReady();
						setResult(items.removeFirst());
					}
					// Reschedule
					if(!items.isEmpty()) {
						graph.processor.updatePrimitive(id);
					}

				}
				
			}

			@Override
			public void removeEntry(QueryProcessor processor) {
				processor.cache.remove(ExternalReadEntry.this);
			}

			@Override
			public int type() {
				return RequestFlags.IMMEDIATE_UPDATE;
			}
			
			@Override
			public String toString() {
				var lid = id;
				if(lid == null) return "DISCARDED ExternalRead";
				else return lid.toString();
			}
        	
        };
        
    }

	@Override
	public String toString() {
		var lid = id;
		if(lid == null) return "DISCARDED ExternalRead " + System.identityHashCode(this);
		else return lid.toString() + " " + System.identityHashCode(this);
	}

    @SuppressWarnings("unchecked")
    @Override
    public Object performFromCache(ReadGraphImpl graph, AsyncProcedure<T> procedure) {

        AsyncProcedure<T> proc = (AsyncProcedure<T>)procedure;

        if(isExcepted()) {

            proc.exception(graph, (Throwable)getResult());

        } else {

            proc.execute(graph, (T)getResult());

        }

        return getResult();

    }

    @Override
    void prepareRecompute(QuerySupport querySupport) {
    	// Do nothing - the state is already set and cannot be recomputed on demand
    }

    public Object compute(ReadGraphImpl graph, AsyncProcedure<T> procedure) throws DatabaseException {

        ReadGraphImpl queryGraph = graph.withParent(this, null, true);
        try {
            if(!registered) {
                id.register(queryGraph, this);
                registered = true;
            }
        } catch (Throwable t) {
            except(t);
        } finally {
            queryGraph.asyncBarrier.dec();
            queryGraph.asyncBarrier.waitBarrier(id, queryGraph);
        }

        performFromCache(graph, procedure);

        return getResult();

    }

	@Override
	public void execute(T result) {

		if(this.result == REQUIRES_COMPUTATION) {

			setResult(result);
			setReady();

		} else {

			synchronized(items) {

				if(!items.isEmpty()) {
					T last = items.getLast();
					if(!result.equals(last)) {
						items.addLast(result);
						processor.updatePrimitive(id);
					}
				} else {
					T last = getResult();
					if(!Objects.equals(last, result)) {
						items.addLast(result);
						processor.updatePrimitive(id);
					}
				}
				// TODO: implement flags/logic in ExternalRead to state that all but the latest request result can be evaporated
				// In some cases where data is produced really fast this might be necessary but currently this queuing will do.
			}

		}

	}

	@Override
	public void exception(Throwable t) {
		except(t);
	}

	@Override
	public boolean isDisposed() {
		return registered && (isDiscarded() || !processor.isBound(this));
	}

    @Override
    public QueryFactoryKey classId() {
        return null;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ExternalReadEntry<?> other = (ExternalReadEntry<?>) obj;
        return Objects.equals(id, other.id);
    }

}
