/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.impl.graph;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import org.eclipse.core.runtime.Platform;
import org.simantics.databoard.Accessors;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.Accessor;
import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.RuntimeBindingConstructionException;
import org.simantics.databoard.binding.impl.ObjectVariantBinding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.binary.BinaryFile;
import org.simantics.databoard.util.binary.RandomAccessBinary;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ComputationalValue;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.ExternalValueSupport;
import org.simantics.db.GraphHints;
import org.simantics.db.ReadGraph;
import org.simantics.db.RelationContext;
import org.simantics.db.RelationInfo;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.adaption.AdaptionService;
import org.simantics.db.common.primitiverequest.Adapter;
import org.simantics.db.common.primitiverequest.Builtin;
import org.simantics.db.common.primitiverequest.DatatypeBinding;
import org.simantics.db.common.primitiverequest.ForEachAssertedObject;
import org.simantics.db.common.primitiverequest.ForEachAssertedStatement;
import org.simantics.db.common.primitiverequest.HasStatement;
import org.simantics.db.common.primitiverequest.HasStatementSubject;
import org.simantics.db.common.primitiverequest.HasStatementSubjectObject;
import org.simantics.db.common.primitiverequest.HasValue;
import org.simantics.db.common.primitiverequest.Inverse;
import org.simantics.db.common.primitiverequest.IsInheritedFrom;
import org.simantics.db.common.primitiverequest.IsInstanceOf;
import org.simantics.db.common.primitiverequest.IsSubrelationOf;
import org.simantics.db.common.primitiverequest.OrderedSet;
import org.simantics.db.common.primitiverequest.PossibleAdapter;
import org.simantics.db.common.primitiverequest.PossibleInverse;
import org.simantics.db.common.primitiverequest.PossibleObject;
import org.simantics.db.common.primitiverequest.PossibleRelatedValue;
import org.simantics.db.common.primitiverequest.PossibleRelatedValueImplied;
import org.simantics.db.common.primitiverequest.PossibleStatement;
import org.simantics.db.common.primitiverequest.PossibleType;
import org.simantics.db.common.primitiverequest.PossibleUniqueAdapter;
import org.simantics.db.common.primitiverequest.PossibleValue;
import org.simantics.db.common.primitiverequest.PossibleValueImplied;
import org.simantics.db.common.primitiverequest.RelatedValue;
import org.simantics.db.common.primitiverequest.RelatedValueImplied;
import org.simantics.db.common.primitiverequest.SingleObject;
import org.simantics.db.common.primitiverequest.SingleStatement;
import org.simantics.db.common.primitiverequest.SingleType;
import org.simantics.db.common.primitiverequest.SingleTypeAny;
import org.simantics.db.common.primitiverequest.Types;
import org.simantics.db.common.primitiverequest.UniqueAdapter;
import org.simantics.db.common.primitiverequest.Value;
import org.simantics.db.common.primitiverequest.ValueImplied;
import org.simantics.db.common.primitiverequest.VariantValueImplied;
import org.simantics.db.common.procedure.adapter.AsyncMultiProcedureAdapter;
import org.simantics.db.common.procedure.adapter.AsyncProcedureAdapter;
import org.simantics.db.common.procedure.adapter.ProcedureAdapter;
import org.simantics.db.common.procedure.adapter.SyncMultiProcedureAdapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.procedure.single.SyncReadProcedure;
import org.simantics.db.common.procedure.single.wrapper.DeepSingleOrErrorProcedure;
import org.simantics.db.common.procedure.single.wrapper.DeepSingleOrNullProcedure;
import org.simantics.db.common.procedure.single.wrapper.ExceptionToNullProcedure;
import org.simantics.db.common.procedure.single.wrapper.NullSingleOrNullProcedure;
import org.simantics.db.common.procedure.single.wrapper.SingleFunctionalOrNullProcedure;
import org.simantics.db.common.procedure.single.wrapper.SingleOrErrorProcedure;
import org.simantics.db.common.procedure.single.wrapper.SingleOrNullProcedure;
import org.simantics.db.common.procedure.wrapper.NoneToAsyncListener;
import org.simantics.db.common.procedure.wrapper.NoneToAsyncMultiListener;
import org.simantics.db.common.procedure.wrapper.NoneToAsyncMultiProcedure;
import org.simantics.db.common.procedure.wrapper.NoneToAsyncProcedure;
import org.simantics.db.common.procedure.wrapper.NoneToAsyncSetProcedure;
import org.simantics.db.common.procedure.wrapper.NoneToSyncMultiListener;
import org.simantics.db.common.procedure.wrapper.NoneToSyncMultiProcedure;
import org.simantics.db.common.procedure.wrapper.SyncToAsyncListener;
import org.simantics.db.common.procedure.wrapper.SyncToAsyncMultiListener;
import org.simantics.db.common.procedure.wrapper.SyncToAsyncMultiProcedure;
import org.simantics.db.common.procedure.wrapper.SyncToAsyncProcedure;
import org.simantics.db.common.procedure.wrapper.SyncToAsyncSetProcedure;
import org.simantics.db.common.request.AdaptValue;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.validation.L0Validations;
import org.simantics.db.exception.AdaptionException;
import org.simantics.db.exception.ArgumentException;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.DoesNotContainValueException;
import org.simantics.db.exception.EmptyResourceException;
import org.simantics.db.exception.InternalException;
import org.simantics.db.exception.InvalidLiteralException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.NoInverseException;
import org.simantics.db.exception.NoSingleResultException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.impl.RelationContextImpl;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.internal.RandomAccessValueSupport;
import org.simantics.db.impl.internal.ResourceData;
import org.simantics.db.impl.procedure.ResultCallWrappedSyncQueryProcedure;
import org.simantics.db.impl.query.CacheEntry;
import org.simantics.db.impl.query.QueryCache;
import org.simantics.db.impl.query.QueryCacheBase;
import org.simantics.db.impl.query.QueryProcessor;
import org.simantics.db.impl.query.QueryProcessor.SessionTask;
import org.simantics.db.impl.query.QuerySupport;
import org.simantics.db.impl.query.TripleIntProcedure;
import org.simantics.db.impl.support.ResourceSupport;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.AsyncMultiListener;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.AsyncSetListener;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.procedure.MultiListener;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SetListener;
import org.simantics.db.procedure.StatementProcedure;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.procedure.SyncMultiListener;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procedure.SyncProcedure;
import org.simantics.db.procedure.SyncSetListener;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.DelayedWrite;
import org.simantics.db.request.DelayedWriteResult;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;
import org.simantics.db.request.ReadInterface;
import org.simantics.db.request.Write;
import org.simantics.db.request.WriteInterface;
import org.simantics.db.request.WriteOnly;
import org.simantics.db.request.WriteOnlyResult;
import org.simantics.db.request.WriteResult;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.exceptions.SCLTypeParseException;
import org.simantics.scl.reflection.ReflectionUtils;
import org.simantics.scl.reflection.ValueNotFoundException;
import org.simantics.scl.runtime.function.Function3;
import org.simantics.utils.DataContainer;
import org.simantics.utils.Development;
import org.simantics.utils.datastructures.ArrayMap;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.collections.CollectionUtils;
import org.slf4j.LoggerFactory;

import gnu.trove.ext.TIntArrayListModified;
import gnu.trove.map.hash.TObjectIntHashMap;

public class ReadGraphImpl implements AsyncReadGraph {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(ReadGraphImpl.class);

    final static boolean EMPTY_RESOURCE_CHECK = false;
    
	final public CacheEntry parent;
	public final ReadGraphImpl parentGraph;
	final public QueryProcessor processor;
	
	public final AsyncBarrierImpl asyncBarrier;
	
	final static Binding DATA_TYPE_BINDING_INTERNAL = Bindings.getBindingUnchecked(Datatype.class);
	final static Serializer DATA_TYPE_SERIALIZER = Bindings.getSerializerUnchecked(DATA_TYPE_BINDING_INTERNAL);

	final public static TObjectIntHashMap<String> counters = new TObjectIntHashMap<String>(); 
	
	public static void resetCounters() {
		counters.clear();
	}
	
	public static String listCounters(File file) throws IOException {
		
		PrintStream b = new PrintStream(new BufferedOutputStream(new FileOutputStream(file)));

		for(Pair<String,Integer> p : CollectionUtils.valueSortedEntries(counters)) {
			b.print(-p.second + " " + p.first + "\n");
		}

		b.close();

		return "Dumped " + counters.size() + " queries.";
		
	}
	
	/*
	 * Implementation of the interface ReadGraph
	 */
	final public String getURI(final Resource resource)	throws AssumptionException, ValidationException, ServiceException {
		
		assert (resource != null);

		try {

			return syncRequest(new org.simantics.db.common.uri.ResourceToURI(resource));

		} catch (AssumptionException e) {

			throw new AssumptionException(e);

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}
	
	final public String getPossibleURI(final Resource resource)	throws ValidationException,	ServiceException {

		assert (resource != null);

		try {

			return syncRequest(new org.simantics.db.common.uri.ResourceToPossibleURI(resource));

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	final public Resource getResource(final String id)
			throws ResourceNotFoundException, ValidationException,
			ServiceException {

		assert (id != null);

		try {

			Integer rid = QueryCache.resultURIToResource(this, id, parent, null);
			// FIXME: stupid to throw this here and catch and wrap it right away
			if(rid == 0) throw new ResourceNotFoundException(id);
			return processor.querySupport.getResource(rid);

		} catch (ResourceNotFoundException e) {

			throw new ResourceNotFoundException(id, e);

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	final public Resource getPossibleResource(final String id)
	throws ResourceNotFoundException, ValidationException,
	ServiceException {

		assert (id != null);

		try {

			return getResource(id);

		} catch (ResourceNotFoundException e) {
			
			return null;

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}
	
	@Override
	public void forChildMap(Resource resource, AsyncProcedure<Map<String, Resource>> procedure) {
		
		assert (resource != null);
		assert (procedure != null);

		processor.forChildMap(this, resource, procedure);

	}
	
	@Override
	public Map<String, Resource> getChildren(Resource resource) throws ValidationException, ServiceException {
		
		assert (resource != null);

		try {

			int rId = processor.querySupport.getId(resource);
			return QueryCache.resultChildMap(this, rId, parent, null);

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}
		
	}

	final public Resource getRootLibrary() {
		return processor.getRootLibraryResource();
	}
	
	final public Resource getBuiltin(final String id)
			throws ResourceNotFoundException, ServiceException {

		assert (id != null);

		try {

			return syncRequest(new Builtin(id));

		} catch (ResourceNotFoundException e) {

			throw new ResourceNotFoundException(id, e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	static class StatementReadProcedure extends TIntArrayListModified implements StatementProcedure {

		private static Throwable DONE = new Throwable();
		
		Throwable exception = null;
		
		final ResourceSupport support;
		
		public StatementReadProcedure(ResourceSupport support) {
			this.support = support;
		}
		
		@Override
		public synchronized void execute(AsyncReadGraph graph, int s, int p, int o) {
			add(s);
			add(p);
			add(o);
		}
		
		@Override
		public void finished(AsyncReadGraph graph) {
			exception = DONE;
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable t) {
			exception = t;
		}
		
		public void checkAndThrow() throws DatabaseException {
			if(exception != DONE) {
				if (exception instanceof DatabaseException)
					throw (DatabaseException) exception;
				else
					throw new DatabaseException(
							"Unexpected exception in ReadGraph.syncRequest(AsyncMultiRead)",
							exception);
			}
		}
		
		public boolean done() {
			return exception != null;
		}

		@Override
		public boolean contains(Object obj) {
		    if(!(obj instanceof InternalStatement))
		        return false;
		    InternalStatement statement = (InternalStatement)obj;
		    int s = statement.s;
		    int p = statement.p;
		    int o = statement.o;
		    for(int i=0;i<sizeInternal();i+=3)
                if(s==getQuick(i) && p==getQuick(i+1) && o==getQuick(i+2))
                    return true;
		    return false;
		}

		@SuppressWarnings("unchecked")
        @Override
		public <T> T[] toArray(T[] a) {
		    int length = sizeInternal() / 3;
		    if(length > a.length) {
		        Class<?> arrayType = a.getClass();
		        a = (arrayType == Object[].class) 
		                ? (T[]) new Object[length]
		                : (T[]) Array.newInstance(arrayType.getComponentType(), length);
		    }
		    else {
		        for(int i=length;i<a.length;++i)
		            a[i] = null;
		    }
            for(int i=0,j=0;i<sizeInternal();i+=3,++j)
                a[j] = (T)new InternalStatement(support, getQuick(i), getQuick(i+1), getQuick(i+2));
            return a;
		}

		@Override
		public boolean add(Statement e) {
			throw new UnsupportedOperationException();
		}

		@Override
		public boolean remove(Object o) {
			throw new UnsupportedOperationException();
		}

		@Override
		public boolean addAll(Collection<? extends Statement> c) {
			throw new UnsupportedOperationException();
		}

		class IteratorImpl implements ListIterator<Statement> {
            
            int index;
            
            public IteratorImpl(int index) {
                this.index = index;
            }

            @Override
            public boolean hasNext() {
                return index < sizeInternal(); 
            }

            @Override
            public Statement next() {
                Statement result = new InternalStatement(support, getQuick(index), getQuick(index+1), getQuick(index+2)); 
                index += 3;
                return result;
            }

            @Override
            public void remove() {
                throw new Error("Not supported");
            }

            @Override
            public boolean hasPrevious() {
                return index > 0;
            }

            @Override
            public Statement previous() {
                index -= 3;
                Statement result = new InternalStatement(support, getQuick(index), getQuick(index+1), getQuick(index+2)); 
                return result;
            }

            @Override
            public int nextIndex() {
                return index/3;
            }

            @Override
            public int previousIndex() {
                return index/3-1;
            }

            @Override
            public void set(Statement e) {
                throw new UnsupportedOperationException();
            }

            @Override
            public void add(Statement e) {
                throw new UnsupportedOperationException();
            }
            
        };
        
		@Override
		public Iterator<Statement> iterator() {
			return new IteratorImpl(0);
		}
		
		@Override
		public int size() {
			return sizeInternal() / 3;
		}

		@Override
		public Object[] toArray() {
		    Object[] result = new Object[sizeInternal() / 3];
		    for(int i=0,j=0;j<sizeInternal();i++,j+=3)
		        result[i] = new InternalStatement(support, getQuick(j), getQuick(j+1), getQuick(j+2));
			return result;
		}

        @Override
        public boolean addAll(int index, Collection<? extends Statement> c) {
            throw new UnsupportedOperationException();
        }

        @Override
        public Statement get(int index) {
            index += 3;
            if(index < 0 || index >= sizeInternal())
                throw new IndexOutOfBoundsException();
            return new InternalStatement(support, getQuick(index), getQuick(index+1), getQuick(index+2));
        }

        @Override
        public Statement set(int index, Statement element) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void add(int index, Statement element) {
            throw new UnsupportedOperationException();
        }

        @Override
        public Statement remove(int index) {
            throw new UnsupportedOperationException();
        }

        @Override
        public int indexOf(Object obj) {
            if(!(obj instanceof InternalStatement))
                return -1;
            InternalStatement statement = (InternalStatement)obj;
            int s = statement.s;
            int p = statement.p;
            int o = statement.o;
            for(int i=0;i<sizeInternal();i+=3)
                if(s==getQuick(i) && p==getQuick(i+1) && o==getQuick(i+2))
                    return i/3;
            return -1;
        }

        @Override
        public int lastIndexOf(Object obj) {
            if(!(obj instanceof InternalStatement))
                return -1;
            InternalStatement statement = (InternalStatement)obj;
            int s = statement.s;
            int p = statement.p;
            int o = statement.o;
            for(int i=sizeInternal()-3;i>=0;i-=3)
                if(s==getQuick(i) && p==getQuick(i+1) && o==getQuick(i+2))
                    return i/3;
            return -1;
        }

        @Override
        public ListIterator<Statement> listIterator() {
            return new IteratorImpl(0);
        }

        @Override
        public ListIterator<Statement> listIterator(int index) {
            return new IteratorImpl(index*3);
        }

        @Override
        public List<Statement> subList(int fromIndex, int toIndex) {
            if(fromIndex < 0 || toIndex*3 >= sizeInternal() || fromIndex > toIndex)
                throw new IndexOutOfBoundsException();
            return new RandomAccessSubList<Statement>(this, fromIndex, toIndex-fromIndex);
        }
	}
	
	@Override
	final public Collection<Statement> getStatements(final Resource subject,
			final Resource relation)
			throws ManyObjectsForFunctionalRelationException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {

			StatementReadProcedure procedure = new StatementReadProcedure(getResourceSupport());
			processor.forEachStatement(this, subject, relation, procedure);
			procedure.checkAndThrow();
			return procedure;
			
		} catch (DatabaseException e) {

			System.err.println(INTERNAL_ERROR_STRING + " getStatements " + subject + " " + relation);

			StatementReadProcedure procedure = new StatementReadProcedure(getResourceSupport());
			processor.forEachStatement(this, subject, relation, procedure);
			
			return Collections.emptyList();
		
//			throw new ServiceException(INTERNAL_ERROR_STRING + " getStatements " + subject + " " + relation, e);

		}

	}

	@Override
	final public Collection<Statement> getAssertedStatements(final Resource subject, final Resource relation)
			throws ManyObjectsForFunctionalRelationException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {

			return syncRequest(new ForEachAssertedStatement(subject, relation));

		} catch (ManyObjectsForFunctionalRelationException e) {

			throw new ManyObjectsForFunctionalRelationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Collection<Resource> getPredicates(final Resource subject) throws ServiceException {

		assert (subject != null);

		try {

			return processor.getPredicates(this, subject);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}  catch (Throwable e) {

			throw new ServiceException(e);

		}

	}

	@Override
	final public Collection<Resource> getDirectPredicates(final Resource subject) throws ServiceException {

		assert (subject != null);

		try {

			return processor.getDirectPredicates(this, subject);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}  catch (Throwable e) {

			throw new ServiceException(e);

		}

	}

	@Override
	final public Collection<Resource> getPrincipalTypes(final Resource subject)
			throws ServiceException {

		assert (subject != null);

		try {

			AsyncMultiReadProcedure<Resource> procedure = new AsyncMultiReadProcedure<Resource>();
			processor.forEachPrincipalType(this, subject, procedure);
			procedure.checkAndThrow();
			return procedure;
			
		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Set<Resource> getTypes(final Resource subject) throws ServiceException {

		assert (subject != null);

		try {
			
			return processor.getTypes(this, subject);
			
		} catch (ServiceException e) {

			throw new ServiceException(e);
			
		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

        } catch (Throwable e) {

            throw new ServiceException(e);

		}

	}

	@Override
	final public Set<Resource> getSupertypes(final Resource subject)
			throws ServiceException {

		assert (subject != null);

		try {

			SyncReadProcedure<Set<Resource>> procedure = new SyncReadProcedure<Set<Resource>>();
			processor.forSupertypes(this, subject, procedure);
			procedure.checkAndThrow();
			return procedure.result;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Set<Resource> getSuperrelations(final Resource subject)
			throws ServiceException {

		assert (subject != null);

		try {

			SyncReadProcedure<Set<Resource>> procedure = new SyncReadProcedure<Set<Resource>>();
			processor.forSuperrelations(this, subject, procedure);
			procedure.checkAndThrow();
			return procedure.result;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}
	
	@Override
	public Resource getPossibleSuperrelation(Resource subject) throws ServiceException {
		
		try {

			SyncReadProcedure<Resource> procedure = new SyncReadProcedure<Resource>();
			processor.forPossibleSuperrelation(this, subject, procedure);
			procedure.checkAndThrow();
			return procedure.result;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Collection<Resource> getObjects(final Resource subject, final Resource relation)
			throws ServiceException {

		assert (subject != null);
		assert (relation != null);

		if(Development.DEVELOPMENT) {
            if(Development.isTrue(DevelopmentKeys.READGRAPH_COUNT)) {
            	counters.adjustOrPutValue("objects $" + subject.getResourceId() + " $" + relation.getResourceId(), 1, 1);
            }
            //if(subject.getResourceId()==xx && relation.getResourceId()==xx) new Exception().printStackTrace();
		}
		
		try {

			AsyncMultiReadProcedure<Resource> procedure = new AsyncMultiReadProcedure<Resource>();
			processor.forEachObject(this, subject, relation, procedure);
			procedure.checkAndThrow();
			return procedure;
			
		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Collection<Resource> getAssertedObjects(
			final Resource subject, final Resource relation)
			throws ManyObjectsForFunctionalRelationException, ServiceException {

        if (subject == null)
            throw new ArgumentException("Subject must not be null.");
        if (relation == null)
            throw new ArgumentException("Relation must not be null. Subject=" + subject);

		try {

			return syncRequest(new ForEachAssertedObject(subject, relation));

		} catch (ManyObjectsForFunctionalRelationException e) {

			throw new ManyObjectsForFunctionalRelationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Resource getInverse(final Resource relation) throws NoInverseException, ServiceException {

		assert (relation != null);

		try {

			return getSingleObject(relation, processor.querySupport.getResource(processor.getInverseOf()));

		} catch (NoSingleResultException e) {

			throw new NoInverseException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		}

	}

	@Override
	final public Resource getSingleObject(final Resource subject, final Resource relation) throws NoSingleResultException, ServiceException {

		if( subject == null) throw new IllegalArgumentException("subject can not be null");
		if( relation == null) throw new IllegalArgumentException("relation can not be null");

		try {
			int single = processor.getSingleObject(this, subject, relation);
			if (single == 0) {
				if (EMPTY_RESOURCE_CHECK) {
					if (!hasStatement(subject)) {
						throw new EmptyResourceException("Resource " + debugString(subject));
					}
				}
				throw new NoSingleResultException("No single object for subject " + debugString(subject)
						+ " and relation " + debugString(relation), single);
			}
			return processor.querySupport.getResource(single);
		} catch (NoSingleResultException e) {
			throw e;
		} catch (DatabaseException e) {
			throw new ServiceException(e);
		} 
	}

	@Override
	final public Statement getSingleStatement(final Resource subject, final Resource relation) throws NoSingleResultException, ManyObjectsForFunctionalRelationException, ServiceException {
		assert (subject != null);
		assert (relation != null);
		try {
			Collection<Statement> statements = getStatements(subject, relation);
			if (statements.size() == 1) {
				return statements.iterator().next();
			} else {
				if (EMPTY_RESOURCE_CHECK)
					if (!hasStatement(subject))
						throw new EmptyResourceException("Resource " + debugString(subject));
				throw new NoSingleResultException("No single statement for subject " + debugString(subject)
						+ " and relation " + debugString(relation), statements.size());
			}
		} catch (ServiceException e) {
			throw new ServiceException(e);
		} 
	}

	@Override
	final public Resource getSingleType(final Resource subject) throws NoSingleResultException, ServiceException {
		assert (subject != null);
		try {
			ArrayList<Resource> principalTypes = (ArrayList<Resource>)getPrincipalTypes(subject);
			if (principalTypes.size() == 1) {
			    return principalTypes.get(0);
			} else {
			    throw new NoSingleResultException("No single type for subject " + debugString(subject), principalTypes.size());
			}
		} catch (ServiceException e) {
			throw new ServiceException(e);
		} 
	}

	@Override
	final public Resource getSingleType(final Resource subject,
			final Resource baseType) throws NoSingleResultException,
			ServiceException {

		assert (subject != null);
		assert (baseType != null);

		try {
			return syncRequest(new SingleType(subject, baseType));
		} catch (DatabaseException e) {
		    throw new NoSingleResultException("subject=" + subject + ", baseType=" + baseType, 0, e);
		}
	}

	@Override
	final public <T> T getValue(final Resource subject) throws DoesNotContainValueException, ServiceException {

		assert (subject != null);

		try {

			Layer0 L0 = processor.getL0(this);
			int object = processor.getSingleObject(this, subject, L0.HasDataType);
			if(object == 0) throw new DoesNotContainValueException("No data type for " + subject);

			if(processor.isImmutableForReading(object)) {
				Binding binding = syncRequest(new DatatypeBinding(processor.querySupport.getResource(object)), TransientCacheListener.<Binding>instance()); 
				return getValue(subject, binding);
			} else {
				byte[] dt = processor.getValue(this, object);
				if(dt == null) throw new ServiceException("No data type for " + subject);
				Datatype datatype = (Datatype)DATA_TYPE_SERIALIZER.deserialize(dt);
				Binding binding = Bindings.getBinding(datatype);
				return getValue(subject, binding);
			}

		} catch (IOException e) {

			throw new ServiceException(e);

		} catch (DoesNotContainValueException e) {

			throw new DoesNotContainValueException(e, subject);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

    @Override
    final public Variant getVariantValue(final Resource subject) throws DoesNotContainValueException, ServiceException {

        assert (subject != null);

        try {

            Layer0 L0 = processor.getL0(this);
            int object = processor.getSingleObject(this, subject, L0.HasDataType);
            if(object == 0) throw new DoesNotContainValueException("No data type for " + subject);
            
            if(processor.isImmutableForReading(object)) {
                Binding binding = syncRequest(new DatatypeBinding(processor.querySupport.getResource(object)), TransientCacheListener.<Binding>instance()); 
                return new Variant(binding, getValue(subject, binding));
            } else {
                byte[] dt = processor.getValue(this, object);
                if(dt == null) throw new ServiceException("No data type for " + subject);
                Datatype datatype = (Datatype)DATA_TYPE_SERIALIZER.deserialize(dt);
                Binding binding = Bindings.getBinding(datatype);
                return new Variant(binding, getValue(subject, binding));
            }
            
        } catch (IOException e) {

            throw new ServiceException(e);

        } catch (DoesNotContainValueException e) {

            throw new DoesNotContainValueException(e, subject);

        } catch (ServiceException e) {

            throw new ServiceException(e);

        } catch (DatabaseException e) {

            throw new ServiceException(INTERNAL_ERROR_STRING, e);

        }		
	}

	static final IdentityHashMap<Binding,Serializer> serializers = new IdentityHashMap<Binding,Serializer>();
	
	static {
		serializers.put(Bindings.STRING, Bindings.STRING.serializer());
	}
	
	final protected Serializer getSerializer(Binding binding) {
	    return binding.serializer();
	}
	
	@Override
	final public <T> T getValue(final Resource subject, final Binding binding) throws DoesNotContainValueException, BindingException,
			ServiceException {

		assert (subject != null);

		byte[] bytes = null;
		try {
			
			bytes = processor.getValue(this, subject);
			if (bytes == null) throw new DoesNotContainValueException("No value for resource " + subject);

			Serializer serializer = getSerializer(binding);
			return (T)serializer.deserialize(bytes);

		} catch (DoesNotContainValueException e) {

			throw new DoesNotContainValueException(e);

		} catch (Throwable t) {
			throw new ServiceException("Could not getValue for subject " + debugString(subject) + " and binding " + String.valueOf(binding) + " with bytes " + safeArrayToString(bytes), t);
		}
	}

	@Override
	final public <T> T getRelatedValue(final Resource subject, final Resource relation) throws NoSingleResultException,
			DoesNotContainValueException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {
			Resource object = getSingleObject(subject, relation);
			return getValue(object);
		} catch (NoSingleResultException e) {
			throw new NoSingleResultException("No single value found for subject " + debugString(subject) + " and relation " + debugString(relation), e.getResultCount(), e);
		} catch (DoesNotContainValueException e) {
			try {
				Layer0 L0 = processor.getL0(this);
				Resource object = getPossibleObject(subject, relation);
				if(isInstanceOf(object, L0.Value)) {
					if(isInstanceOf(object, L0.Literal)) {
						throw new DoesNotContainValueException(e);
					} else {
						throw new InvalidLiteralException("The object " + object + " is not an instance of L0.Literal (use getRelatedValue2 instead)", e);
					}
				} else {
					throw new DoesNotContainValueException("The object " + object + " is not an instance of L0.Value", e);
				}
			} catch (DoesNotContainValueException e2) {
				throw e2;
			} catch (DatabaseException e2) {
				throw new InternalException("The client failed to analyse the cause of the following exception", e);
			}
		} catch (ServiceException e) {
			throw new ServiceException(e);
		}
	}

    @Override
    final public Variant getRelatedVariantValue(final Resource subject, final Resource relation) throws NoSingleResultException,
            DoesNotContainValueException, ServiceException {

        assert (subject != null);
        assert (relation != null);

        try {
            Resource object = getSingleObject(subject, relation);
            return getVariantValue(object);
        } catch (NoSingleResultException e) {
            throw new NoSingleResultException("No single object for subject " + debugString(subject) + " and relation " + debugString(relation), e.getResultCount(), e);
        } catch (DoesNotContainValueException e) {
            try {
                Layer0 L0 = processor.getL0(this);
                Resource object = getPossibleObject(subject, relation);
                if(isInstanceOf(object, L0.Value)) {
                    if(isInstanceOf(object, L0.Literal)) {
                        throw new DoesNotContainValueException(e);
                    } else {
                        throw new InvalidLiteralException("The object " + object + " is not an instance of L0.Literal (use getRelatedValue2 instead)", e);
                    }
                } else {
                    throw new DoesNotContainValueException("The object " + object + " is not an instance of L0.Value", e);
                }
            } catch (DoesNotContainValueException e2) {
                throw e2;
            } catch (DatabaseException e2) {
                throw new InternalException("The client failed to analyse the cause of the following exception", e);
            }
        } catch (ServiceException e) {
            throw new ServiceException(e);
        } 
    }
    
	@Override
	final public <T> T getRelatedValue(final Resource subject, final Resource relation, final Binding binding)
			throws NoSingleResultException, DoesNotContainValueException, BindingException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {
			Resource object = getSingleObject(subject, relation);
			return getValue(object, binding);
		} catch (NoSingleResultException e) {
		    String message = "";
		    try {
    		    String subjectName = NameUtils.getSafeName(this, subject, true);
    		    String relationName = NameUtils.getSafeName(this, relation, true);
    		    message = "Subject: " + subjectName + ", Relation: " + relationName;
		    } catch (DatabaseException e2) {
		        
		    }
            throw new NoSingleResultException(message, e.getResultCount(), e);
		} catch (DoesNotContainValueException e) {
			throw new DoesNotContainValueException(e);
		} catch (ServiceException e) {
			throw new ServiceException(e);
		}
	}

	@Override
	final public <T> T adapt(final Resource resource, final Class<T> clazz)
			throws AdaptionException, ValidationException, ServiceException {

		assert (resource != null);
		assert (clazz != null);

		try {

			return syncRequest(new Adapter<T>(resource, clazz));

		} catch (AdaptionException e) {

			throw new AdaptionException(e);

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public <T,C> T adaptContextual(final Resource resource, final C context, final Class<C> contextClass, final Class<T> clazz)
			throws AdaptionException, ValidationException, ServiceException {

		assert (resource != null);
		assert (context != null);

		class ContextualAdapter implements AsyncRead<T> {

			final private Resource resource;
			final private C context;
		    final private Class<T> clazz;
		    
		    @Override
		    public int hashCode() {
		        return resource.hashCode() + 31 * (clazz.hashCode() + 41 * context.hashCode());
		    }
		    
		    @Override
		    final public int threadHash() {
		    	return resource.getThreadHash();
		    }
		    
		    @Override
		    public boolean equals(Object object) {
		        if (this == object)
		            return true;
		        else if (object == null)
		            return false;
		        else if (getClass() != object.getClass())
		            return false;
		        ContextualAdapter r = (ContextualAdapter)object;
		        return resource.equals(r.resource) && context.equals(r.context) && clazz.equals(r.clazz);
		    }

		    @Override
		    public int getFlags() {
		        return 0;
		    }
		    
		    public ContextualAdapter(Resource resource, C context, Class<T> clazz) {
		        this.resource = resource;
		        this.context = context;
		        this.clazz = clazz;
		    }

		    @Override
		    public void perform(AsyncReadGraph graph, AsyncProcedure<T> procedure) {
		        
				final AdaptionService service = getSession().peekService(AdaptionService.class);
				if (service == null)
					procedure.exception(graph, new ServiceException("No AdaptionService available")); 
				else
					service.adapt(graph, resource, context, contextClass, clazz, false, procedure); 
		        
		    }

		    @Override
		    public String toString() {
		        return "Adapter for (" + resource + "," + context + ") as " + clazz.getName();
		    }
		    
		}
		
		try {

			return syncRequest(new ContextualAdapter(resource, context, clazz));

		} catch (AdaptionException e) {

			throw new AdaptionException(e);

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}
	
	@Override
	final public <T> T adaptRelated(final Resource resource, final Resource relation, final Class<T> clazz)
			throws AdaptionException, NoSingleResultException, ValidationException, ServiceException {

		assert (resource != null);
		assert (clazz != null);

		Statement stm = getSingleStatement(resource, relation);
		
		Object o = adaptContextual(stm.getObject(), new RelationContextImpl(resource, stm), RelationContext.class, clazz);
		if (clazz.isInstance(o))
			return (T)o;
		throw new AdaptionException("Returned value is not expected class , got " + o.getClass().getName()+ " , expected " + clazz.getName());
		
	}

	@Override
	final public <T> T getPossibleRelatedAdapter(final Resource resource, final Resource relation, final Class<T> clazz)
			throws ValidationException, ServiceException {

		try {
			return adaptRelated(resource, relation, clazz);
		} catch (DatabaseException e) {
			return null;
		}
		
	}

	@Override
	final public <T,C> T getPossibleContextualAdapter(final Resource resource, final C context, final Class<C> contextClass, final Class<T> clazz)
			throws ValidationException, ServiceException {

		assert (resource != null);
		assert (context != null);

		class PossibleContextualAdapter implements AsyncRead<T> {

			final private Resource resource;
			final private C context;
		    final private Class<T> clazz;
		    
		    @Override
		    public int hashCode() {
		        return resource.hashCode() + 31 * (clazz.hashCode() + 41 * context.hashCode());
		    }
		    
		    @Override
		    final public int threadHash() {
		    	return resource.getThreadHash();
		    }
		    
		    @Override
		    public boolean equals(Object object) {
		        if (this == object)
		            return true;
		        else if (object == null)
		            return false;
		        else if (getClass() != object.getClass())
		            return false;
		        PossibleContextualAdapter r = (PossibleContextualAdapter)object;
		        return resource.equals(r.resource) && context.equals(r.context) && clazz.equals(r.clazz);
		    }

		    @Override
		    public int getFlags() {
		        return 0;
		    }
		    
		    public PossibleContextualAdapter(Resource resource, C context, Class<T> clazz) {
		        this.resource = resource;
		        this.context = context;
		        this.clazz = clazz;
		    }

		    @Override
		    public void perform(AsyncReadGraph graph, AsyncProcedure<T> procedure) {
		        
				final AdaptionService service = getSession().peekService(AdaptionService.class);
				if (service == null)
					procedure.exception(graph, new ServiceException("No AdaptionService available")); 
				else
					service.adapt(graph, resource, context, contextClass, clazz, true, procedure); 
		        
		    }

		    @Override
		    public String toString() {
		        return "Possible adapter for (" + resource + "," + context + ") as " + clazz.getName();
		    }
		    
		}
		
		try {

			return syncRequest(new PossibleContextualAdapter(resource, context, clazz));

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}
		
	}

	@Override
	final public <T> T adaptUnique(final Resource resource, final Class<T> clazz)
			throws AdaptionException, ValidationException, ServiceException {

		assert (resource != null);
		assert (clazz != null);

		try {

			return syncRequest(new UniqueAdapter<T>(resource, clazz));

		} catch (AdaptionException e) {

			throw new AdaptionException(e);

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public Resource getPossibleInverse(final Resource relation)
			throws ServiceException {

		assert (relation != null);

		try {

			return getPossibleObject(relation, processor.querySupport.getResource(processor.getInverseOf()));

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	public Resource getPossibleObject(final Resource subject, final Resource relation)
			throws ManyObjectsForFunctionalRelationException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {

			int result = processor.getSingleObject(this, subject, relation);
			if(result == 0) return null;

			return processor.querySupport.getResource(result);

		} catch (ManyObjectsForFunctionalRelationException e) {

			throw new ManyObjectsForFunctionalRelationException("Many objects in " + subject + " for functional relation " + relation);

		} catch (DatabaseException e) {

			throw new ServiceException(e);

		}
		
	}

	@Override
	final public Statement getPossibleStatement(final Resource subject, final Resource relation)
			throws ManyObjectsForFunctionalRelationException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {

			Collection<Statement> statements = getStatements(subject, relation);
			if(statements.size() == 1) return statements.iterator().next();
			else return null;

		} catch (ManyObjectsForFunctionalRelationException e) {

			throw new ManyObjectsForFunctionalRelationException("Many objects in " + subject + " for functional relation " + relation);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} 

	}

	@Override
	final public Resource getPossibleType(final Resource subject, final Resource baseType) throws ServiceException {

		assert (subject != null);
		assert (baseType != null);

		try {

			AsyncReadProcedure<Resource> procedure = new AsyncReadProcedure<Resource>();
			forPossibleType(subject, baseType, procedure);
			procedure.checkAndThrow();
			return procedure.result;			

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public <T> T getPossibleValue(final Resource subject) throws ServiceException {

		assert (subject != null);

		try {

			int object = processor.getSingleObject(this, subject, processor.getL0(this).HasDataType);
			if(object == 0) return null;

			if(processor.isImmutableForReading(object)) {
				Binding binding = syncRequest(new DatatypeBinding(processor.querySupport.getResource(object)), TransientCacheListener.<Binding>instance()); 
				return getPossibleValue(subject, binding);
			} else {
				byte[] dt = processor.getValue(this, object);
				if(dt == null) return null;
				Datatype datatype = (Datatype)DATA_TYPE_SERIALIZER.deserialize(dt);
				Binding binding = Bindings.getBinding(datatype);
				return getPossibleValue(subject, binding);
			}

		} catch (IOException e) {

			throw new ServiceException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public <T> T getPossibleValue(final Resource subject, final Binding binding) throws BindingException, ServiceException {

		assert (subject != null);
		assert (binding != null);

		try {

            byte[] dt = processor.getValue(this, subject);
            if(dt == null) return null;
			Serializer serializer = getSerializer(binding);
            return (T)serializer.deserialize(dt);

        } catch (IOException e) {

            throw new ServiceException(e);
            
		} catch (BindingException e) {

			throw new BindingException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {
			e.printStackTrace();
			throw new ServiceException(INTERNAL_ERROR_STRING, e);
        }

	}

	@Override
	public <T> T getPossibleRelatedValue(final Resource subject, final Resource relation)
			throws ManyObjectsForFunctionalRelationException, ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {

			Resource object = getPossibleObject(subject, relation);
			if(object == null) return null;
			else return getPossibleValue(object);

		} catch (ManyObjectsForFunctionalRelationException e) {

			throw new ManyObjectsForFunctionalRelationException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} 

	}

	@Override
	public <T> T getPossibleRelatedValue(final Resource subject, final Resource relation, final Binding binding)
			throws ManyObjectsForFunctionalRelationException, BindingException, ServiceException {

		assert (subject != null);
		assert (relation != null);
		assert (binding != null);

		try {

			Resource object = getPossibleObject(subject, relation);
			if(object == null) return null;
			else return getPossibleValue(object, binding);

		} catch (ManyObjectsForFunctionalRelationException e) {

			throw new ManyObjectsForFunctionalRelationException(e);

		} catch (BindingException e) {

			throw new BindingException(e);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		}

	}

	@Override
	public <T> T getPossibleAdapter(Resource resource, Class<T> clazz) throws ValidationException, ServiceException {

		assert (resource != null);
		assert (clazz != null);

		try {

			return syncRequest(new PossibleAdapter<T>(resource, clazz));

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (AdaptionException e) {

			return null;

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}
	}

	@Override
	public <T> T getPossibleUniqueAdapter(Resource resource, Class<T> clazz) throws ValidationException, ServiceException {

		assert (resource != null);
		assert (clazz != null);

		try {

			return syncRequest(new PossibleUniqueAdapter<T>(resource, clazz));

		} catch (AdaptionException e) {

			return null;

		} catch (ValidationException e) {

			throw new ValidationException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

    @Override
    final public boolean isInstanceOf(final Resource resource, final Resource type) throws ServiceException {

        assert (resource != null);
        assert (type != null);

        Set<Resource> resources = getTypes(resource);
        // This check was necessary because some of the callers of this method got stuck when the NPE was thrown from here.
        if (null == resources)
            return false;
        
        if(EMPTY_RESOURCE_CHECK) {
            if (resources.isEmpty()) {
                if(!hasStatement(resource)) throw new EmptyResourceException("Resource " + debugString(resource));
            }
        }
        
        return resources.contains(type);

    }

	@Override
	final public boolean isInheritedFrom(final Resource resource, final Resource type) throws ServiceException {

		assert (resource != null);
		assert (type != null);

		try {

			if(resource.equals(type)) return true;
			
			return getSupertypes(resource).contains(type);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} 
		
	}

	@Override
	final public boolean isSubrelationOf(final Resource resource, final Resource type) throws ServiceException {

		assert (resource != null);
		assert (type != null);

		try {

			if(resource.equals(type)) return true;
			
			return getSuperrelations(resource).contains(type);

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} 

	}

	@Override
	final public boolean hasStatement(final Resource subject) throws ServiceException {

		assert (subject != null);

		try {

			SyncReadProcedure<Boolean> procedure = new SyncReadProcedure<Boolean>();
			processor.forHasStatement(this, subject, procedure);
			procedure.checkAndThrow();
			return procedure.result;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	@Override
	final public boolean hasStatement(final Resource subject, final Resource relation) throws ServiceException {

		assert (subject != null);
		assert (relation != null);

		try {

			RelationInfo rinfo = processor.getRelationInfo(this, relation);
			Collection<Resource> predicates = getPredicates(subject);

			if (rinfo.isFinal) {

				return predicates.contains(relation);

			} else if (rinfo.isFunctional) {

				try {
					return processor.getSingleObject(this, subject, relation) != 0;
				} catch (ManyObjectsForFunctionalRelationException e) {
					return true;
				} catch (DatabaseException e) {
					throw new ServiceException(e);
				}

			} else {

				for (Resource predicate : getPredicates(subject)) {
					if (isSubrelationOf(predicate, relation))
						return true;
				}

			}

			return false;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		} 
		
	}

	@Override
	final public boolean hasStatement(final Resource subject, final Resource relation, final Resource object) throws ServiceException {

		assert (subject != null);
		assert (relation != null);
		assert (object != null);

		try {

			for(Resource o : getObjects(subject, relation)) {
				if(object.equals(o)) return true;
			}
			
			return false;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		}

	}

	@Override
	final public boolean hasValue(final Resource subject) throws ServiceException {

		assert (subject != null);

		try {

			SyncReadProcedure<Boolean> procedure = new SyncReadProcedure<Boolean>();
			processor.forHasValue(this, subject, procedure);
			procedure.checkAndThrow();
			return procedure.result;

		} catch (ServiceException e) {

			throw new ServiceException(e);

		} catch (DatabaseException e) {

			throw new ServiceException(INTERNAL_ERROR_STRING, e);

		}

	}

	final AsyncProcedure<?> NONE = new AsyncProcedure<Object>() {

		@Override
		public void execute(AsyncReadGraph graph, Object result) {
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable throwable) {
		}
		
	};
	
	/*
	 * Implementation of the interface RequestProcessor
	 */

    @Override
    public <T> T syncRequest(final Read<T> request) throws DatabaseException {
        assert (request != null);
        return (T)QueryCache.runnerReadEntry(this, request, parent, null, null, true);
    }

	@Override
	public <T> T syncRequest(Read<T> request, SyncListener<T> procedure)
			throws DatabaseException {
		return syncRequest(request, new SyncToAsyncListener<T>(procedure));
	}

	@Override
	public <T> T syncRequest(Read<T> request, final Listener<T> procedure)
			throws DatabaseException {
		return syncRequest(request, new NoneToAsyncListener<T>(procedure));
	}

    @Override
    public <T> T syncRequest(final Read<T> request, final AsyncProcedure<T> procedure) throws DatabaseException {
        assert (request != null);
        ListenerBase listener = procedure != null ? getListenerBase(procedure) : null;
        return (T)QueryCache.runnerReadEntry(this, request, parent, listener, procedure, true);
    }

	@Override
	public <T> T syncRequest(final Read<T> request,
			final SyncProcedure<T> procedure) throws DatabaseException {
		return syncRequest(request, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> T syncRequest(Read<T> request, Procedure<T> procedure)
			throws DatabaseException {
		return syncRequest(request, new NoneToAsyncProcedure<T>(procedure));
	}

	static class AsyncReadProcedure<T> implements AsyncProcedure<T> {

		private static Throwable DONE = new Throwable();
		
		T result = null;
		Throwable exception = null;
		
		@Override
		public void execute(AsyncReadGraph graph, T t) {
			result = t;
			exception = DONE;
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable t) {
			exception = t;
		}
		
		public void checkAndThrow() throws DatabaseException {
			if(exception != DONE) {
				if (exception instanceof DatabaseException)
					throw (DatabaseException) exception;
				else
					throw new DatabaseException(
							"Unexpected exception in ReadGraph.syncRequest(AsyncRead)",
							exception);
			}
		}
		
		public boolean done() {
			return exception != null;
		}
		
	}
	
	@Override
	public <T> T syncRequest(final AsyncRead<T> request)
			throws DatabaseException {

		assert (request != null);
		return syncRequest(request, new AsyncProcedureAdapter<>() );

	}

	@Override
	public <T> T syncRequest(AsyncRead<T> request, AsyncListener<T> procedure)
			throws DatabaseException {
		return syncRequest(request, (AsyncProcedure<T>) procedure);
	}

	@Override
	public <T> T syncRequest(AsyncRead<T> request, SyncListener<T> procedure)
			throws DatabaseException {
		return syncRequest(request, new SyncToAsyncListener<T>(procedure));
	}

	@Override
	public <T> T syncRequest(AsyncRead<T> request, Listener<T> procedure)
			throws DatabaseException {
		return syncRequest(request, new NoneToAsyncListener<T>(procedure));
	}

    @Override
    final public <T> T syncRequest(final AsyncRead<T> request,
            final AsyncProcedure<T> procedure) throws DatabaseException {
        assert (request != null);
        ListenerBase listener = getListenerBase(procedure);
        return (T)QueryCache.runnerAsyncReadEntry(this, request, parent, listener, procedure, true); 
    }

	@Override
	public <T> T syncRequest(AsyncRead<T> request,
			final SyncProcedure<T> procedure) throws DatabaseException {
		return syncRequest(request, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	final public <T> T syncRequest(final AsyncRead<T> request,
			final Procedure<T> procedure) throws DatabaseException {
		return syncRequest(request, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> Collection<T> syncRequest(final MultiRead<T> request)
			throws DatabaseException {

		assert (request != null);

		final ArrayList<T> result = new ArrayList<T>();
		final DataContainer<Throwable> exception = new DataContainer<Throwable>();

		syncRequest(request, new SyncMultiProcedure<T>() {

			@Override
			public void execute(ReadGraph graph, T t) {
				synchronized (result) {
					result.add(t);
				}
			}

			@Override
			public void finished(ReadGraph graph) {
			}

			@Override
			public void exception(ReadGraph graph, Throwable t) {
				exception.set(t);
			}

			@Override
			public String toString() {
				return "syncRequest(MultiRead) -> " + request;
			}

		});

		Throwable t = exception.get();
		if (t != null) {
			if (t instanceof DatabaseException)
				throw (DatabaseException) t;
			else
				throw new DatabaseException(
						"Unexpected exception in ReadGraph.syncRequest(Read)",
						t);
		}

		return result;

	}

	@Override
	public <T> Collection<T> syncRequest(MultiRead<T> request,
			SyncMultiListener<T> procedure) {
		return syncRequest(request, (SyncMultiProcedure<T>)procedure);
	}

	@Override
	public <T> Collection<T> syncRequest(MultiRead<T> request,
			MultiListener<T> procedure) {
		return syncRequest(request, new NoneToSyncMultiListener<T>(procedure));
	}

	@Override
	public <T> Collection<T> syncRequest(MultiRead<T> request,
			SyncMultiProcedure<T> procedure) {

		assert (request != null);

		ListenerBase listener = getListenerBase(procedure);

		final ResultCallWrappedSyncQueryProcedure<T> wrapper = new ResultCallWrappedSyncQueryProcedure<T>(procedure);

		if (parent != null || listener != null) {

//			Object syncParent = request;

//			final ReadGraphImpl newGraph = newSync();

			processor.query(this, request, parent, wrapper, listener);

//			newGraph.waitAsync(syncParent);

		} else {

//			Object syncParent = request;

//			final ReadGraphImpl newGraph = newSync();

			try {
				request.perform(this, wrapper);
			} catch (Throwable t) {
				wrapper.exception(this, t);
			}

		}

		return wrapper.get();

	}

	@Override
	public <T> Collection<T> syncRequest(MultiRead<T> request,
			MultiProcedure<T> procedure) {
		return syncRequest(request, new NoneToSyncMultiProcedure<T>(procedure));
	}

	static class AsyncMultiReadProcedure<T> extends ArrayList<T> implements AsyncMultiProcedure<T> {

		private static Throwable DONE = new Throwable();
		
		private static final long serialVersionUID = -6494230465108115812L;
		
		Throwable exception = null;
		
		@Override
		public synchronized void execute(AsyncReadGraph graph, T t) {
			add(t);
		}

		@Override
		public void finished(AsyncReadGraph graph) {
			exception = DONE;
		}

		@Override
		public void exception(AsyncReadGraph graph, Throwable t) {
			exception = t;
		}
		
		public void checkAndThrow() throws DatabaseException {
			if(exception != DONE) {
				if (exception instanceof DatabaseException)
					throw (DatabaseException) exception;
				else
					throw new DatabaseException(
							"Unexpected exception in ReadGraph.syncRequest(AsyncMultiRead)",
							exception);
			}
		}
		
		public boolean done() {
			return exception != null;
		}
		
	}

	@Override
	final public <T> Collection<T> syncRequest(AsyncMultiRead<T> request)
			throws DatabaseException {

		assert (request != null);

		final AsyncMultiReadProcedure<T> procedure = new AsyncMultiReadProcedure<T>();
		
		syncRequest(request, procedure);
		
		procedure.checkAndThrow();
		return procedure;

	}

	@Override
	public <T> Collection<T> syncRequest(AsyncMultiRead<T> request,
			AsyncMultiListener<T> procedure) {
		return syncRequest(request, (AsyncMultiProcedure<T>) procedure);
	}

	@Override
	public <T> Collection<T> syncRequest(AsyncMultiRead<T> request,
			SyncMultiListener<T> procedure) {
		return syncRequest(request, new SyncToAsyncMultiListener<T>(procedure));
	}

	@Override
	public <T> Collection<T> syncRequest(AsyncMultiRead<T> request,
			MultiListener<T> procedure) {
		return syncRequest(request, new NoneToAsyncMultiListener<T>(procedure));
	}

	final private <T> void syncRequest(final AsyncMultiRead<T> request,
			final AsyncMultiReadProcedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);

		ListenerBase listener = getListenerBase(procedure);

		if (parent != null || listener != null) {

//			Object syncParent = request;

//			final ReadGraphImpl newGraph = newSync();

			processor.query(this, request, parent, procedure, listener);

//			newGraph.waitAsync(syncParent);
			waitAsyncProcedure(procedure);

		} else {

//			Object syncParent = callerThread == Integer.MIN_VALUE ? null
//					: request;
//
//			final ReadGraphImpl newGraph = newSyncAsync(syncParent);

			try {

//				inc();
//				ReadGraphImpl sync = newSync();
				request.perform(this, procedure);
//				sync.waitAsync(null);
				waitAsyncProcedure(procedure);
//				dec();

			} catch (Throwable t) {

				waitAsyncProcedure(procedure);
//				dec();

			}

		}

	}
	
	
	@Override
	final public <T> Collection<T> syncRequest(final AsyncMultiRead<T> request,
			final AsyncMultiProcedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);

		ListenerBase listener = getListenerBase(procedure);

		if (parent != null || listener != null) {

//			Object syncParent = request;

//			final ReadGraphImpl newGraph = newSync();

			processor.query(this, request, parent, procedure, listener);

//			newGraph.waitAsync(syncParent);

		} else {

//			Object syncParent = request;

//			final ReadGraphImpl newGraph = newSync();

			try {

				request.perform(this, new AsyncMultiProcedure<T>() {

					@Override
					public void execute(AsyncReadGraph graph, T result) {
						procedure.execute(graph, result);
					}

					@Override
					public void finished(AsyncReadGraph graph) {
						procedure.finished(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						procedure.exception(graph, t);
					}

					@Override
					public String toString() {
						return "syncRequest(AsyncMultiRead) -> " + procedure;
					}

				});

			} catch (Throwable t) {

			}

		}

		// TODO: implement return value
		return null;

	}

	@Override
	public <T> Collection<T> syncRequest(AsyncMultiRead<T> request,
			final SyncMultiProcedure<T> procedure) {
		return syncRequest(request, new SyncToAsyncMultiProcedure<T>(procedure));
	}

	@Override
	final public <T> Collection<T> syncRequest(final AsyncMultiRead<T> request,
			final MultiProcedure<T> procedure) {
		return syncRequest(request, new NoneToAsyncMultiProcedure<T>(procedure));
	}

	@Override
	public <T> T syncRequest(final ExternalRead<T> request)
			throws DatabaseException {

		assert (request != null);

		return syncRequest(request, new Procedure<T>() {

			@Override
			public void execute(T t) {
			}

			@Override
			public void exception(Throwable t) {
			}

			@Override
			public String toString() {
				return "syncRequest(AsyncRead) -> " + request;
			}

		});

	}

	@Override
	public <T> T syncRequest(ExternalRead<T> request, Listener<T> procedure) throws DatabaseException {
		return syncRequest(request, (Procedure<T>) procedure);
	}

	@Override
	final public <T> T syncRequest(final ExternalRead<T> request,
			final Procedure<T> procedure) throws DatabaseException {

        assert (request != null);

        ListenerBase listener = procedure != null ? getListenerBase(procedure) : null;
        return QueryCache.resultExternalReadEntry(this, request, parent, listener, procedure);

	}

	@Override
	public void syncRequest(final Write request) throws DatabaseException {

		assert (request != null);

		throw new DatabaseException(
				"Write operations are not supported during read transactions!");

	}

	@Override
	public <T> T syncRequest(final WriteResult<T> request) throws DatabaseException {

		assert (request != null);

		throw new DatabaseException(
				"Write operations are not supported during read transactions!");

	}

	@Override
	public void syncRequest(final DelayedWrite request)
			throws DatabaseException {

		assert (request != null);

		throw new DatabaseException(
				"Write operations are not supported during read transactions!");

	}

	@Override
	public <T> T syncRequest(final DelayedWriteResult<T> request) throws DatabaseException {

		assert (request != null);

		throw new DatabaseException(
				"Write operations are not supported during read transactions!");

	}
	
	@Override
	public void syncRequest(final WriteOnly request) throws DatabaseException {

		assert (request != null);

		throw new DatabaseException(
				"Write operations are not supported during read transactions!");

	}

	@Override
	public <T> T syncRequest(final WriteOnlyResult<T> request) throws DatabaseException {

		assert (request != null);

		throw new DatabaseException(
				"Write operations are not supported during read transactions!");

	}
	
	@Override
	public <T> void async(ReadInterface<T> r, AsyncProcedure<T> procedure) {
		r.request(this, procedure);
	}
	
	@Override
	public <T> void async(ReadInterface<T> r, Procedure<T> procedure) {
		r.request(this, procedure);
	}
	
	@Override
	public <T> void async(ReadInterface<T> r, SyncProcedure<T> procedure) {
		r.request(this, procedure);
	}
	
	@Override
	public <T> void async(ReadInterface<T> r, AsyncListener<T> procedure) {
		r.request(this, procedure);
	}
	
	@Override
	public <T> void async(ReadInterface<T> r, Listener<T> procedure) {
		r.request(this, procedure);
	}
	
	@Override
	public <T> void async(ReadInterface<T> r, SyncListener<T> procedure) {
		r.request(this, procedure);
	}

	@Override
	public <T> T sync(ReadInterface<T> r) throws DatabaseException {
		return r.request(this);
	}
	
	@Override
	public <T> T sync(WriteInterface<T> r) throws DatabaseException {
		return r.request(this);
	}
	
	@Override
	public <T> void async(WriteInterface<T> r, Procedure<T> procedure) {
		r.request(this, procedure);
	}

	@Override
	public <T> void async(WriteInterface<T> r) {
		r.request(this, new ProcedureAdapter<T>());
	}

	/*
	 * Implementation of the interface AsyncReadGraph
	 */

	@Override
	public void forURI(Resource resource, AsyncListener<String> listener) {
		asyncRequest(new org.simantics.db.common.uri.ResourceToURI(resource),
				listener);
	}

	@Override
	public void forURI(Resource resource, SyncListener<String> listener) {
		asyncRequest(new org.simantics.db.common.uri.ResourceToURI(resource),
				listener);
	}

	@Override
	public void forURI(Resource resource, Listener<String> listener) {
		asyncRequest(new org.simantics.db.common.uri.ResourceToURI(resource),
				listener);
	}

	@Override
	final public void forURI(final Resource resource,
			final AsyncProcedure<String> procedure) {

		assert (resource != null);
		assert (procedure != null);

		asyncRequest(new org.simantics.db.common.uri.ResourceToURI(resource),
				procedure);

	}

	@Override
	public void forURI(Resource resource, SyncProcedure<String> procedure) {
		forURI(resource, new SyncToAsyncProcedure<String>(procedure));
	}

	@Override
	public void forURI(Resource resource, Procedure<String> procedure) {
		forURI(resource, new NoneToAsyncProcedure<String>(procedure));
	}
	
	@Override
	public void forResource(String id, AsyncListener<Resource> listener) {
		asyncRequest(new org.simantics.db.common.primitiverequest.Resource(id),
				listener);
	}

	@Override
	public void forResource(String id, SyncListener<Resource> listener) {
		asyncRequest(new org.simantics.db.common.primitiverequest.Resource(id),
				listener);
	}

	@Override
	public void forResource(String id, Listener<Resource> listener) {
		asyncRequest(new org.simantics.db.common.primitiverequest.Resource(id),
				listener);
	}

	@Override
	final public void forResource(final String id,
			final AsyncProcedure<Resource> procedure) {

		assert (id != null);
		assert (procedure != null);

		processor.forResource(this, id, procedure);

	}

	@Override
	public void forResource(String id, SyncProcedure<Resource> procedure) {
		forResource(id, new SyncToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forResource(String id, Procedure<Resource> procedure) {
		forResource(id, new NoneToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forBuiltin(String id, AsyncListener<Resource> listener) {
		asyncRequest(new Builtin(id), listener);
	}

	@Override
	public void forBuiltin(String id, SyncListener<Resource> listener) {
		asyncRequest(new Builtin(id), listener);
	}

	@Override
	public void forBuiltin(String id, Listener<Resource> listener) {
		asyncRequest(new Builtin(id), listener);
	}

	@Override
	final public void forBuiltin(final String id,
			final AsyncProcedure<Resource> procedure) {

		assert (id != null);
		assert (procedure != null);

		processor.forBuiltin(this, id, procedure);

	}

	@Override
	public void forBuiltin(String id, SyncProcedure<Resource> procedure) {
		forBuiltin(id, new SyncToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forBuiltin(String id, Procedure<Resource> procedure) {
		forBuiltin(id, new NoneToAsyncProcedure<Resource>(procedure));
	}

	@Override
	final public void forEachStatement(Resource subject, Resource relation,
			AsyncMultiProcedure<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachStatement(this, subject, relation, procedure);

	}

	@Override
	public void forEachStatement(Resource subject, Resource relation,
			SyncMultiProcedure<Statement> procedure) {
		forEachStatement(subject, relation,
				new SyncToAsyncMultiProcedure<Statement>(procedure));
	}

	@Override
	final public void forEachStatement(Resource subject, Resource relation,
			MultiProcedure<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachStatement(this, subject, relation, procedure);

	}

	@Override
	final public void forStatementSet(Resource subject, Resource relation,
			AsyncSetListener<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forStatementSet(this, subject, relation, procedure);

	}

	@Override
	final public void forStatementSet(Resource subject, Resource relation,
			SyncSetListener<Statement> procedure) {
		forStatementSet(subject, relation,
				new SyncToAsyncSetProcedure<Statement>(procedure));
	}

	@Override
	public void forStatementSet(Resource subject, Resource relation,
			SetListener<Statement> listener) {
		forStatementSet(subject, relation,
				new NoneToAsyncSetProcedure<Statement>(listener));
	}

	@Override
	final public void forEachAssertedStatement(final Resource subject,
			final Resource relation,
			final AsyncMultiProcedure<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachAssertedStatement(this, subject, relation, procedure);

	}

	@Override
	public void forEachAssertedStatement(Resource subject, Resource relation,
			SyncMultiProcedure<Statement> procedure) {
		forEachAssertedStatement(subject, relation,
				new SyncToAsyncMultiProcedure<Statement>(procedure));
	}

	@Override
	public void forEachAssertedStatement(Resource subject, Resource relation,
			MultiProcedure<Statement> procedure) {
		forEachAssertedStatement(subject, relation,
				new NoneToAsyncMultiProcedure<Statement>(procedure));
	}

	@Override
	public void forAssertedStatementSet(Resource subject, Resource relation,
			AsyncSetListener<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forAssertedStatementSet(this, subject, relation, procedure);

	}

	@Override
	public void forAssertedStatementSet(Resource subject, Resource relation,
			SyncSetListener<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forAssertedStatementSet(subject, relation,
				new SyncToAsyncSetProcedure<Statement>(procedure));

	}

	@Override
	public void forAssertedStatementSet(Resource subject, Resource relation,
			SetListener<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forAssertedStatementSet(subject, relation,
				new NoneToAsyncSetProcedure<Statement>(procedure));

	}

	@Override
	final public void forEachPredicate(final Resource subject,
			final AsyncMultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forEachPredicate(this, subject, procedure);

	}

	@Override
	public void forEachPredicate(Resource subject,
			SyncMultiProcedure<Resource> procedure) {
		forEachPredicate(subject, new SyncToAsyncMultiProcedure<Resource>(
				procedure));
	}

	@Override
	final public void forEachPredicate(final Resource subject,
			final MultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forEachPredicate(this, subject, procedure);

	}

	@Override
	final public void forPredicateSet(final Resource subject,
			final AsyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forPredicateSet(this, subject, procedure);

	}

	@Override
	final public void forPredicateSet(final Resource subject,
			final SyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		forPredicateSet(subject, new SyncToAsyncSetProcedure<Resource>(
				procedure));

	}

	@Override
	final public void forPredicateSet(final Resource subject,
			final SetListener<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		forPredicateSet(subject, new NoneToAsyncSetProcedure<Resource>(
				procedure));

	}

	@Override
	final public void forEachPrincipalType(final Resource subject,
			final AsyncMultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forEachPrincipalType(this, subject, procedure);

	}

	@Override
	public void forEachPrincipalType(Resource subject,
			SyncMultiProcedure<Resource> procedure) {
		forEachPrincipalType(subject, new SyncToAsyncMultiProcedure<Resource>(
				procedure));
	}

	@Override
	final public void forEachPrincipalType(final Resource subject,
			final MultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forEachPrincipalType(this, subject, procedure);

	}

	@Override
	final public void forPrincipalTypeSet(final Resource subject,
			final AsyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forPrincipalTypeSet(this, subject, procedure);

	}

	@Override
	final public void forPrincipalTypeSet(final Resource subject,
			final SyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		forPrincipalTypeSet(subject, new SyncToAsyncSetProcedure<Resource>(
				procedure));

	}

	@Override
	final public void forPrincipalTypeSet(final Resource subject,
			final SetListener<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		forPrincipalTypeSet(subject, new NoneToAsyncSetProcedure<Resource>(
				procedure));

	}

	@Override
	public void forTypes(Resource subject, AsyncListener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	public void forTypes(Resource subject, SyncListener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	public void forTypes(Resource subject, Listener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	final public void forTypes(final Resource subject,
			final AsyncProcedure<Set<Resource>> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forTypes(this, subject, procedure);

	}

	@Override
	public void forTypes(Resource subject,
			SyncProcedure<Set<Resource>> procedure) {
		forTypes(subject, new SyncToAsyncProcedure<Set<Resource>>(procedure));
	}

	@Override
	public void forTypes(Resource subject, Procedure<Set<Resource>> procedure) {
		forTypes(subject, new NoneToAsyncProcedure<Set<Resource>>(procedure));
	}

	@Override
	public void forSupertypes(Resource subject,
			AsyncListener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	public void forSupertypes(Resource subject,
			SyncListener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	public void forSupertypes(Resource subject, Listener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	final public void forSupertypes(final Resource subject,
			final AsyncProcedure<Set<Resource>> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forSupertypes(this, subject, procedure);

	}

	@Override
	public void forSupertypes(Resource subject,
			SyncProcedure<Set<Resource>> procedure) {
		forSupertypes(subject, new SyncToAsyncProcedure<Set<Resource>>(
				procedure));
	}

	@Override
	public void forSupertypes(Resource subject,
			Procedure<Set<Resource>> procedure) {
		forSupertypes(subject, new NoneToAsyncProcedure<Set<Resource>>(
				procedure));
	}

	@Override
	public void forDirectSuperrelations(Resource subject,
			AsyncMultiProcedure<Resource> procedure) {
		
		assert (subject != null);
		assert (procedure != null);

		processor.forDirectSuperrelations(this, subject, procedure);
		
	}

	@Override
	public void forPossibleSuperrelation(Resource subject, AsyncProcedure<Resource> procedure) {
		
		assert (subject != null);
		assert (procedure != null);

		processor.forPossibleSuperrelation(this, subject, procedure);
		
	}

	@Override
	public void forSuperrelations(Resource subject,
			AsyncListener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	public void forSuperrelations(Resource subject,
			SyncListener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	public void forSuperrelations(Resource subject,
			Listener<Set<Resource>> listener) {
		asyncRequest(new Types(subject), listener);
	}

	@Override
	final public void forSuperrelations(final Resource subject,
			final AsyncProcedure<Set<Resource>> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forSuperrelations(this, subject, procedure);

	}

	@Override
	public void forSuperrelations(Resource subject,
			SyncProcedure<Set<Resource>> procedure) {
		forSuperrelations(subject, new SyncToAsyncProcedure<Set<Resource>>(
				procedure));
	}

	@Override
	public void forSuperrelations(Resource subject,
			Procedure<Set<Resource>> procedure) {
		forSuperrelations(subject, new NoneToAsyncProcedure<Set<Resource>>(
				procedure));
	}

	@Override
	final public void forEachObject(final Resource subject, final Resource relation, final AsyncMultiProcedure<Resource> procedure) {
		processor.forEachObject(this, subject, relation, procedure);
	}

	@Override
	public void forEachObject(Resource subject, Resource relation,
			SyncMultiProcedure<Resource> procedure) {
		forEachObject(subject, relation,
				new SyncToAsyncMultiProcedure<Resource>(procedure));
	}

	@Override
	public void forEachObject(Resource subject, Resource relation,
			MultiProcedure<Resource> procedure) {

		processor.forEachObject(this, subject, relation, procedure);

	}

	@Override
	final public void forEachDirectPredicate(final Resource subject, final AsyncProcedure<Set<Resource>> procedure) {
		processor.forEachDirectPredicate(this, subject, procedure);
	}

	@Override
	final public void forEachDirectPredicate(final Resource subject, final SyncProcedure<Set<Resource>> procedure) {
		forEachDirectPredicate(subject, new SyncToAsyncProcedure<Set<Resource>>(procedure));
	}

	@Override
	public void forEachDirectPredicate(Resource subject, Procedure<Set<Resource>> procedure) {
		forEachDirectPredicate(subject, new NoneToAsyncProcedure<Set<Resource>>(procedure));
	}

	@Override
	final public void forObjectSet(final Resource subject,
			final Resource relation, final AsyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forObjectSet(this, subject, relation, procedure);

	}

	@Override
	final public void forObjectSet(final Resource subject,
			final Resource relation, final SyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forObjectSet(subject, relation, new SyncToAsyncSetProcedure<Resource>(
				procedure));

	}

	@Override
	final public void forObjectSet(final Resource subject,
			final Resource relation, final SetListener<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forObjectSet(subject, relation, new NoneToAsyncSetProcedure<Resource>(
				procedure));

	}

	@Override
	final public void forEachAssertedObject(final Resource subject,
			final Resource relation,
			final AsyncMultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachAssertedObject(this, subject, relation, procedure);

	}

	@Override
	public void forEachAssertedObject(Resource subject, Resource relation,
			SyncMultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forEachAssertedObject(subject, relation,
				new SyncToAsyncMultiProcedure<Resource>(procedure));

	}

	@Override
	public void forEachAssertedObject(Resource subject, Resource relation,
			MultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forEachAssertedObject(subject, relation,
				new NoneToAsyncMultiProcedure<Resource>(procedure));

	}

	@Override
	public void forAssertedObjectSet(Resource subject, Resource relation,
			AsyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forAssertedObjectSet(this, subject, relation, procedure);

	}

	@Override
	public void forAssertedObjectSet(Resource subject, Resource relation,
			SyncSetListener<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forAssertedObjectSet(subject, relation,
				new SyncToAsyncSetProcedure<Resource>(procedure));

	}

	@Override
	public void forAssertedObjectSet(Resource subject, Resource relation,
			SetListener<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		forAssertedObjectSet(subject, relation,
				new NoneToAsyncSetProcedure<Resource>(procedure));

	}

	@Override
	public void forInverse(Resource relation, AsyncListener<Resource> listener) {
		asyncRequest(new Inverse(relation), listener);
	}

	@Override
	public void forInverse(Resource relation, SyncListener<Resource> listener) {
		asyncRequest(new Inverse(relation), listener);
	}

	@Override
	public void forInverse(Resource relation, Listener<Resource> listener) {
		asyncRequest(new Inverse(relation), listener);
	}

	@Override
	final public void forInverse(final Resource relation,
			final AsyncProcedure<Resource> procedure) {

		assert (relation != null);
		assert (procedure != null);

		processor.forInverse(this, relation, new AsyncProcedure<Resource>() {

			@Override
			public void execute(AsyncReadGraph graph, Resource result) {
				if (result != null)
					procedure.execute(graph, result);
				else {
					procedure.exception(graph, new NoInverseException(relation
							.toString()));
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				procedure.exception(graph, throwable);
			}

			@Override
			public String toString() {
				return "forInverse -> " + procedure;
			}

		});

	}

	@Override
	public void forInverse(Resource relation, SyncProcedure<Resource> procedure) {
		forInverse(relation, new SyncToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forInverse(Resource relation, Procedure<Resource> procedure) {
		forInverse(relation, new NoneToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forSingleObject(Resource subject, Resource relation,
			AsyncListener<Resource> listener) {
		asyncRequest(new SingleObject(subject, relation), listener);
	}

	@Override
	public void forSingleObject(Resource subject, Resource relation,
			SyncListener<Resource> listener) {
		asyncRequest(new SingleObject(subject, relation), listener);
	}

	@Override
	public void forSingleObject(Resource subject, Resource relation,
			Listener<Resource> listener) {
		asyncRequest(new SingleObject(subject, relation), listener);
	}

	@Override
	final public void forSingleObject(final Resource subject,
			final Resource relation, final AsyncProcedure<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachObject(this, subject, relation,
				new SingleOrErrorProcedure<Resource>(procedure));

	}

	@Override
	public void forSingleObject(Resource subject, Resource relation,
			SyncProcedure<Resource> procedure) {
		forSingleObject(subject, relation, new SyncToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forSingleObject(Resource subject, Resource relation,
			Procedure<Resource> procedure) {
		forSingleObject(subject, relation, new NoneToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forSingleStatement(Resource subject, Resource relation,
			AsyncListener<Statement> listener) {
		asyncRequest(new SingleStatement(subject, relation), listener);
	}

	@Override
	public void forSingleStatement(Resource subject, Resource relation,
			SyncListener<Statement> listener) {
		asyncRequest(new SingleStatement(subject, relation), listener);
	}

	@Override
	public void forSingleStatement(Resource subject, Resource relation,
			Listener<Statement> listener) {
		asyncRequest(new SingleStatement(subject, relation), listener);
	}

	@Override
	final public void forSingleStatement(final Resource subject,
			final Resource relation, final AsyncProcedure<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachStatement(this, subject, relation,
				new SingleOrErrorProcedure<Statement>(procedure));

	}

	@Override
	public void forSingleStatement(Resource subject, Resource relation,
			SyncProcedure<Statement> procedure) {
		forSingleStatement(subject, relation,
				new SyncToAsyncProcedure<Statement>(procedure));
	}

	@Override
	public void forSingleStatement(Resource subject, Resource relation,
			Procedure<Statement> procedure) {
		forSingleStatement(subject, relation,
				new NoneToAsyncProcedure<Statement>(procedure));
	}

	@Override
	public void forSingleType(Resource subject,
			AsyncListener<Resource> listener) {
		asyncRequest(new SingleTypeAny(subject), listener);
	}

	@Override
	public void forSingleType(Resource subject,
			SyncListener<Resource> listener) {
		asyncRequest(new SingleTypeAny(subject), listener);
	}

	@Override
	public void forSingleType(Resource subject,
			Listener<Resource> listener) {
		asyncRequest(new SingleTypeAny(subject), listener);
	}

	@Override
	final public void forSingleType(final Resource subject, final AsyncProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		final DeepSingleOrErrorProcedure<Resource> checkedProcedure = new DeepSingleOrErrorProcedure<Resource>(procedure);

		processor.forEachPrincipalType(this, subject, new AsyncMultiProcedureAdapter<Resource>() {

			@Override
			public void execute(AsyncReadGraph graph, final Resource principalType) {
				checkedProcedure.offer(graph, principalType);
			}

			@Override
			public void finished(AsyncReadGraph graph) {
				checkedProcedure.dec(graph);
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				checkedProcedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forSingleType -> " + procedure;
			}

		});

	}

	@Override
	public void forSingleType(Resource subject, SyncProcedure<Resource> procedure) {
		forSingleType(subject, new SyncToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forSingleType(Resource subject, Procedure<Resource> procedure) {
		forSingleType(subject, new NoneToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forSingleType(Resource subject, Resource relation,
			AsyncListener<Resource> listener) {
		asyncRequest(new SingleType(subject, relation), listener);
	}

	@Override
	public void forSingleType(Resource subject, Resource relation,
			SyncListener<Resource> listener) {
		asyncRequest(new SingleType(subject, relation), listener);
	}

	@Override
	public void forSingleType(Resource subject, Resource relation,
			Listener<Resource> listener) {
		asyncRequest(new SingleType(subject, relation), listener);
	}

	@Override
	final public void forSingleType(final Resource subject,
			final Resource baseType, final AsyncProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		final DeepSingleOrErrorProcedure<Resource> checkedProcedure = new DeepSingleOrErrorProcedure<Resource>(procedure);

		processor.forEachPrincipalType(this, subject,
				new AsyncMultiProcedureAdapter<Resource>() {

					@Override
					public void execute(AsyncReadGraph graph,
							final Resource principalType) {

						checkedProcedure.inc();

						if(baseType == null) {

							checkedProcedure.offer(graph, principalType);
							checkedProcedure.dec(graph);

						} else if(principalType.equals(baseType)) {

							checkedProcedure.offer(graph, principalType);
							checkedProcedure.dec(graph);

						} else {

							processor.forSupertypes((ReadGraphImpl)graph, principalType,
									new AsyncProcedure<Set<Resource>>() {

										@Override
										public void execute(
												AsyncReadGraph graph,
												Set<Resource> result) {

											if (result.contains(baseType))
												checkedProcedure.offer(graph,
														principalType);
											checkedProcedure.dec(graph);

										}

										@Override
										public void exception(
												AsyncReadGraph graph,
												Throwable t) {
											checkedProcedure
													.exception(graph, t);
										}

									});

						}

					}

					@Override
					public void finished(AsyncReadGraph graph) {
						checkedProcedure.dec(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						checkedProcedure.exception(graph, t);
					}

					@Override
					public String toString() {
						return "forSingleType -> " + procedure;
					}

				});

	}

	@Override
	public void forSingleType(Resource subject, Resource relation,
			SyncProcedure<Resource> procedure) {
		forSingleType(subject, relation, new SyncToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forSingleType(Resource subject, Resource relation,
			Procedure<Resource> procedure) {
		forSingleType(subject, relation, new NoneToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public <T> void forValue(Resource subject, Binding binding,
			AsyncListener<T> listener) {
		asyncRequest(new Value<T>(subject, binding), listener);
	}

	@Override
	public <T> void forValue(Resource subject, Binding binding,
			SyncListener<T> listener) {
		asyncRequest(new Value<T>(subject, binding), listener);
	}

	@Override
	public <T> void forValue(Resource subject, Binding binding,
			Listener<T> listener) {
		asyncRequest(new Value<T>(subject, binding), listener);
	}

	@Override
	public <T> void forValue(final Resource resource, final Binding binding,
			final AsyncProcedure<T> procedure) {

		assert (resource != null);
		assert (binding != null);
		assert (procedure != null);
		
		processor.forValue(this, resource, new AsyncProcedure<byte[]>() {

			@Override
			public void execute(AsyncReadGraph graph, byte[] result) {
				
				try {

					if (result == null) {
						procedure.exception(graph,
								new DoesNotContainValueException(
										"No value for resource " + resource));
						return;
					}

					Serializer serializer = binding.serializer();
//					Serializer serializer = Bindings.getSerializer( binding );
					Object obj = serializer.deserialize(result);
//					if (!binding.isInstance(obj))
//						procedure.exception(graph, new ClassCastException(
//								"Cannot get value " + obj + " with binding "
//										+ binding));
//					else
						procedure.execute(graph, (T) obj);

				} catch (Throwable t) {
				    procedure.exception(graph, new ServiceException("Could not forValue for subject " + debugString(resource) + " and binding " + String.valueOf(binding) + " with bytes " + safeArrayToString(result), t));
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
		    		Logger.defaultLogError(t2);
				}
			}

			@Override
			public String toString() {
				return "forValue -> " + procedure;
			}

		});

	}
	
    private static String safeArrayToString(byte[] a) {
        if (a == null)
            return "null";
        int iMax = a.length - 1;
        if (iMax == -1)
            return "[]";

        StringBuilder b = new StringBuilder();
        b.append('[');
        for (int i = 0; i < 100; i++) { // limit to first 100 items 
            b.append(a[i]);
            if (i == iMax)
                return b.append(']').toString();
            b.append(", ");
        }
        return b.append(", ... (" + a.length + ")]").toString();
    }

	@Override
	public <T> void forValue(Resource subject, Binding binding,
			SyncProcedure<T> procedure) {
		forValue(subject, binding, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forValue(Resource subject, Binding binding,
			Procedure<T> procedure) {
		forValue(subject, binding, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forValue(Resource subject, AsyncListener<T> listener) {
		asyncRequest(new ValueImplied<T>(subject), listener);
	}

	@Override
	public <T> void forValue(Resource subject, SyncListener<T> listener) {
		asyncRequest(new ValueImplied<T>(subject), listener);
	}

	@Override
	public <T> void forValue(Resource subject, Listener<T> listener) {
		asyncRequest(new ValueImplied<T>(subject), listener);
	}

	@Override
	final public <T> void forValue(final Resource subject, final AsyncProcedure<T> procedure) {

		assert (subject != null);
		assert (procedure != null);
		
		forRelatedValue(subject, processor.getL0(this).HasDataType, DATA_TYPE_BINDING_INTERNAL, new AsyncProcedure<Datatype>() {

			@Override
			public void execute(AsyncReadGraph graph, Datatype type) {
				// TODO: consider trying Bindings.getBeanBinding(type);
				Binding binding = Bindings.getBinding(type);
				graph.forValue(subject, binding, procedure);
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				procedure.exception(graph, new DoesNotContainValueException("Invalid data type", throwable));
			}
			
		});

	}

	@Override
	public <T> void forValue(Resource subject, SyncProcedure<T> procedure) {
		forValue(subject, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forValue(Resource subject, Procedure<T> procedure) {
		forValue(subject, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			AsyncListener<T> listener) {
		asyncRequest(new RelatedValueImplied<T>(subject, relation), listener);
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			SyncListener<T> listener) {
		asyncRequest(new RelatedValueImplied<T>(subject, relation), listener);
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Listener<T> listener) {
		asyncRequest(new RelatedValueImplied<T>(subject, relation), listener);
	}

	@Override
	final public <T> void forRelatedValue(final Resource subject,
			final Resource relation, final AsyncProcedure<T> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		final DeepSingleOrErrorProcedure<T> checkedProcedure = new DeepSingleOrErrorProcedure<T>(procedure);

		processor.forEachObject(this, subject, relation,
				new AsyncMultiProcedureAdapter<Resource>() {

					@Override
					public void execute(AsyncReadGraph graph,
							final Resource object) {

						checkedProcedure.inc();

						graph.forValue(object, new AsyncProcedure<Object>() {

							@Override
							public void execute(AsyncReadGraph graph,
									Object result) {
								checkedProcedure.offer(graph, (T) result);
								checkedProcedure.dec(graph);
							}

							@Override
							public void exception(AsyncReadGraph graph,
									Throwable t) {
								checkedProcedure.exception(graph, t);
							}

							@Override
							public String toString() {
								return "forRelatedValue -> " + procedure;
							}

						});

					}

					@Override
					public void finished(AsyncReadGraph graph) {
						checkedProcedure.dec(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						checkedProcedure.exception(graph, t);
					}

				});

	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			SyncProcedure<T> procedure) {
		forRelatedValue(subject, relation, new SyncToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Procedure<T> procedure) {
		forRelatedValue(subject, relation, new NoneToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Binding binding, AsyncListener<T> listener) {
		asyncRequest(new RelatedValue<T>(subject, relation, binding), listener);
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Binding binding, SyncListener<T> listener) {
		asyncRequest(new RelatedValue<T>(subject, relation, binding), listener);
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Binding binding, Listener<T> listener) {
		asyncRequest(new RelatedValue<T>(subject, relation, binding), listener);
	}

	@Override
	final public <T> void forRelatedValue(final Resource subject,
			final Resource relation, final Binding binding,
			final AsyncProcedure<T> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (binding != null);
		assert (procedure != null);

		final DeepSingleOrErrorProcedure<T> checkedProcedure = new DeepSingleOrErrorProcedure<T>(procedure);
		
		processor.forEachObject(this, subject, relation,
				new AsyncMultiProcedureAdapter<Resource>() {

					@Override
					public void execute(AsyncReadGraph graph,
							final Resource object) {

						checkedProcedure.inc();

						graph.forValue(object, binding, new AsyncProcedure<Object>() {

									@Override
									public void execute(AsyncReadGraph graph,
											Object result) {
										
										checkedProcedure.offer(graph,
												(T) result);
										checkedProcedure.dec(graph);
									}

									@Override
									public void exception(AsyncReadGraph graph,
											Throwable t) {
										checkedProcedure.exception(graph, t);
									}

									@Override
									public String toString() {
										return "forRelatedValue -> "
												+ procedure;
									}

								});

					}

					@Override
					public void finished(AsyncReadGraph graph) {
						checkedProcedure.dec(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						checkedProcedure.exception(graph, t);
					}

				});

	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Binding binding, SyncProcedure<T> procedure) {
		forRelatedValue(subject, relation, binding,
				new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forRelatedValue(Resource subject, Resource relation,
			Binding binding, Procedure<T> procedure) {
		forRelatedValue(subject, relation, binding,
				new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forAdapted(Resource resource, Class<T> clazz,
			AsyncListener<T> listener) {
		asyncRequest(new Adapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forAdapted(Resource resource, Class<T> clazz,
			SyncListener<T> listener) {
		asyncRequest(new Adapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forAdapted(Resource resource, Class<T> clazz,
			Listener<T> listener) {
		asyncRequest(new Adapter<T>(resource, clazz), listener);
	}

	@Override
	final public <T> void forAdapted(final Resource resource,
			final Class<T> clazz, final AsyncProcedure<T> procedure) {

		assert (resource != null);
		assert (clazz != null);
		assert (procedure != null);

		final AdaptionService service = getSession().peekService(AdaptionService.class);
		if (service == null)
			procedure.exception(this, new ServiceException("No AdaptionService available")); 
		else
			service.adapt(this, resource, resource, Resource.class, clazz, false, procedure); 

	}

	@Override
	public <T> void forAdapted(Resource resource, Class<T> clazz,
			SyncProcedure<T> procedure) {
		forAdapted(resource, clazz, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forAdapted(Resource resource, Class<T> clazz,
			Procedure<T> procedure) {
		forAdapted(resource, clazz, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forUniqueAdapted(Resource resource, Class<T> clazz,
			AsyncListener<T> listener) {
		asyncRequest(new UniqueAdapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forUniqueAdapted(Resource resource, Class<T> clazz,
			SyncListener<T> listener) {
		asyncRequest(new UniqueAdapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forUniqueAdapted(Resource resource, Class<T> clazz,
			Listener<T> listener) {
		asyncRequest(new UniqueAdapter<T>(resource, clazz), listener);
	}

	@Override
	final public <T> void forUniqueAdapted(final Resource resource,
			final Class<T> clazz, final AsyncProcedure<T> procedure) {

		assert (resource != null);
		assert (clazz != null);
		assert (procedure != null);

		final AdaptionService service = getSession().peekService(AdaptionService.class);
		if (service == null)
			procedure.exception(this, new ServiceException("No AdaptionService available")); 
		else
			service.adaptNew(this, resource, clazz, false, procedure);

	}

	@Override
	public <T> void forUniqueAdapted(Resource resource, Class<T> clazz,
			SyncProcedure<T> procedure) {
		forUniqueAdapted(resource, clazz,
				new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forUniqueAdapted(Resource resource, Class<T> clazz,
			Procedure<T> procedure) {
		forUniqueAdapted(resource, clazz,
				new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public void forPossibleInverse(Resource subject,
			AsyncListener<Resource> listener) {
		asyncRequest(new PossibleInverse(subject), listener);
	}

	@Override
	public void forPossibleInverse(Resource subject,
			SyncListener<Resource> listener) {
		asyncRequest(new PossibleInverse(subject), listener);
	}

	@Override
	public void forPossibleInverse(Resource subject, Listener<Resource> listener) {
		asyncRequest(new PossibleInverse(subject), listener);
	}

	@Override
	final public void forPossibleInverse(final Resource relation,
			final AsyncProcedure<Resource> procedure) {

		assert (relation != null);
		assert (procedure != null);

		processor.forInverse(this, relation, new ExceptionToNullProcedure<Resource>(procedure));

	}

	@Override
	public void forPossibleInverse(Resource subject,
			SyncProcedure<Resource> procedure) {
		forPossibleInverse(subject, new SyncToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forPossibleInverse(Resource subject,
			Procedure<Resource> procedure) {
		forPossibleInverse(subject, new NoneToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forPossibleObject(Resource subject, Resource relation,
			AsyncListener<Resource> listener) {
		asyncRequest(new PossibleObject(subject, relation), listener);
	}

	@Override
	public void forPossibleObject(Resource subject, Resource relation,
			SyncListener<Resource> listener) {
		asyncRequest(new PossibleObject(subject, relation), listener);
	}

	@Override
	public void forPossibleObject(Resource subject, Resource relation,
			Listener<Resource> listener) {
		asyncRequest(new PossibleObject(subject, relation), listener);
	}

	@Override
	final public void forPossibleObject(final Resource subject,
			final Resource relation, final AsyncProcedure<Resource> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachObject(this, subject, relation,
				new SingleOrNullProcedure<Resource>(procedure));

	}

	@Override
	public void forPossibleObject(Resource subject, Resource relation,
			SyncProcedure<Resource> procedure) {
		forPossibleObject(subject, relation,
				new SyncToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forPossibleObject(Resource subject, Resource relation,
			Procedure<Resource> procedure) {
		forPossibleObject(subject, relation,
				new NoneToAsyncProcedure<Resource>(procedure));
	}

	@Override
	public void forPossibleStatement(Resource subject, Resource relation,
			AsyncListener<Statement> listener) {
		asyncRequest(new PossibleStatement(subject, relation), listener);
	}

	@Override
	public void forPossibleStatement(Resource subject, Resource relation,
			SyncListener<Statement> listener) {
		asyncRequest(new PossibleStatement(subject, relation), listener);
	}

	@Override
	public void forPossibleStatement(Resource subject, Resource relation,
			Listener<Statement> listener) {
		asyncRequest(new PossibleStatement(subject, relation), listener);
	}

	@Override
	final public void forPossibleStatement(final Resource subject,
			final Resource relation, final AsyncProcedure<Statement> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forEachStatement(this, subject, relation,
				new SingleFunctionalOrNullProcedure<Statement>(
						"forPossibleStatement", procedure));

	}

	@Override
	public void forPossibleStatement(Resource subject, Resource relation,
			SyncProcedure<Statement> procedure) {
		forPossibleStatement(subject, relation,
				new SyncToAsyncProcedure<Statement>(procedure));
	}

	@Override
	public void forPossibleStatement(Resource subject, Resource relation,
			Procedure<Statement> procedure) {
		forPossibleStatement(subject, relation,
				new NoneToAsyncProcedure<Statement>(procedure));
	}

	@Override
	public void forPossibleType(Resource subject, Resource relation,
			AsyncListener<Resource> listener) {
		asyncRequest(new PossibleType(subject, relation), listener);
	}

	@Override
	public void forPossibleType(Resource subject, Resource relation,
			SyncListener<Resource> listener) {
		asyncRequest(new PossibleType(subject, relation), listener);
	}

	@Override
	public void forPossibleType(Resource subject, Resource relation,
			Listener<Resource> listener) {
		asyncRequest(new PossibleType(subject, relation), listener);
	}

	@Override
	final public void forPossibleType(final Resource subject,
			final Resource baseType, final AsyncProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		final NullSingleOrNullProcedure<Resource> checkedProcedure = new NullSingleOrNullProcedure<Resource>(procedure);

		processor.forEachPrincipalType(this, subject, new AsyncMultiProcedureAdapter<Resource>() {

					@Override
					public void execute(AsyncReadGraph graph,
							final Resource principalType) {

						if (baseType == null) {

							checkedProcedure.offer(graph, principalType);

						} else if (principalType.equals(baseType)) {

							checkedProcedure.offer(graph, principalType);

						} else {

							checkedProcedure.inc();

							processor.forSupertypes((ReadGraphImpl)graph, principalType,
									new AsyncProcedure<Set<Resource>>() {

										@Override
										public void execute(
												AsyncReadGraph graph,
												Set<Resource> result) {

											if (result.contains(baseType)) {
												checkedProcedure.offer(graph,
														principalType);
											}

											checkedProcedure.dec(graph);

										}

										@Override
										public void exception(
												AsyncReadGraph graph,
												Throwable t) {
											checkedProcedure.exception(graph, t);
											checkedProcedure.dec(graph);
										}

										@Override
										public String toString() {
											return "forPossibleType -> "
													+ procedure;
										}

									});

						}

					}

					@Override
					public void finished(AsyncReadGraph graph) {
						checkedProcedure.dec(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						checkedProcedure.exception(graph, t);
						checkedProcedure.dec(graph);
					}

				});

	}

	@Override
	public void forPossibleType(Resource subject, Resource relation,
			SyncProcedure<Resource> procedure) {
		forPossibleType(subject, relation, new SyncToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public void forPossibleType(Resource subject, Resource relation,
			Procedure<Resource> procedure) {
		forPossibleType(subject, relation, new NoneToAsyncProcedure<Resource>(
				procedure));
	}

	@Override
	public <T> void forPossibleValue(Resource subject, AsyncListener<T> listener) {
		asyncRequest(new PossibleValueImplied<T>(subject), listener);
	}

	@Override
	public <T> void forPossibleValue(Resource subject, SyncListener<T> listener) {
		asyncRequest(new PossibleValueImplied<T>(subject), listener);
	}

	@Override
	public <T> void forPossibleValue(Resource subject, Listener<T> listener) {
		asyncRequest(new PossibleValueImplied<T>(subject), listener);
	}

	@Override
	final public <T> void forPossibleValue(final Resource subject,
			final AsyncProcedure<T> procedure) {

		assert (subject != null);
		assert (procedure != null);
		
		forPossibleRelatedValue(subject, processor.getL0(this).HasDataType, DATA_TYPE_BINDING_INTERNAL, new AsyncProcedure<Datatype>() {

			@Override
			public void execute(AsyncReadGraph graph, final Datatype type) {
				if (type == null) {
					procedure.execute(graph, null);
				} else {
					try {
						// TODO: consider trying Bindings.getBeanBinding(type);
						Binding binding = Bindings.getBinding(type);
						graph.forPossibleValue(subject, binding, procedure);
					} catch (RuntimeBindingConstructionException e) {
						procedure.exception(graph, e);
					}
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				procedure.exception(graph, t);
			}

			@Override
			public String toString() {
				return "forPossibleValue -> " + procedure;
			}

		});

	}

	@Override
	public <T> void forPossibleValue(Resource subject,
			SyncProcedure<T> procedure) {
		forPossibleValue(subject, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forPossibleValue(Resource subject, Procedure<T> procedure) {
		forPossibleValue(subject, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forPossibleValue(Resource subject, Binding binding,
			AsyncListener<T> listener) {
		asyncRequest(new PossibleValue<T>(subject, binding), listener);
	}

	@Override
	public <T> void forPossibleValue(Resource subject, Binding binding,
			SyncListener<T> listener) {
		asyncRequest(new PossibleValue<T>(subject, binding), listener);
	}

	@Override
	public <T> void forPossibleValue(Resource subject, Binding binding,
			Listener<T> listener) {
		asyncRequest(new PossibleValue<T>(subject, binding), listener);
	}

	@Override
	final public <T> void forPossibleValue(final Resource resource,
			final Binding binding, final AsyncProcedure<T> procedure) {

		assert (resource != null);
		assert (binding != null);
		assert (procedure != null);

		processor.forValue(this, resource, new AsyncProcedure<byte[]>() {

			@Override
			public void execute(AsyncReadGraph graph, byte[] result) {

				try {

					if (result == null) {
						procedure.execute(graph, null);
						return;
					}

					Serializer serializer = Bindings.getSerializer( binding );
					Object obj = serializer.deserialize(result);
					if (!binding.isInstance(obj))
						procedure.exception(graph, new ClassCastException(
								"Cannot get value " + obj + " with binding "
										+ binding));
					else
						procedure.execute(graph, (T) obj);

				} catch (Throwable t) {
					procedure.exception(graph, new ServiceException("Could not forValue for subject " + debugString(resource) + " and binding " + String.valueOf(binding) + " with bytes " + safeArrayToString(result), t));
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
		    		Logger.defaultLogError(t2);
				}
			}

			@Override
			public String toString() {
				return "forPossibleValue -> " + procedure;
			}

		});

	}

	@Override
	public <T> void forPossibleValue(Resource subject, Binding binding,
			SyncProcedure<T> procedure) {
		forPossibleValue(subject, binding, new SyncToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleValue(Resource subject, Binding binding,
			Procedure<T> procedure) {
		forPossibleValue(subject, binding, new NoneToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, AsyncListener<T> listener) {
		asyncRequest(new PossibleRelatedValueImplied<T>(subject, relation),
				listener);
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, SyncListener<T> listener) {
		asyncRequest(new PossibleRelatedValueImplied<T>(subject, relation),
				listener);
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Listener<T> listener) {
		asyncRequest(new PossibleRelatedValueImplied<T>(subject, relation),
				listener);
	}

	@Override
	final public <T> void forPossibleRelatedValue(final Resource subject,
			final Resource relation, final AsyncProcedure<T> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		final DeepSingleOrNullProcedure<T> checkedProcedure = new DeepSingleOrNullProcedure<T>(procedure);
		
		processor.forEachObject(this, subject, relation,
				new AsyncMultiProcedureAdapter<Resource>() {

					@Override
					public void execute(AsyncReadGraph graph,
							final Resource object) {

						checkedProcedure.inc();

						graph.forValue(object, new AsyncProcedure<Object>() {

							@Override
							public void execute(AsyncReadGraph graph,
									Object result) {
								checkedProcedure.offer(graph, (T) result);
								checkedProcedure.dec(graph);
							}

							@Override
							public void exception(AsyncReadGraph graph,
									Throwable t) {
								checkedProcedure.exception(graph, t);
								checkedProcedure.dec(graph);
							}

						});

					}

					@Override
					public void finished(AsyncReadGraph graph) {

						checkedProcedure.dec(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						checkedProcedure.exception(graph, t);
						checkedProcedure.dec(graph);
					}

					@Override
					public String toString() {
						return "forPossibleRelatedValue -> " + procedure;
					}
				});

	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, SyncProcedure<T> procedure) {
		forPossibleRelatedValue(subject, relation, new SyncToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Procedure<T> procedure) {
		forPossibleRelatedValue(subject, relation, new NoneToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Binding binding, AsyncListener<T> listener) {
		asyncRequest(new PossibleRelatedValue<T>(subject, relation, binding),
				listener);
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Binding binding, SyncListener<T> listener) {
		asyncRequest(new PossibleRelatedValue<T>(subject, relation, binding),
				listener);
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Binding binding, Listener<T> listener) {
		asyncRequest(new PossibleRelatedValue<T>(subject, relation, binding),
				listener);
	}

	final public <T> void forPossibleRelatedValue(final Resource subject, final Resource relation, final Binding binding,
			final AsyncProcedure<T> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forPossibleObject(this, subject, relation, new AsyncProcedure<Resource>() {
			
			@Override
			public void execute(AsyncReadGraph graph, Resource object) {
				
				if(object == null) {
					procedure.execute(graph, null);
					return;
				}

				processor.forPossibleValue((ReadGraphImpl)graph, object, new AsyncProcedure<byte[]>() {

					@Override
					public void execute(AsyncReadGraph graph, byte[] bytes) {

						if(bytes != null) {

							try {
							
								Serializer serializer = binding.serializer();
								Object obj = serializer.deserialize(bytes);
								if (!binding.isInstance(obj)) {
									procedure.exception(graph, new ClassCastException("Cannot get value " + obj + " with binding " + binding));
								} else {
									procedure.execute(graph, (T) obj);
								}

							} catch (Throwable t) {
								
								procedure.exception(graph, t);
								
							}
							
						} else {
							
							procedure.execute(graph, null);
							
						}
						
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						procedure.exception(graph, t);
					}

				});
				
			}
			
			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				throwable.printStackTrace();
				procedure.exception(graph, throwable);
			}
			
		});

	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Binding binding, SyncProcedure<T> procedure) {
		forPossibleRelatedValue(subject, relation, binding,
				new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void forPossibleRelatedValue(Resource subject,
			Resource relation, Binding binding, Procedure<T> procedure) {
		forPossibleRelatedValue(subject, relation, binding,
				new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public void forIsInstanceOf(Resource subject, Resource relation,
			AsyncListener<Boolean> listener) {
		asyncRequest(new IsInstanceOf(subject, relation), listener);
	}

	@Override
	public void forIsInstanceOf(Resource subject, Resource relation,
			SyncListener<Boolean> listener) {
		asyncRequest(new IsInstanceOf(subject, relation), listener);
	}

	@Override
	public void forIsInstanceOf(Resource subject, Resource relation,
			Listener<Boolean> listener) {
		asyncRequest(new IsInstanceOf(subject, relation), listener);
	}

	@Override
	final public void forIsInstanceOf(final Resource resource,
			final Resource type, final AsyncProcedure<Boolean> procedure) {

		assert (resource != null);
		assert (type != null);
		assert (procedure != null);

		forTypes(resource, new AsyncProcedure<Set<Resource>>() {

			@Override
			public void execute(AsyncReadGraph graph, Set<Resource> result) {
				
				try {
					if (result.contains(type))
						procedure.execute(graph, true);
					else
						procedure.execute(graph, false);
				} catch (Throwable t) {
		    		Logger.defaultLogError(t);
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
		    		Logger.defaultLogError(t2);
				}
			}

			@Override
			public String toString() {
				return "forIsInstanceOf -> " + procedure;
			}

		});

	}

	@Override
	public void forIsInstanceOf(Resource subject, Resource relation,
			SyncProcedure<Boolean> procedure) {
		forIsInstanceOf(subject, relation, new SyncToAsyncProcedure<Boolean>(
				procedure));
	}

	@Override
	public void forIsInstanceOf(Resource subject, Resource relation,
			Procedure<Boolean> procedure) {
		forIsInstanceOf(subject, relation, new NoneToAsyncProcedure<Boolean>(
				procedure));
	}

	@Override
	public void forIsInheritedFrom(Resource subject, Resource relation,
			AsyncListener<Boolean> listener) {
		asyncRequest(new IsInheritedFrom(subject, relation), listener);
	}

	@Override
	public void forIsInheritedFrom(Resource subject, Resource relation,
			SyncListener<Boolean> listener) {
		asyncRequest(new IsInheritedFrom(subject, relation), listener);
	}

	@Override
	public void forIsInheritedFrom(Resource subject, Resource relation,
			Listener<Boolean> listener) {
		asyncRequest(new IsInheritedFrom(subject, relation), listener);
	}

	@Override
	final public void forIsInheritedFrom(final Resource resource,
			final Resource type, final AsyncProcedure<Boolean> procedure) {

		assert (resource != null);
		assert (type != null);
		assert (procedure != null);

		if (resource.equals(type)) {
			try {
				procedure.execute(this, true);
			} catch (Throwable t) {
	    		Logger.defaultLogError(t);
			}
			return;
		}

		forSupertypes(resource, new AsyncProcedure<Set<Resource>>() {

			@Override
			public void execute(AsyncReadGraph graph, Set<Resource> result) {
				try {
					if (result.contains(type))
						procedure.execute(graph, true);
					else
						procedure.execute(graph, false);
				} catch (Throwable t) {
		    		Logger.defaultLogError(t);
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
		    		Logger.defaultLogError(t2);
				}
			}

			@Override
			public String toString() {
				return "forIsInheritedFrom -> " + procedure;
			}

		});

	}

	@Override
	public void forIsInheritedFrom(Resource subject, Resource relation,
			SyncProcedure<Boolean> procedure) {
		forIsInheritedFrom(subject, relation,
				new SyncToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forIsInheritedFrom(Resource subject, Resource relation,
			Procedure<Boolean> procedure) {
		forIsInheritedFrom(subject, relation,
				new NoneToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forIsSubrelationOf(Resource subject, Resource relation,
			AsyncListener<Boolean> listener) {
		asyncRequest(new IsSubrelationOf(subject, relation), listener);
	}

	@Override
	public void forIsSubrelationOf(Resource subject, Resource relation,
			SyncListener<Boolean> listener) {
		asyncRequest(new IsSubrelationOf(subject, relation), listener);
	}

	@Override
	public void forIsSubrelationOf(Resource subject, Resource relation,
			Listener<Boolean> listener) {
		asyncRequest(new IsSubrelationOf(subject, relation), listener);
	}

	@Override
	final public void forIsSubrelationOf(final Resource resource,
			final Resource relation, final AsyncProcedure<Boolean> procedure) {

		assert (resource != null);
		assert (relation != null);
		assert (procedure != null);

		if (resource.equals(relation)) {
			procedure.execute(this, true);
			return;
		}

		forSuperrelations(resource, new AsyncProcedure<Set<Resource>>() {

			@Override
			public void execute(AsyncReadGraph graph, Set<Resource> result) {
				try {
					if (result.contains(relation))
						procedure.execute(graph, true);
					else
						procedure.execute(graph, false);
				} catch (Throwable t) {
		    		Logger.defaultLogError(t);
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
		    		Logger.defaultLogError(t2);
				}
			}

			@Override
			public String toString() {
				return "forIsSubrelationOf -> " + procedure;
			}

		});

	}

	@Override
	public void forIsSubrelationOf(Resource subject, Resource relation,
			SyncProcedure<Boolean> procedure) {
		forIsSubrelationOf(subject, relation,
				new SyncToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forIsSubrelationOf(Resource subject, Resource relation,
			Procedure<Boolean> procedure) {
		forIsSubrelationOf(subject, relation,
				new NoneToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forHasStatement(Resource subject,
			AsyncListener<Boolean> listener) {
		asyncRequest(new HasStatementSubject(subject), listener);
	}

	@Override
	public void forHasStatement(Resource subject, SyncListener<Boolean> listener) {
		asyncRequest(new HasStatementSubject(subject), listener);
	}

	@Override
	public void forHasStatement(Resource subject, Listener<Boolean> listener) {
		asyncRequest(new HasStatementSubject(subject), listener);
	}

	@Override
	final public void forHasStatement(final Resource subject,
			final AsyncProcedure<Boolean> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forHasStatement(this, subject, procedure);

	}

	@Override
	public void forHasStatement(Resource subject,
			SyncProcedure<Boolean> procedure) {
		forHasStatement(subject, new SyncToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forHasStatement(Resource subject, Procedure<Boolean> procedure) {
		forHasStatement(subject, new NoneToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			AsyncListener<Boolean> listener) {
		asyncRequest(new HasStatement(subject, relation), listener);
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			SyncListener<Boolean> listener) {
		asyncRequest(new HasStatement(subject, relation), listener);
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Listener<Boolean> listener) {
		asyncRequest(new HasStatement(subject, relation), listener);
	}

	@Override
	final public void forHasStatement(final Resource subject,
			final Resource relation, final AsyncProcedure<Boolean> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (procedure != null);

		processor.forHasStatement(this, subject, relation, procedure);

	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			SyncProcedure<Boolean> procedure) {
		forHasStatement(subject, relation, new SyncToAsyncProcedure<Boolean>(
				procedure));
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Procedure<Boolean> procedure) {
		forHasStatement(subject, relation, new NoneToAsyncProcedure<Boolean>(
				procedure));
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Resource object, AsyncListener<Boolean> listener) {
		asyncRequest(new HasStatementSubjectObject(subject, relation, object),
				listener);
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Resource object, SyncListener<Boolean> listener) {
		asyncRequest(new HasStatementSubjectObject(subject, relation, object),
				listener);
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Resource object, Listener<Boolean> listener) {
		asyncRequest(new HasStatementSubjectObject(subject, relation, object),
				listener);
	}

	@Override
	final public void forHasStatement(final Resource subject,
			final Resource relation, final Resource object,
			final AsyncProcedure<Boolean> procedure) {

		assert (subject != null);
		assert (relation != null);
		assert (object != null);
		assert (procedure != null);

		processor.forHasStatement(this, subject, relation, object, procedure);

	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Resource object, SyncProcedure<Boolean> procedure) {
		forHasStatement(subject, relation, object,
				new SyncToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forHasStatement(Resource subject, Resource relation,
			Resource object, Procedure<Boolean> procedure) {
		forHasStatement(subject, relation, object,
				new NoneToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forHasValue(Resource subject, AsyncListener<Boolean> listener) {
		asyncRequest(new HasValue(subject), listener);
	}

	@Override
	public void forHasValue(Resource subject, SyncListener<Boolean> listener) {
		asyncRequest(new HasValue(subject), listener);
	}

	@Override
	public void forHasValue(Resource subject, Listener<Boolean> listener) {
		asyncRequest(new HasValue(subject), listener);
	}

	@Override
	final public void forHasValue(final Resource subject,
			final AsyncProcedure<Boolean> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forValue(this, subject, new AsyncProcedure<byte[]>() {

			@Override
			public void execute(AsyncReadGraph graph, byte[] result) {
				try {
					if (result == null)
						procedure.execute(graph, false);
					else
						procedure.execute(graph, true);
				} catch (Throwable t) {
		    		Logger.defaultLogError(t);
				}
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				try {
					procedure.exception(graph, t);
				} catch (Throwable t2) {
		    		Logger.defaultLogError(t2);
				}
			}

			@Override
			public String toString() {
				return "forHasValue -> " + procedure;
			}

		});

	}

	@Override
	public void forHasValue(Resource subject, SyncProcedure<Boolean> procedure) {
		forHasValue(subject, new SyncToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forHasValue(Resource subject, Procedure<Boolean> procedure) {
		forHasValue(subject, new NoneToAsyncProcedure<Boolean>(procedure));
	}

	@Override
	public void forOrderedSet(Resource subject,
			AsyncMultiListener<Resource> listener) {
		asyncRequest(new OrderedSet(subject), listener);
	}

	@Override
	public void forOrderedSet(Resource subject,
			SyncMultiListener<Resource> listener) {
		asyncRequest(new OrderedSet(subject), listener);
	}

	@Override
	public void forOrderedSet(Resource subject, MultiListener<Resource> listener) {
		asyncRequest(new OrderedSet(subject), listener);
	}

	@Override
	final public void forOrderedSet(final Resource subject,
			final AsyncMultiProcedure<Resource> procedure) {

		assert (subject != null);
		assert (procedure != null);

		processor.forOrderedSet(this, subject,
				new AsyncMultiProcedure<Resource>() {

					@Override
					public void finished(AsyncReadGraph graph) {
						try {
							procedure.finished(graph);
						} catch (Throwable t) {
				    		Logger.defaultLogError(t);
						}
					}

					@Override
					public void execute(AsyncReadGraph graph, Resource result) {
						try {
							procedure.execute(graph, result);
						} catch (Throwable t) {
				    		Logger.defaultLogError(t);
						}
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						try {
							procedure.exception(graph, t);
						} catch (Throwable t2) {
				    		Logger.defaultLogError(t2);
						}
					}

					@Override
					public String toString() {
						return "forOrderedSet -> " + procedure;
					}

				});

	}

	@Override
	public void forOrderedSet(Resource subject,
			SyncMultiProcedure<Resource> procedure) {
		forOrderedSet(subject, new SyncToAsyncMultiProcedure<Resource>(
				procedure));
	}

	@Override
	public void forOrderedSet(Resource subject,
			MultiProcedure<Resource> procedure) {
		forOrderedSet(subject, new NoneToAsyncMultiProcedure<Resource>(
				procedure));
	}

	@Override
	public <T> void forPossibleAdapted(Resource resource, Class<T> clazz,
			AsyncListener<T> listener) {
		asyncRequest(new PossibleAdapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forPossibleAdapted(Resource resource, Class<T> clazz,
			SyncListener<T> listener) {
		asyncRequest(new PossibleAdapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forPossibleAdapted(Resource resource, Class<T> clazz,
			Listener<T> listener) {
		asyncRequest(new PossibleAdapter<T>(resource, clazz), listener);
	}

	@Override
	final public <T> void forPossibleAdapted(final Resource resource,
			final Class<T> clazz, final AsyncProcedure<T> procedure) {

		assert (resource != null);
		assert (clazz != null);
		assert (procedure != null);

		final AdaptionService service = getSession().peekService(AdaptionService.class);
		if (service == null)
			procedure.exception(this, new ServiceException("No AdaptionService available")); 
		else
			service.adapt(this, resource, resource, Resource.class, clazz, true, procedure);
	}

	@Override
	public <T> void forPossibleAdapted(Resource resource, Class<T> clazz,
			SyncProcedure<T> procedure) {
		forPossibleAdapted(resource, clazz, new SyncToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleAdapted(Resource resource, Class<T> clazz,
			Procedure<T> procedure) {
		forPossibleAdapted(resource, clazz, new NoneToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz,
			AsyncListener<T> listener) {
		asyncRequest(new PossibleUniqueAdapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz,
			SyncListener<T> listener) {
		asyncRequest(new PossibleUniqueAdapter<T>(resource, clazz), listener);
	}

	@Override
	public <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz,
			Listener<T> listener) {
		asyncRequest(new PossibleUniqueAdapter<T>(resource, clazz), listener);
	}

	@Override
	final public <T> void forPossibleUniqueAdapted(final Resource resource,
			final Class<T> clazz, final AsyncProcedure<T> procedure) {

		assert (resource != null);
		assert (clazz != null);
		assert (procedure != null);

		final AdaptionService service = getSession().peekService(AdaptionService.class);
		if (service == null)
			procedure.exception(this, new ServiceException("No AdaptionService available")); 
		else
			service.adaptNew(this, resource, clazz, true, procedure);

	}

	@Override
	public <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz,
			SyncProcedure<T> procedure) {
		forPossibleUniqueAdapted(resource, clazz, new SyncToAsyncProcedure<T>(
				procedure));
	}

	@Override
	public <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz,
			Procedure<T> procedure) {
		forPossibleUniqueAdapted(resource, clazz, new NoneToAsyncProcedure<T>(
				procedure));
	}

	/*
	 * Implementation of the interface AsyncRequestProcessor
	 */

	@Override
	final public Session getSession() {
		return processor.getSession();
	}

	@Override
	public <T> void asyncRequest(final Read<T> request) {

		asyncRequest(request, new AsyncProcedure<T>() {

			@Override
			public void execute(AsyncReadGraph graph, T result) {
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				LOGGER.error("asyncRequest(Read={}) failed", request, t);
			}

			@Override
			public String toString() {
				return "asyncRequest(Read) -> " + request;
			}

		});

	}

	@Override
	public <T> void asyncRequest(Read<T> request, AsyncListener<T> procedure) {
		asyncRequest(request, (AsyncProcedure<T>) procedure);
	}

	@Override
	public <T> void asyncRequest(Read<T> request,
			final SyncListener<T> procedure) {
		asyncRequest(request, new SyncToAsyncListener<T>(procedure));
	}

	@Override
	public <T> void asyncRequest(Read<T> request, final Listener<T> procedure) {
		asyncRequest(request, new NoneToAsyncListener<T>(procedure));
	}

	@Override
	public <T> void asyncRequest(final Read<T> request, final AsyncProcedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);
		
		AsyncBarrierImpl barrier = asyncBarrier;
		if(barrier != null)
		    barrier.inc();
		
		processor.scheduleNow(new SessionTask(this) {

			@Override
			public void run0(int thread) {
				try {
					final ListenerBase listener = getListenerBase(procedure);
					QueryCache.runnerReadEntry(ReadGraphImpl.this, request, parent, listener, procedure, false);
				} catch (DatabaseException e) {
					// This exception has already been transferred to procedure at this point - log here only for debugging
					if (LOGGER.isDebugEnabled())
						LOGGER.debug("asyncRequest(Read={}, AsyncProcedure={}) failed", request, procedure, e);
				} finally {
					if(barrier != null)
						barrier.dec();
				}
			}
			
			@Override
			public String toString() {
				return "asyncRequest " + request.toString();
			}

		});

	}

    public static ReadGraphImpl createAsync(QueryProcessor support) {
        return new ReadGraphImpl(null, null, support);
    }

	@Override
	public <T> void asyncRequest(Read<T> request, SyncProcedure<T> procedure) {
		asyncRequest(request, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void asyncRequest(Read<T> request, final Procedure<T> procedure) {
		asyncRequest(request, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(final AsyncRead<T> request) {

		assert (request != null);

		asyncRequest(request, new AsyncProcedure<T>() {

			@Override
			public void execute(AsyncReadGraph graph, T result) {
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				LOGGER.error("asyncRequest(AsyncRead={}) failed", request, t);
			}

			@Override
			public String toString() {
				return "asyncRequest(AsyncRead) -> " + request;
			}

		});

	}

	@Override
	public <T> void asyncRequest(AsyncRead<T> request,
			AsyncListener<T> procedure) {
		asyncRequest(request, (AsyncProcedure<T>) procedure);
	}

	@Override
	final public <T> void asyncRequest(AsyncRead<T> request,
			final SyncListener<T> procedure) {
		asyncRequest(request, new SyncToAsyncListener<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(AsyncRead<T> request,
			final Listener<T> procedure) {
		asyncRequest(request, new NoneToAsyncListener<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(final AsyncRead<T> request,
			final AsyncProcedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);

		AsyncBarrierImpl barrier = asyncBarrier;
		if(barrier != null)
			barrier.inc();

		processor.scheduleNow(new SessionTask(this) {

			@Override
			public void run0(int thread) {

				if(barrier != null)
					barrier.inc();

				try {
					final ListenerBase listener = getListenerBase(procedure);
					QueryCache.runnerAsyncReadEntry(ReadGraphImpl.this, request, parent, listener, new AsyncProcedure<T>() {

						@Override
						public void execute(AsyncReadGraph graph, T result) {
							procedure.execute(graph, result);
							if(barrier != null)
								barrier.dec();
						}

						@Override
						public void exception(AsyncReadGraph graph, Throwable throwable) {
							procedure.exception(graph, throwable);
							if(barrier != null)
								barrier.dec();
						}

					}, false);
				} catch (DatabaseException e) {
					LOGGER.error("Error while executing async request", e);
				} finally {
					if(barrier != null)
						barrier.dec();
				}
			}

			@Override
			public String toString() {
				return "asyncRequest " + request;
			}

		});

	}

	@Override
	public <T> void asyncRequest(AsyncRead<T> request,
			SyncProcedure<T> procedure) {
		asyncRequest(request, new SyncToAsyncProcedure<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(final AsyncRead<T> request,
			final Procedure<T> procedure) {
		asyncRequest(request, new NoneToAsyncProcedure<T>(procedure));
	}

	@Override
	public <T> void asyncRequest(final MultiRead<T> request) {

		assert (request != null);

		asyncRequest(request, new SyncMultiProcedureAdapter<T>() {
			@Override
			public void exception(ReadGraph graph, Throwable t) {
				LOGGER.error("asyncRequest(MultiRead={}) failed", request, t);
			}

			@Override
			public String toString() {
				return "asyncRequest(MultiRead) -> " + request;
			}
		});

	}

	@Override
	public <T> void asyncRequest(MultiRead<T> request,
			SyncMultiListener<T> procedure) {
		asyncRequest(request, (SyncMultiProcedure<T>)procedure);
	}

	@Override
	public <T> void asyncRequest(MultiRead<T> request,
			MultiListener<T> procedure) {
		asyncRequest(request, new NoneToSyncMultiListener<T>(procedure));
	}

	
	@Override
	public <T> void asyncRequest(final MultiRead<T> request,
			final SyncMultiProcedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		if (parent != null || listener != null) {

//		    final ReadGraphImpl newGraph = newSync();
		    processor.query(this, request, parent, procedure,listener);

		} else {

//		    final ReadGraphImpl newGraph = newSync();

		    try {

		        request.perform(this, procedure);

		    } catch (Throwable t) {

		        try {
					procedure.exception(this, t);
				} catch (DatabaseException e) {
					LOGGER.error("Unexpected exception while handling exception", e);
				}

		    }				

		}

	}

	@Override
	public <T> void asyncRequest(MultiRead<T> request,
			MultiProcedure<T> procedure) {
		asyncRequest(request, new NoneToSyncMultiProcedure<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(final AsyncMultiRead<T> request) {

		assert (request != null);

		asyncRequest(request, new AsyncMultiProcedureAdapter<T>() {
			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				LOGGER.error("asyncRequest(AsyncMultiRead={}) failed", request, t);
			}

			@Override
			public String toString() {
				return "asyncRequest(AsyncMultiRead) -> " + request;
			}
		});

	}

	@Override
	public <T> void asyncRequest(AsyncMultiRead<T> request,
			AsyncMultiListener<T> procedure) {
		asyncRequest(request, (AsyncMultiProcedure<T>) procedure);
	}

	@Override
	public <T> void asyncRequest(AsyncMultiRead<T> request,
			SyncMultiListener<T> procedure) {
		asyncRequest(request, new SyncToAsyncMultiListener<T>(procedure));
	}

	@Override
	public <T> void asyncRequest(AsyncMultiRead<T> request,
			MultiListener<T> procedure) {
		asyncRequest(request, new NoneToAsyncMultiListener<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(AsyncMultiRead<T> request,
			final AsyncMultiProcedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);

		ListenerBase listener = getListenerBase(procedure);

		if (parent != null || listener != null) {

			processor.query(this, request, parent, procedure, listener);

		} else {

			try {

				request.perform(this, new AsyncMultiProcedure<T>() {

					@Override
					public void execute(AsyncReadGraph graph, T result) {
						procedure.execute(graph, result);
					}

					@Override
					public void finished(AsyncReadGraph graph) {
						procedure.finished(graph);
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						procedure.exception(graph, t);
					}

					@Override
					public String toString() {
						return "asyncRequest(AsyncMultiRead) -> " + procedure;
					}

				});

			} catch (Throwable t) {

				procedure.exception(this, new DatabaseException(t));

			}
		}

	}

	@Override
	public <T> void asyncRequest(AsyncMultiRead<T> request,
			SyncMultiProcedure<T> procedure) {
		asyncRequest(request, new SyncToAsyncMultiProcedure<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(AsyncMultiRead<T> request,
			final MultiProcedure<T> procedure) {
		asyncRequest(request, new NoneToAsyncMultiProcedure<T>(procedure));
	}

	@Override
	final public <T> void asyncRequest(final ExternalRead<T> request) {

		assert (request != null);

		asyncRequest(request, new Procedure<T>() {

			@Override
			public void execute(T result) {
			}

			@Override
			public void exception(Throwable t) {
				LOGGER.error("asyncRequest(ExternalRead={}) failed", request, t);
			}

			@Override
			public String toString() {
				return "asyncRequest(PrimitiveRead) -> " + request;
			}

		});

	}

	@Override
	public <T> void asyncRequest(ExternalRead<T> request,
			final Listener<T> procedure) {
		asyncRequest(request, (Procedure<T>) procedure);
	}

	@Override
	final public <T> void asyncRequest(final ExternalRead<T> request,
			final Procedure<T> procedure) {

		assert (request != null);
		assert (procedure != null);

		final ListenerBase listener = getListenerBase(procedure);

		if (parent != null || listener != null) {

			try {
				QueryCacheBase.resultExternalReadEntry(this, request, parent, listener, procedure);
			} catch (DatabaseException e) {
				// This exception has already been transferred to procedure at this point - log here only for debugging
				if (LOGGER.isDebugEnabled())
					LOGGER.debug("asyncRequest(ExternalRead={}, Procedure={}) failed", request, procedure, e);
			}

		} else {

			request.register(this, new Listener<T>() {
				@Override
				public void execute(T result) {
					procedure.execute(result);
				}

				@Override
				public void exception(Throwable t) {
					procedure.exception(t);
				}

				@Override
				public String toString() {
					return "asyncRequest(PrimitiveRead) -> " + request;
				}

				@Override
				public boolean isDisposed() {
					return true;
				}

			});

		}

	}

	@Override
	public void asyncRequest(final Write request) {

		assert (request != null);
		
		getSession().asyncRequest(request);

//		processor.asyncWrite(request);

	}

	@Override
	public <T> void asyncRequest(final WriteResult<T> request, Procedure<T> procedure) {
		throw new Error("Not implemented.");
	}
	
	@Override
	public void asyncRequest(Write request, Consumer<DatabaseException> callback) {
		
		assert (request != null);
		
		getSession().asyncRequest(request, callback);
		
	}

	@Override
	public void asyncRequest(final DelayedWrite request) {

		assert (request != null);

		getSession().asyncRequest(request);

	}

	@Override
	public <T> void asyncRequest(final DelayedWriteResult<T> request, Procedure<T> procedure) {
		throw new Error("Not implemented.");
	}
	
	@Override
	public void asyncRequest(DelayedWrite r,
			Consumer<DatabaseException> callback) {
		throw new Error("Not implemented.");
	}

	@Override
	public void asyncRequest(final WriteOnly request) {

		assert (request != null);

		getSession().asyncRequest(request);

	}

	@Override
	public <T> void asyncRequest(final WriteOnlyResult<T> request, Procedure<T> procedure) {
		throw new Error("Not implemented.");
	}
	
	@Override
	public void asyncRequest(WriteOnly r, Consumer<DatabaseException> callback) {
		throw new Error("Not implemented.");
	}

	/*
	 * Implementation of the interface ServiceLocator
	 */

	@Override
	public <T> T getService(Class<T> api) {
	    if(WriteSupport.class == api) {
	        if(this instanceof WriteGraphImpl) {
	            WriteGraphImpl impl = (WriteGraphImpl)this;
	            return (T)impl.writeSupport;
	        }
	    }
		return getSession().getService(api);
	}

	@Override
	public <T> T peekService(Class<T> api) {
		return getSession().peekService(api);
	}

	@Override
	public boolean hasService(Class<?> api) {
		return getSession().hasService(api);
	}

	@Override
	public <T> void registerService(Class<T> api, T service) {
		getSession().registerService(api, service);
	}
	
	@Override
	public boolean isImmutable(Resource resource) throws DatabaseException {
		ResourceImpl impl = (ResourceImpl)resource;
		return processor.isImmutableForReading(impl.id);
	}

	@Override
	public boolean isImmutableForWriting(Resource resource) throws DatabaseException {
		return processor.isImmutableForWriting(resource);
	}

	/*
	 * Internal routines
	 */

	protected static String INTERNAL_ERROR_STRING = "Transaction aborted due to internal client error.";

	/*
	 * callerThread is the currently running thread state.syncThread is blocking for
	 * this execution state.syncParent is the blocking request
	 */

	ReadGraphImpl(ReadGraphImpl parentGraph, CacheEntry parent, QueryProcessor support) {
		this.parentGraph = parentGraph;
		this.parent = parent;
		this.processor = support;
		this.asyncBarrier = prepareBarrier(parentGraph, parent, null, false);
	}

	ReadGraphImpl(ReadGraphImpl parentGraph, CacheEntry parent, QueryProcessor support, AsyncBarrierImpl asyncBarrier) {
		this.parentGraph = parentGraph;
		this.parent = parent;
		this.processor = support;
		this.asyncBarrier = asyncBarrier;
	}

	ReadGraphImpl(ReadGraphImpl graph, CacheEntry parent) {
		this(graph, parent, graph.processor);
	}

	static AsyncBarrierImpl prepareBarrier(ReadGraphImpl parentGraph, CacheEntry parent, Runnable callback, boolean needsToBlock) {
		return new AsyncBarrierImpl(parentGraph != null ? parentGraph.asyncBarrier : null, parent, callback, needsToBlock);
	}

	ReadGraphImpl(ReadGraphImpl graph) {
		this(graph, graph.parent);
	}

	public ReadGraphImpl withParent(CacheEntry parent, Runnable callback, boolean needsToBlock) {
		return new ReadGraphImpl(this, parent, processor, prepareBarrier(this, parent, callback, needsToBlock));
	}

	public ReadGraphImpl forRecompute(CacheEntry parent) {
		return new ReadGraphImpl(null, parent, processor, prepareBarrier(null, parent, null, true));
	}
	
	public ReadGraphImpl forSyncExecute() {
		return withParent(null, null, true);
	}

    public static ReadGraphImpl create(QueryProcessor support) {
        ReadGraphImpl result = new ReadGraphImpl(null, null, support);
        return result;
    }

	public ReadGraphImpl newRestart(ReadGraphImpl impl) {

		WriteGraphImpl write = processor.getSession().getService(
				WriteGraphImpl.class);

		return write;

	}

	final private ListenerBase getListenerBase(final Object procedure) {
		if (procedure instanceof ListenerBase)
			return (ListenerBase) procedure;
		else
			return null;
	}

	public <T> void waitAsyncProcedure(AsyncMultiReadProcedure<T> procedure) {
		
		assert(procedure.done());
		
	}

	public <T> void waitAsyncProcedure(AsyncReadProcedure<T> procedure) {
		
		assert(procedure.done());
		
	}

	Class<?> singleClass(Set<Resource> types) {
		Class<?> result = null;
		for (Resource type : types) {
			Class<?> clazz = processor.getBuiltinValue(type);
			if (clazz != null) {
				if (result != null)
					return null;
				else
					result = clazz;
			}
		}
		return result;
	}

	private String debugString(Resource r) {
		String name = null;
		try {
			name = getPossibleRelatedValue(r, processor.getL0(this).HasName);
		} catch (ManyObjectsForFunctionalRelationException e) {
    		Logger.defaultLogError(e);
		} catch (ServiceException e) {
    		Logger.defaultLogError(e);
		}
		return "[" + name + " - " + r + "]";
	}

	@Override
	public String toString() {
		return "ReadGraphImpl[thread=" + Thread.currentThread() + "]";
	}

	@Override
	final public int thread() {
		return 0;
	}
	
    static class MultiTripleIntProcedure implements TripleIntProcedure {

    	final private AsyncMultiProcedure<Statement> procedure;
    	final private ReadGraphImpl impl;
    	final private QuerySupport support;
    	
    	public MultiTripleIntProcedure(AsyncMultiProcedure<Statement> procedure, ReadGraphImpl impl, QuerySupport support) {
    		this.procedure = procedure;
    		this.impl = impl;
    		this.support = support;
    	}
    	
    	@Override
    	public void execute(ReadGraphImpl graph, int s, int p, int o) {
    		try {
    			procedure.execute(graph, support.getStatement(s, p, o));
    		} catch (Throwable t2) {
	    		Logger.defaultLogError(t2);
    		}
    	}

    	@Override
    	public void finished(ReadGraphImpl graph) {
    		try {
    			procedure.finished(graph);
//    			impl.state.barrier.dec("ReadGraphSupportImpl.516");
    		} catch (Throwable t2) {
	    		Logger.defaultLogError(t2);
    		}
    	}

    	@Override
    	public void exception(ReadGraphImpl graph, Throwable t) {
    		try {
    			procedure.exception(graph, t);
    		} catch (Throwable t2) {
	    		Logger.defaultLogError(t2);
    		}
//    		impl.state.barrier.dec("ReadGraphSupportImpl.516");
    	}

    	@Override
    	public String toString() {
    		return "forEachObject with " + procedure;
    	}
    	
    }

//    private AsyncMultiProcedure<Resource> cacheKey = null;
//    private MultiIntProcedure cacheResult = null;
//    
//    final IntProcedure forMultiProcedure(final AsyncMultiProcedure<Resource> procedure) {
//    	
//    	if(procedure == cacheKey) return cacheResult; 
//    	
//    	cacheResult = new MultiIntProcedure(procedure, this, processor.support);
//    	cacheKey = procedure;
//    	
//    	return cacheResult;
//    	
//    }
//
//    private AsyncMultiProcedure<Statement> cacheKey2 = null;
//    private MultiTripleIntProcedure cacheResult2 = null;
//
//    final synchronized TripleIntProcedure forMultiProcedure(final AsyncMultiProcedure<Statement> procedure) {
//    	
//    	if(procedure == cacheKey2) return cacheResult2; 
//    	
//    	cacheResult2 = new MultiTripleIntProcedure(procedure, this, processor.support);
//    	cacheKey2 = procedure;
//    	
//    	return cacheResult2;
//    	
//    }
    
    @Override
    public Datatype getDataType(Resource subject) throws DatabaseException {
    	for(Resource dataTypeResource : getObjects(subject, processor.getL0(this).HasDataType))
    		return getValue(dataTypeResource, Bindings.getBindingUnchecked(Datatype.class));
    	throw new DoesNotContainValueException("The literal has no data type.");
    }
    
    protected <T extends Accessor> T getAccessor4File(Resource subject)
    throws DatabaseException {
        return null;
        /*        
        byte[]  bytes = processor.support.getValue(g, subject);
        if (null == bytes)
            return null;
        try {
            BinaryVariant va = (BinaryVariant) Accessors.getAccessor(bytes, Datatypes.VARIANT);
            Accessor ca = va.getContentAccessor();
            return (T)ca;
        } catch (AccessorConstructionException e) {
            throw new DatabaseException(e);
        }
        */
        /*
        if (null == bytes)
            return null;
        Binding datatype_binding = Bindings.getBindingUnchecked(DataType.class);
        Serializer datatype_serializer = datatype_binding.serializer();
        DataType datatype;
        try {
            BinaryMemory in = new BinaryMemory(ByteBuffer.wrap(bytes));
            datatype = (DataType)datatype_serializer.deserialize(in);
            Binding data_binding = Bindings.getBinding(datatype);
            Serializer data_serializer = data_binding.serializer(BinarySerializationFormat.INSTANCE);
            Object o = data_serializer.deserialize(in);
            try {
                return (T)Accessors.getAccessor(data_binding, o);
            } catch(AccessorConstructionException e) {
                return null;
            }
        } catch (Exception e) {
            throw new DatabaseException(e);
        }*/
    }
    @SuppressWarnings("unchecked")
    @Override
    public <T extends Accessor> T getAccessor(Resource subject) throws DatabaseException {
        RandomAccessBinary rab = getRandomAccessBinary(subject);
        try {
            return (T)Accessors.getAccessor(rab, getDataType(subject));
        } catch(AccessorConstructionException e) {
            throw new DatabaseException(e);
        }
    }
    @SuppressWarnings("unchecked")
    protected <T extends Accessor> T createAccessor(Resource resource, Datatype datatype, Object intialValue)
    throws DatabaseException {
        RandomAccessBinary rab = createRandomAccessBinary(resource, datatype, intialValue);
        try {
            return (T)Accessors.getAccessor(rab, datatype);
        } catch(AccessorConstructionException e) {
            throw new DatabaseException(e);
        }
    }
    @Override
    public RandomAccessBinary getRandomAccessBinary(Resource subject) throws DatabaseException {
        RandomAccessValueSupport ravs = getSession().getService(RandomAccessValueSupport.class);
        ResourceData rd = ravs.get(subject);
        if (null != rd)
            return rd;
        try {
            ExternalValueSupport evs = getService(ExternalValueSupport.class);
            long size = evs.getValueSize(this, subject); // Throws DatabaseException if no old external value.
            try {
                File platform = Platform.getLocation().toFile();
                File tempFiles = new File(platform, "tempFiles");
                File dbDir = new File(tempFiles, "db");
                dbDir.mkdirs();
                File file = new File(dbDir, "ResourceFile" + subject.getResourceId());
                rd = new ResourceData(new BinaryFile(file), true); // Old external value.
                final int N = 1<<20;
                long left = size;
                long offset = 0;
                while (left > 0) {
                    int length = N < left ? N : (int)left;
                    byte[] bytes = evs.readValue(this, subject, offset, length);
                    offset += bytes.length;
                    left -= bytes.length;
                    rd.binaryFile.write(bytes);
                }
                ravs.put(subject, rd);
                return rd;
            } catch (Exception e) {
                throw new DatabaseException("Resource " + subject + " have value but there was problem with accessing value data.", e);
            }
        } catch (Exception e) {
            if(Development.DEVELOPMENT) {
                if(Development.<Boolean>getProperty(DevelopmentKeys.WRITEGRAPH_EXCEPTION_STACKTRACES, Bindings.BOOLEAN)) {
                    e.printStackTrace();
                }
            }
        }
        Datatype datatype = getDataType(subject);
        Object value = getPossibleValue(subject, Bindings.getBinding(datatype));
        return createRandomAccessBinary(subject, datatype, value);
    }
    public RandomAccessBinary createRandomAccessBinary(Resource resource, Datatype datatype, Object initialValue)
    throws DatabaseException {
        RandomAccessValueSupport ravs = getSession().getService(RandomAccessValueSupport.class);
        try {
            File platform = Platform.getLocation().toFile();
            File tempFiles = new File(platform, "tempFiles");
            File dbDir = new File(tempFiles, "db");
            dbDir.mkdirs();
            File file = new File(dbDir, "ResourceFile" + resource.getResourceId());
            ResourceData rd = new ResourceData(new BinaryFile(file), false);
            Binding binding = Bindings.getBinding(datatype);
            if (null == initialValue) {
                initialValue = binding.createDefault();
            }
            Serializer serializer = binding.serializer();
            byte[] bytes = serializer.serialize(initialValue);
            // In case the file has been previously accessed and was larger we set the correct size now
            rd.binaryFile.setLength(bytes.length);
            rd.binaryFile.write(bytes);
            ravs.put(resource, rd);
            return rd;
        } catch (Exception e) {
            if (e instanceof DatabaseException)
                throw (DatabaseException)e;
            else
                throw new DatabaseException(e);
        }
    }

//    static class ExternalValueRequest<T> extends ResourceRead<T> {
//
//		public ExternalValueRequest(Resource resource) {
//			super(resource);
//		}
//
//		@SuppressWarnings("unchecked")
//		@Override
//		public T perform(ReadGraph graph) throws DatabaseException {
//    		try {
//    			
//    			String uri = graph.getURI(resource);
//    			if(Layer0.URIs.Functions_functionApplication.equals(uri)) return (T)functionApplication;
//    			
//    			return (T)ReflectionUtils.getValue(uri).getValue();
//    			
//    		} catch(ValueNotFoundException e) {
//    			throw new DatabaseException("Couldn't convert to external value (r=" + resource + ")", e);
//    		} catch(ClassCastException e) {
//    			throw new DatabaseException("Couldn't convert to external value (r=" + resource + ")", e);
//    		}
//		}
//    	
//    }
    
    @SuppressWarnings("unchecked")
    @Override
    public <T> T getValue2(Resource r, Object context) throws DatabaseException {
    	Layer0 L0 = processor.getL0(this);
    	Set<Resource> types = getTypes(r);
    	
    	if(types.contains(L0.Literal)) {
    		if(isImmutable(r)) {
    			return syncRequest(new ValueImplied<T>(r));
    		} else {
    			return getValue(r);
    		}
    	}
    	else if(types.contains(L0.ExternalValue)) {
    		return (T)syncRequest(new AdaptValue(r), TransientCacheListener.<Object>instance());
    	}
    	else {

    	    Function3<ReadGraph,Resource,Object,T> function = requestValueFunction(r);
    	    if(function == null) throw new DoesNotContainValueException("Couldn't convert to a value function " + r);
    	    try {
    	        return function.apply(this, r, context);
    	    } catch(RuntimeException e) {
    	        DatabaseException dte = findPossibleRootException(e);
    	        if(dte != null) throw dte;
    	        else throw new DatabaseException(e);
    	    }
    		
    	}
    }

    @Override
    public Variant getVariantValue2(Resource r, Object context) throws DatabaseException {
        Layer0 L0 = processor.getL0(this);
        Set<Resource> types = getTypes(r);
        
        if(types.contains(L0.Literal)) {
            if(isImmutable(r)) {
                return syncRequest(new VariantValueImplied(r));
            } else {
                return getVariantValue(r);
            }
        }
        else if(types.contains(L0.ExternalValue)) {
            Object value = syncRequest(new AdaptValue(r), TransientCacheListener.<Object>instance());
            try {
                return new Variant(Bindings.OBJECT.getContentBinding(value), value);
            } catch ( org.simantics.databoard.binding.error.BindingException e ) {
                throw new BindingException( "No binding found for class " + value.getClass().getName(), e );
            }
        }
        else {

            Function3<ReadGraph,Resource,Object,Object> function = requestValueFunction(r);
            if(function == null) throw new DoesNotContainValueException("Couldn't convert to a value function " + r);
            try {
                Object value = function.apply(this, r, context);
                try {
                    return new Variant(Bindings.OBJECT.getContentBinding(value), value);
                } catch ( org.simantics.databoard.binding.error.BindingException e ) {
                    throw new BindingException( "No binding found for class " + value.getClass().getName(), e );
                }
            } catch(RuntimeException e) {
                DatabaseException dte = findPossibleRootException(e);
                if(dte != null) throw dte;
                else throw new DatabaseException(e);
            }
        }
    }
    
    @Override
    public <T> T getPossibleValue2(Resource subject, Object context) throws DatabaseException {
    	try {
    		return getValue2(subject, context);
    	} catch (DatabaseException e) {
    		return null;
    	}
    }

    static class PossibleConverterFunction<T> extends ResourceRead<Function3<ReadGraph,Resource,Object,T>> {
    	
    	public PossibleConverterFunction(Resource resource) {
			super(resource);
		}

        @Override
        public Function3<ReadGraph,Resource,Object,T> perform(ReadGraph graph) throws DatabaseException {
        	return compute(graph, resource);
        }
        
        @SuppressWarnings("unchecked")
		public static <T> Function3<ReadGraph,Resource,Object,T> compute(ReadGraph graph, Resource resource) throws DatabaseException {
            Layer0 L0 = Layer0.getInstance(graph);
            for(Resource converter : graph.getObjects(resource, L0.ConvertsToValueWith)) {
                try {
                	if(L0.Functions_functionApplication.equals(converter)) {
                		return (Function3<ReadGraph,Resource,Object,T>)graph.syncRequest(new AdaptValue(resource));
                	} else {
                		return graph.getValue2(converter, resource);
                	}
                } catch(RuntimeException e) {
                    DatabaseException dte = findPossibleRootException(e);
                    if(dte != null) throw dte;
                    else throw new DatabaseException(e);
                }
            }
            return null;
        }
        
    }
    
    <T> Function3<ReadGraph,Resource,Object,T> requestValueFunction(Resource r) throws DatabaseException {
    	if(isImmutable(r))
    		return syncRequest(new PossibleConverterFunction<T>(r), TransientCacheAsyncListener.<Function3<ReadGraph,Resource,Object,T>>instance());
    	else
    		return syncRequest(new PossibleConverterFunction<T>(r));
    }
    
    /**
     * Get a value associated with a graph {@link Resource}, using a possible context object and
     * a desired value binding. The following methods are tried in order to retreive the value:
     * <ol>
     *   <li>If the given resource is a {@code L0.Literal}, the value of the literal is returned, using the binding specified by {@code binding}.</li>
     *   <li>If the resource is a {@code L0.ExternalValue}, the value is acquired using
     *       {@link ReflectionUtils#getValue(String)}.</li>
     *   <li>If the resource is associated with a suitable value converter with relation {@code L0.ConvertsToValueWith}
     *       (see {@link #requestValueFunction(Resource)}), the value function is called with the graph, the resource
     *       and the context object.</li>
     * </ul>
     * 
     * @param r  A graph resource with which the value is associated
     * @param context  A context object that is used for acquiring the value (only applied in case 3)
     * @param binding  A binding for the value type (only applied in case 1)
     * @return  The value of the graph node.
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     */
    @SuppressWarnings("unchecked")
	@Override
    public <T> T getValue2(Resource r, Object context, Binding binding) throws DatabaseException {
        if (binding instanceof ObjectVariantBinding)
            return getValue2(r, context);

        Layer0 L0 = processor.getL0(this);
        Set<Resource> types = getTypes(r);
        if(types.contains(L0.Literal)) {
            if(isImmutable(r)) {
                return syncRequest(new Value<T>(r, binding));
            } else {
                return getValue(r, binding);
            }
        } else if(types.contains(L0.ExternalValue)) {
            ComputationalValue cv = syncRequest(new PossibleAdapter<ComputationalValue>(r, ComputationalValue.class), TransientCacheAsyncListener.instance());
            if(cv != null) {
                return cv.getValue(this, r);
            } else {
                // This should not even be possible since L0 defines an adapter for all values
                try {
                    return (T)ReflectionUtils.getValue(getURI(r)).getValue();
                } catch(ValueNotFoundException e) {
                    throw new DatabaseException(e);
                } catch(ClassCastException e) {
                    throw new DatabaseException(e);
                }
            }
        }
        else {
            Function3<ReadGraph,Resource,Object,T> function = requestValueFunction(r);
            if(function == null) throw new DoesNotContainValueException("Couldn't convert to a value function.");
            try {
                Object value = function.apply(this, r, context);
                if(value == null)
                    return null;
                if(binding.isInstance(value))
                    return (T)value;
                Binding srcBinding = Bindings.OBJECT.getContentBinding(value);
                return (T)Bindings.adapt(value, srcBinding, binding);
            } catch(RuntimeException e) {
                DatabaseException dte = findPossibleRootException(e);
                if(dte != null) throw dte;
                else throw new DatabaseException(e);
            } catch (AdaptException e) {
                throw new DatabaseException(e);
            } catch (org.simantics.databoard.binding.error.BindingException e) {
                throw new DatabaseException(e);
            }
        }
    }
    
    @Override
    public <T> T getPossibleValue2(Resource subject, Object context, Binding binding) throws DatabaseException {
    	try {
    		return getValue2(subject, context, binding);
    	} catch (DatabaseException e) {
    		return null;
    	}
    }
    
	private static DatabaseException findPossibleRootException(Throwable t) {
		if(t == null) return null;
		if(t instanceof DatabaseException) return (DatabaseException)t;
		if(t instanceof RuntimeException || t instanceof InvocationTargetException) {
			return findPossibleRootException(t.getCause());
		}
		return null;
	}
    
    @Override
    public <T> T getRelatedValue2(Resource subject, Resource relation) throws DatabaseException {
    	return getRelatedValue2(subject, relation, subject);
    }
    
    @Override
    public Variant getRelatedVariantValue2(Resource subject, Resource relation) throws DatabaseException {
        return getRelatedVariantValue2(subject, relation, subject);
    }
    
    @Override
    public <T> T getPossibleRelatedValue2(Resource subject, Resource relation) throws DatabaseException {
    	try {
    		Resource object = getPossibleObject(subject, relation);
    		if(object == null) return null;
    		else return getValue2(object, subject);
    	} catch (DatabaseException e) {
    		return null;
    	}
    }

    @Override
    public <T> T getRelatedValue2(Resource subject, Resource relation, Object context) throws DatabaseException {
		if(Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.L0_VALIDATION, Bindings.BOOLEAN)) {
				String error = L0Validations.checkValueType(this, subject, relation);
				if(error != null) {
					Logger.defaultLogError(new ValidationException(error));
					throw new ValidationException(error);
				}
			}
		}
        return getValue2(getSingleObject(subject, relation), context);
    }
    
    @Override
    public Variant getRelatedVariantValue2(Resource subject, Resource relation, Object context) throws DatabaseException {
        if(Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.L0_VALIDATION, Bindings.BOOLEAN)) {
				String error = L0Validations.checkValueType(this, subject, relation);
				if(error != null) {
					Logger.defaultLogError(new ValidationException(error));
					throw new ValidationException(error);
				}
			}
        }
        return getVariantValue2(getSingleObject(subject, relation), context);
    }
    
    @Override
    public <T> T getPossibleRelatedValue2(Resource subject, Resource relation, Object context) throws DatabaseException {
    	try {
    		Resource object = getPossibleObject(subject, relation);
    		if(object == null) return null;
    		else return getValue2(object, context);
    	} catch (DatabaseException e) {
    		e.logPossibleRequest();
    		return null;
    	}
    }

    @Override
    public <T> T getRelatedValue2(Resource subject, Resource relation, Binding binding) throws DatabaseException {
    	return getRelatedValue2(subject, relation, subject, binding);
    }
    
    @Override
    public <T> T getPossibleRelatedValue2(Resource subject, Resource relation, Binding binding) throws DatabaseException {
    	try {
    		Resource object = getPossibleObject(subject, relation);
    		if(object == null) return null;
    		return getValue2(object, subject, binding);
    	} catch (DatabaseException e) {
    		return null;
    	}
    }

    @Override
    public <T> T getRelatedValue2(Resource subject, Resource relation, Object context, Binding binding) throws DatabaseException {
        return getValue2(getSingleObject(subject, relation), context, binding);
    }
    
    @Override
    public <T> T getPossibleRelatedValue2(Resource subject, Resource relation, Object context, Binding binding) throws DatabaseException {
    	try {
    		Resource object = getPossibleObject(subject, relation);
    		if(object == null) return null;
    		else return getValue2(object, context, binding);
    	} catch (DatabaseException e) {
    		return null;
    	}
    }
    
    @Override
    public Type getRelatedValueType(Resource subject, Resource relation) throws DatabaseException {
        Layer0 L0 = processor.getL0(this);
        Resource property = getSingleObject(subject, relation);
        String typeText = (String)getRelatedValue(property, L0.HasValueType, Bindings.STRING);
        try {
            return org.simantics.scl.compiler.types.Types.parseType(typeText);
        } catch (SCLTypeParseException e) {
            throw new DatabaseException(e);
        }
    }

    private static class GraphHintsImpl implements GraphHints {
        
        final private ArrayMap<String,Object> map;
                
        public GraphHintsImpl(String key, Object value) {
            map = ArrayMap.make(new String[] { key }, value);
        }

        public GraphHintsImpl(GraphHintsImpl base, String key, Object value) {
            if(base.map.containsKey(key)) {
                map = ArrayMap.make(base.map.getKeys(), base.map.getValuesCopy());
                map.put(key, value);
            } else {
                String[] newKeys = base.map.getKeysWithSpace(1);
                newKeys[newKeys.length-1] = key;
                Object[] newValues = base.map.getValuesCopyWithSpace(1);
                newValues[newValues.length-1] = value;
                map = new ArrayMap<>(newKeys, newValues);
            }
        }
        
        @Override
        public <T> T get(String key) {
            return (T)map.get(key);
        }

        @Override
        public GraphHints withValue(String key, Object value) {
            Object current = get(key);
            if(value.equals(current))
                return this;
            return new GraphHintsImpl(this, key, value);
        }
        
    }
    
    private static GraphHints BASE_GRAPH_HINTS = new GraphHintsImpl(GRAPH_HINT_SYNCHRONOUS, true);
    
    private static ThreadLocal<GraphHints> syncGraph = new ThreadLocal<GraphHints>() {
        protected GraphHints initialValue() {
            return BASE_GRAPH_HINTS;
        }
    };
    
    public GraphHints getHints() {
        return syncGraph.get();
    }

    @Override
    public GraphHints setHints(GraphHints hints) {
        GraphHints old = getHints();
        if(hints != old)
            syncGraph.set(hints);
        return old;
    }

    @Override
    public <T> T getHintValue(String key) {
        return getHints().get(key);
    }
    
    @Override
    public <T> GraphHints setHintValue(String key, T value) {
        GraphHints old = getHints();
        GraphHints newHints = old.withValue(key, value);
        if(newHints != old)
            syncGraph.set(newHints);
        return old;
    }

    @Override
    public boolean getSynchronous() {
        return getHints().get(GRAPH_HINT_SYNCHRONOUS);
    }
    
    @Override
    public boolean setSynchronous(boolean value) {
        boolean old = getSynchronous();
        setHintValue(GRAPH_HINT_SYNCHRONOUS, value);
        return old;
    }
    
    public void ensureLoaded(int resource) {
    	processor.querySupport.ensureLoaded(this, resource);
    }
    
    public void ensureLoaded(int resource, int predicate) {
    	processor.querySupport.ensureLoaded(this, resource, predicate);
    }

    public byte[] getValue(int resource) throws DatabaseException {
    	return processor.querySupport.getValue(this, resource);
    }
    
    public int thread(int resource) {
    	return (resource >>> 16) & processor.THREAD_MASK;
    }

    public int thread(Resource resource) {
    	return (((ResourceImpl)resource).id >>> 16) & processor.THREAD_MASK;
    }
    
    public ResourceSupport getResourceSupport() {
    	return processor.getResourceSupport();
    }
    
    @Override
    public Object getModificationCounter() {
    	return processor.getSession().getModificationCounter();
    }

    @Override
    public boolean performPending() {
        return processor.performPending(this);
    }
    
    public Set<ReadGraphImpl> ancestorSet() {
        HashSet<ReadGraphImpl> result = new HashSet<>();
        ReadGraphImpl g = this;
        while(g != null) {
            result.add(g);
            g = g.parentGraph;
        }
        return result;
    }
    
    public int getLevel() {
        return getLevelStatic(this);
    }
    
    private static int getLevelStatic(ReadGraphImpl impl) {
        if(impl == null) return 0;
        else return 1 + getLevelStatic(impl.parentGraph);
    }
    
    public boolean isParent(ReadGraphImpl impl) {
        if(impl == null) return false;
        if(this == impl) return true;
        return isParent(impl.parentGraph);
    }
    
    public ReadGraphImpl getTopLevelGraph() {
        return getTopLevelGraphStatic(this);
    }

    private static ReadGraphImpl getTopLevelGraphStatic(ReadGraphImpl impl) {
        if(impl.parentGraph == null) return impl;
        else return getTopLevelGraphStatic(impl.parentGraph);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T l0() {
        return (T) processor.getL0();
    }

    /*
     * This method can be called to obtain a stable resource identifier for debugging
     */
    public String clusterAndIndex(int r) {
        if(r < 0) return "";
        return "" + processor.querySupport.getClusterId(r) + ":" + (r & 0xfff);
    }
    
}
