/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.procedure.guarded;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.common.procedure.adapter.AsyncProcedureSupport;
import org.simantics.db.procedure.AsyncMultiProcedure;

abstract public class GuardedAsyncMultiProcedure<T> implements AsyncMultiProcedure<T>, AsyncProcedureSupport {

    private final AsyncMultiProcedure<T> procedure;
    private final AtomicBoolean     onceGuard = new AtomicBoolean(false);
    private final AtomicInteger     resultCounter;

    public GuardedAsyncMultiProcedure(AsyncMultiProcedure<T> procedure) {
        this.procedure = procedure;
        this.resultCounter = new AtomicInteger(1);
    }

    @Override
    final public void exception(AsyncReadGraph graph, Throwable t) {
    	except(graph, t);
    }

    protected void except(AsyncReadGraph graph, Throwable t) {
        if (onceGuard.compareAndSet(false, true)) {
            procedure.exception(graph, t);
        }
    }

    protected void offer(AsyncReadGraph graph, T item) {
    	if (!onceGuard.get()) {
    		procedure.execute(graph, item);
    	}
    }

    protected void dec(AsyncReadGraph graph) {
        if (resultCounter.decrementAndGet() <= 0) {
            if (onceGuard.compareAndSet(false, true)) {
                procedure.finished(graph);
            }
        }
    }
    
    protected void inc() {
    	resultCounter.incrementAndGet();
    }

	@Override
	final public void finished(AsyncReadGraph graph) {
		dec(graph);
	}

}
