/*******************************************************************************
 * Copyright (c) 2010, 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - Implement java.io.Closeable
 *******************************************************************************/
package org.simantics.databoard.util.binary;

import java.io.Closeable;
import java.io.IOException;

import org.simantics.databoard.file.RandomAccessBinaryList;

/**
 * Random access binary is a interface to access a binary object (BLOB).
 * Binary object is random accessible and has a length.
 * It can be read and written, enlarged and shrunk at any position.
 * <p>
 * There is a pointer which moves on every read and write operation. It can 
 * moved with {@link #position()}. If bytes are written and the pointer is 
 * moved outside the blob or moves while writing, the blob is enlarged.
 * 
 * @see Blob Cascading implementation
 * @see BinaryFile RandomAccessFile implementation
 * @see RandomAccessBinaryList java.util.List implementation 
 * @author Toni Kalajainen
 */
public interface RandomAccessBinary extends Closeable, BinaryWriteable, BinaryReadable {
    
	/**
	 * Flush write buffer 
	 * 
	 * @throws IOException
	 */
	void flush() throws IOException;
	
	/**
	 * Reset internal cache. Before reseting unwritten changes are flushed to 
	 * disc.
	 */
	void reset() throws IOException;
	
	enum ByteSide { Left, Neither, Right }
    
	/**
	 * Remove bytes from the position of the current pointer. If the pointer is 
	 * outside the blob, an error is thrown. 
	 * 
	 * @param bytes
	 * @param side
	 * @throws IOException
	 */
	void removeBytes(long bytes, ByteSide side) throws IOException;
	
	/**
	 * Insert bytes at current pointer. If the pointer is outside the size of
	 * the blob, the blob is enlarged to fit the insertion. 
	 * 
	 * The operation does not move pointer.
	 * 
	 * @param bytes
	 * @param side
	 * @throws IOException
	 */
	void insertBytes(long bytes, ByteSide side) throws IOException;
	
	/**
	 * Set length of the binary object.
	 *   
	 * @param newLength
	 * @throws IOException
	 */
	void setLength(long newLength) throws IOException;
	
	/**
	 * Get the length of the binary object.
	 * 
	 * @return the length
	 * @throws IOException
	 */
	long length() throws IOException;
	
	/**
	 * Flush and close the blob. 
	 */
	void close() throws IOException;

	/**
	 * Check if binary is open.
	 * 
	 * @return true if file is open
	 */
	boolean isOpen();

	/**
	 * Set new cursor position.
	 * 
	 * The position can be set outside the binary object.
	 * 
	 * @param newPosition 
	 */
	void position(long newPosition) throws IOException;

	/**
	 * Get the position of the cursor
	 * 
	 * @return cursor 
	 */
	long position() throws IOException;

}

