/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.util.binary;

import java.io.IOException;
import java.nio.ByteBuffer;

/**
 * IWriteable implementation with ByteBuffer as backend
 * 
 * @author Toni Kalajainen (toni.kalajainen@vtt.fi)
 */
public class ByteBufferWriteable implements BinaryWriteable {

	ByteBuffer buf;
	
	public ByteBufferWriteable(ByteBuffer buf)
	{
		if (buf == null)
			throw new IllegalArgumentException("null");
		this.buf = buf;
	}

	@Override
	public void write(int b) {
		buf.put((byte) b);
	}

	@Override
	public void writeByte(int b) throws IOException {
		buf.put((byte) b);
	}
	
	@Override
	public void writeBoolean(boolean v) throws IOException {
		buf.put( (byte) (v ? 1 : 0) );
	}
	
	@Override
	public void writeFully(ByteBuffer src) {
		buf.put(src);
	}

	@Override
	public void writeFully(ByteBuffer src, int length) {
		if (src.hasArray()) {
			byte[] array = src.array();
			buf.put(array, src.arrayOffset() + src.position(), length);
		} else {
			for (int i=0; i<length; i++)
				buf.put(src.get());
		}
	}

	@Override
	public void write(byte[] src, int offset, int length) {
		buf.put(src, offset, length);
	}

	@Override
	public void write(byte[] src) {
		buf.put(src);
	}

	@Override
	public void writeDouble(double value) {
		buf.putDouble(value);
	}

	@Override
	public void writeFloat(float value) {
		buf.putFloat(value);
	}

	@Override
	public void writeInt(int value) {
		buf.putInt(value);
	}

	@Override
	public void writeLong(long value) {
		buf.putLong(value);
	}

	@Override
	public void writeShort(int value) {
		buf.putShort((short)value);
	}
	
	@Override
	public void writeChar(int value) {
		buf.putChar((char) value);
	}	

	@Override
	public void writeBytes(String s) throws IOException {
		int len = s.length();
		for (int i = 0 ; i < len ; i++) {
		    buf.put((byte)s.charAt(i));
		}
	}
	
	@Override
	public void writeChars(String s) throws IOException {
        int len = s.length();
        for (int i = 0 ; i < len ; i++) {
            char v = s.charAt(i);
            buf.putChar(v);
        }
	}
	
	@Override
	public void writeUTF(String s) throws IOException {
		int len = UTF8.getModifiedUTF8EncodingByteLength(s);
		writeShort(len);
		UTF8.writeModifiedUTF(this, s);
	}

	@Override
	public void flush() {
	}
	
	public long position() throws IOException {
		return buf.position();
	}
	
	public void position(long newPosition) throws IOException {
		if (newPosition>=Integer.MAX_VALUE || newPosition<0) throw new IllegalArgumentException("index out of range");
		buf.position((int) newPosition);		
	}
	
}

