/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.SerializerScheme;
import org.simantics.databoard.type.Datatype;

/**
 * A version of VariantSerializer that uses Bindings.getMutableBinding instead
 * of Bindings.getBinding for deserialization.
 * 
 * @author Tuukka Lehtonen
 */
public class MutableVariantSerializer extends VariantSerializer {

	public MutableVariantSerializer(VariantBinding binding, SerializerScheme scheme) {
		super(binding, scheme);
	}

	@Override
	public Object deserialize(DataInput in, List<Object> identities) throws IOException {
		try {
			List<Object> typeIdentities = new ArrayList<Object>(1);
			Datatype type = (Datatype) datatypeSerializer.deserialize(in, typeIdentities);
			Binding valueBinding = Bindings.getMutableBinding(type);
			Serializer valueSerializer = scheme.getSerializerUnchecked(valueBinding);
			assertRemainingBytes(in, valueSerializer.getMinSize());
			Object value = valueSerializer.deserialize(in, identities);
			return binding.create(valueBinding, value);
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}

	@Override
	public void deserializeTo(DataInput in, List<Object> identities, Object obj) throws IOException {
		try {
			List<Object> typeIdentities = new ArrayList<Object>(1);
			Datatype type = (Datatype) datatypeSerializer.deserialize(in, typeIdentities);
			Datatype oldType = binding.getContentType(obj);

			if (type.equals(oldType)) {
				Binding valueBinding = binding.getContentBinding(obj);
				Serializer valueSerializer = scheme.getSerializerUnchecked(valueBinding);
				Object component = binding.getContent(obj);
//				assertRemainingBytes(in, valueSerializer.getMinSize());
				component = valueSerializer.deserializeToTry(in, identities, component);
				binding.setContent(obj, valueBinding, component);
			} else {
				Binding valueBinding = Bindings.getMutableBinding(type);
				Serializer valueSerializer = scheme.getSerializerUnchecked(valueBinding);
//				assertRemainingBytes(in, valueSerializer.getMinSize());
				Object component = valueSerializer.deserialize(in, identities);
				binding.setContent(obj, valueBinding, component);
			}
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}

	@Override
	public void skip(DataInput in, List<Object> identities) throws IOException {
		List<Object> typeIdentities = new ArrayList<Object>(1);
		Datatype type = (Datatype) datatypeSerializer.deserialize(in, typeIdentities);
		Binding valueBinding = Bindings.getMutableBinding(type);
		Serializer valueSerializer = scheme.getSerializerUnchecked(valueBinding);
		valueSerializer.skip(in, identities);
	}

}