package org.simantics.databoard.binding.impl;

import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.primitives.UnsignedInteger;
import org.simantics.databoard.type.IntegerType;

/**
 * Binding of {@link UnsignedInteger} to integer type.
 * This is bitwise binding, i.e. negative values bound to 0x100000000 + value.
 * 
 * @author Toni Kalajainen <toni.kalajainen@iki.fi>
 */
public abstract class UnsignedIntegerBinding extends IntegerBinding {
	
	UnsignedIntegerBinding(IntegerType type) {
		super(type);
	}

	public static class Immutable extends UnsignedIntegerBinding {

		public Immutable(IntegerType type) {
			super(type);
		}
		
		@Override
		public Object create(int value) throws BindingException {
			return UnsignedInteger.fromBits(value);
		}

		@Override
		public Object create(Integer value) throws BindingException {
			try {
				return UnsignedInteger.valueOf(value.intValue());
			} catch (java.lang.IllegalArgumentException e) {
				throw new BindingException( e );
			}
		}

		@Override
		public Object create(Number value) throws BindingException {
			try { 
				return UnsignedInteger.valueOf(value.longValue());
			} catch (java.lang.IllegalArgumentException e) {
				throw new BindingException( e );
			}
		}

		@Override
		public Object create(String value) throws BindingException {
			try {
				return UnsignedInteger.valueOf( Long.valueOf(value) );
			} catch (java.lang.IllegalArgumentException e) {
				throw new BindingException( e );
			}
		}		
		
		@Override
		public void setValue(Object obj, Number value) throws BindingException {
			throw new BindingException("UnsignedInteger is immutable class");
		}

		@Override
		public void setValue(Object obj, int value) throws BindingException {
			throw new BindingException("UnsignedInteger is immutable class");
		}
		
		@Override
		public boolean isImmutable() {
			return true;
		}
				
		@Override
		public boolean isInstance(Object obj) {
			return obj instanceof UnsignedInteger.Immutable;
		}		
	}
	
	public static class Mutable extends UnsignedIntegerBinding {

		public Mutable(IntegerType type) {
			super(type);
		}
		
		@Override
		public Object create(int value) throws BindingException {
			return new UnsignedInteger.Mutable(value);
		}

		@Override
		public Object create(Integer value) throws BindingException {
			return new UnsignedInteger.Mutable(value);
		}

		@Override
		public Object create(Number value) throws BindingException {
			return UnsignedInteger.Mutable.fromBits(value.intValue());
		}

		@Override
		public Object create(String value) throws BindingException {
			return new UnsignedInteger.Mutable(value);
		}

		@Override
		public void setValue(Object obj, Number value) throws BindingException {
			UnsignedInteger.Mutable mi = (UnsignedInteger.Mutable) obj;
			mi.setBits(value.intValue());
		}

		@Override
		public void setValue(Object obj, int value) throws BindingException {
			UnsignedInteger.Mutable mi = (UnsignedInteger.Mutable) obj;
			mi.setBits(value);
		}

		@Override
		public boolean isImmutable() {
			return false;
		}

		@Override
		public boolean isInstance(Object obj) {
			return obj instanceof UnsignedInteger.Mutable;
		}
		
	}

	@Override
	public Integer getValue(Object obj) throws BindingException {
		return ((UnsignedInteger)obj).toBits();
	}

	@Override
	public int getValue_(Object obj) throws BindingException {
		return ((UnsignedInteger)obj).toBits();
	}

	@Override
	public String toString(Object value) throws BindingException {
		return value.toString();
	}


}
