/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.Adapter;
import org.simantics.databoard.adapter.AdapterConstructionException;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.MapType;

/**
 * Binds java.util.TreeMap to MapType
 * 
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
@SuppressWarnings("all")
public class TreeMapBinding extends MapBinding {

	public TreeMapBinding(Binding keyBinding, Binding valueBinding) {
		super(keyBinding, valueBinding);
	}

	public TreeMapBinding(MapType mapType, Binding keyBinding,
			Binding valueBinding) {
		super(mapType, keyBinding, valueBinding);
	}

	public void postConstruction() {
	}

	@Override
	public Object create() {
		return new TreeMap( getKeyBinding() );
	}

	@Override
	public Object create(Object[] keys, Object[] values) {
		if (keys.length != values.length)
			throw new IllegalArgumentException("Equal length arrays expected");

		int len = keys.length;
		TreeMap result = new TreeMap( getKeyBinding() );

		for (int i = 0; i < len; i++) {
			Object key = keys[i];
			Object value = values[i];
			result.put(key, value);
		}

		return result;
	}
	
	
	@Override
	public Object create(List<Object> keys, List<Object> values) {
		if (keys.size()!=values.size())
			throw new IllegalArgumentException("Equal length arrays expected");
		
		int len = keys.size();
		TreeMap result = new TreeMap( getKeyBinding() );
		
		for (int i=0; i<len; i++) {
			Object key = keys.get(i);
			Object value = values.get(i);
			result.put(key, value);
		}
		
		return result;
	}	

	@Override
	public Object create(Map initialMap) {
		// Replace with TreeMap. Create comparator from binding.
		TreeMap result = new TreeMap( getKeyBinding() );
		result.putAll(initialMap);
		return result;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public void clear(Object map) {
		((Map) map).clear();
	}

	@SuppressWarnings("unchecked")
	@Override
	public boolean containsKey(Object map, Object key) {
		Map m = ((Map) map);
		return m.containsKey(key);
	}

	@SuppressWarnings("unchecked")
	@Override
	public boolean containsValue(Object map, Object value) {
		Map m = ((Map) map);
		Binding vb = getValueBinding();
		for (Object v : m.values())
		{
			if (vb.equals(v, value)) return true;
		}
		return false;
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object get(Object map, Object key) {
		Map m = ((Map) map);
		return m.get(key);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object[] getKeys(Object map) {
		Map m = ((Map) map);
		return m.keySet().toArray(new Object[m.size()]);
	}
	
	@Override
	public void getKeys(Object map, Set<Object> keys) throws BindingException {
		Map m = ((Map)map);
		keys.addAll(m.keySet());
	}	
	
	/**
	 * Count the number of entries between two keyes
	 * @param from
     * @param fromInclusive
	 * @param end 
     * @param endInclusive
	 * @throws BindingException
	 */
	@SuppressWarnings("unchecked")
	@Override
	public int count(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive) throws BindingException {
		// Assert end > from
		if (keyBinding.compare(from, end)>0) return 0;
		
		TreeMap m = (TreeMap) src;
		Map sm = m.subMap(from, fromInclusive, end, endInclusive);
		return sm.size();
	}
	
	/**
	 * Read a range of entries
	 * 
	 * @param src
	 * @param from
     * @param fromInclusive
	 * @param end 
     * @param endInclusive
	 * @param dstKeyArrayBinding
	 * @param dstKeyArray
     * @param dstValueArrayBinding
	 * @param dstValueArray
	 * @throws BindingException
	 */
	public int getEntries(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive, ArrayBinding dstKeyArrayBinding, Object dstKeyArray, ArrayBinding dstValueArrayBinding, Object dstValueArray, int limit) throws BindingException {
		try {
			// Assert end > from
			if (keyBinding.compare(from, end)>0) return 0;

			TreeMap m = (TreeMap) src;
			Map sm = m.subMap(from, fromInclusive, end, endInclusive);
			int dkc = dstKeyArrayBinding.size( dstKeyArray );
			int dvc = dstValueArrayBinding.size( dstValueArray );
			Adapter ka = Bindings.getTypeAdapter(keyBinding, dstKeyArrayBinding.getComponentBinding());
			Adapter va = Bindings.getTypeAdapter(valueBinding, dstValueArrayBinding.getComponentBinding());
			int i = 0;
			Set<Map.Entry<Object, Object>> es = sm.entrySet();
			for (Entry<Object, Object> e : es) {
				if (limit>=0 && i>=limit) break;
				Object dk = ka.adapt( e.getKey() );
				Object dv = va.adapt( e.getValue() );
				if (i<dkc) dstKeyArrayBinding.set(dstKeyArray, i, dk); else dstKeyArrayBinding.add(dstKeyArray, dk);
				if (i<dvc) dstValueArrayBinding.set(dstValueArray, i, dv); else dstValueArrayBinding.add(dstValueArray, dv);
				i++;
			}
			return i;
		} catch (AdapterConstructionException e) {
			throw new BindingException(e);
		} catch (AdaptException e) {
			throw new BindingException(e);
		}
	}
	

	@SuppressWarnings("unchecked")
	@Override
	public Object[] getValues(Object map) {
		Map m = ((Map) map);
		return m.values().toArray(new Object[m.size()]);
	}

	@Override
	public <K, V> void put(Object map, K key, V value) {
		Map<K, V> m = ((Map<K, V>) map);
		m.put(key, value);
	}

	@Override
	public <K, V> void putAll(Object dstMap, Map<K, V> srcMap) {
		Map<K, V> dst = ((Map<K, V>) dstMap);
		dst.putAll(srcMap);
	}

	@Override
	public void getAll(Object mapFrom, Map to) {
		Map<?, ?> m = ((Map<?, ?>) mapFrom);
		to.putAll(m);
	}

	@SuppressWarnings("unchecked")
	@Override
	public void getAll(Object mapFrom, Object[] keys, Object[] values) {
		TreeMap m = (TreeMap) mapFrom;
		int i = 0;
		for (Entry<Object, Object> e : (Set<Entry<Object, Object>>) m.entrySet()) {
			keys[i] = e.getKey();
			values[i] = e.getValue();
			i++;
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public Object remove(Object map, Object key) {
		Map m = ((Map) map);
		return m.remove(key);
	}

	@SuppressWarnings("unchecked")
	@Override
	public int size(Object map) {
		Map m = ((Map) map);
		return m.size();
	}

	@SuppressWarnings("unchecked")
	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof Map;
	}

	@Override
    public int deepHashValue(Object map, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
		int result = 0;
		Map m = ((Map) map);
		Set<Entry> s = m.entrySet();
		for (Entry e : s) {
			int keyTree = getKeyBinding().deepHashValue(e.getKey(), hashedObjects);
			int valueTree = getValueBinding().deepHashValue(e.getValue(), hashedObjects);
			result += (keyTree ^ valueTree);
		}
		return result;
	}

	@Override
	public Object getCeilingKey(Object map, Object key) {
		TreeMap m = ((TreeMap) map);
		return m.ceilingKey(key);
	}

	@Override
	public Object getFirstKey(Object map) {
		TreeMap m = ((TreeMap) map);
		return m.firstKey();
	}

	@Override
	public Object getFloorKey(Object map, Object key) {
		TreeMap m = ((TreeMap) map);
		return m.floorKey(key);
	}

	@Override
	public Object getHigherKey(Object map, Object key) {
		TreeMap m = ((TreeMap) map);
		return m.higherKey(key);
	}

	@Override
	public Object getLastKey(Object map) {
		TreeMap m = ((TreeMap) map);
		return m.lastKey();
	}

	@Override
	public Object getLowerKey(Object map, Object key) {
		TreeMap m = ((TreeMap) map);
		return m.lowerKey(key);
	}

}
