/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.LongBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.IdentityPair;

/**
 * Binds Long[] type to long[] class.
 */
public final class LongArrayBinding extends ArrayBinding {
	
	public static LongArrayBinding createFrom(ArrayType type)
	{
		return new LongArrayBinding(type, new LongBindingDefault((LongType) type.componentType));
	}

	public LongArrayBinding(ArrayType type, Binding componentBinding)
	{
		super(type, componentBinding);
	}

	@Override
	public Object create() {
		return new long[0];
	}
	
	@Override
	public Object create(int length, Iterator<Object> values) {
		long[] array = new long[length];
		for(int i=0;i<length;++i)
			array[i] = (Long)values.next();
		return array;
	}
	
	@Override
    public Object createDefault()
    throws BindingException
    {
		NumberType nt = (NumberType) type().componentType;
		if (nt.getRange() == null) {
	    	return new long[ type().minLength() ];
		}
		return super.createDefault();
    }	
	
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		LongBinding scb = (LongBinding) sb.getComponentBinding();
		
		long[] d = (long[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("long[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
	}	
	
	@Override
	public Object readFromTry(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		LongBinding scb = (LongBinding) sb.getComponentBinding();
		
		long[] d = (long[]) dst;
		int srcSize = sb.size(src);
		if (d.length != srcSize) d = new long[ srcSize ];
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
		return d;
	}		
	
	/**
	 * Create an array object.
	 * Note! The argument is consumed (is used in the result).
	 * 
	 * @param array
	 * @return an object that contains the array
	 */
	public Object create(long[] array) {
		return array.clone();
	}
	
	public Object create(Object[] values) {
		long[] result = new long[values.length];
		for (int i=0; i<values.length; i++)
			result[i] = (Long) values[i];
		return result;
	}	
	

	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", long[] expected");
		return ((long[])array)[index];
	}
	
	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		long[] a = (long[])array;
		for (int i=0; i<a.length; i++)
			result[i] = a[i];
	}
	
	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		long[] a = (long[])array;
		a[index] = (Long) value;
	}

	@Override
	public int size(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", long[] expected");
		return ((long[])array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof long[];
	}

	@Override
	public boolean isImmutable() {
		return false;
	}
	
	public long[] getArray(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", long[] expected");
		return (long[]) array;
	}	

    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	long[] array = (long[]) value;
    	for (int i=0; i<array.length; i++) {
    		long element = array[i];
    		result = 31*result + (int) (element ^ (element >>> 32));
    	}
    	return result;
    }
    
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	long[] a1 = (long[]) o1;
    	long[] a2 = (long[]) o2;
		// Compare Lengths
		int l1 = a1.length;
		int l2 = a2.length;
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		for (int i=0; i<l1; i++) {
			long v1 = a1[i];
			long v2 = a2[i];
			dif = (v1<v2 ? -1 : (v1==v2 ? 0 : 1));
			if (dif!=0) return dif;
		}
		return 0;
    }    

    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
    	throw new UnsupportedOperationException();
    }
    
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
    	throw new UnsupportedOperationException();
	}
	
	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("long[] is length immutable");
	}	
	
	@Override
	public boolean isResizable() {
		return false;
	}
	
}
