/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.IdentityPair;

/**
 * CollectionListBinding binds ArrayType to java.util.LinkedList
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class LinkedListBinding extends ArrayBinding {

	public LinkedListBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
		if (type==null) throw new IllegalArgumentException("null arg");
		this.type = type;
	}
	
	@Override
	public Object create() {
		return new LinkedList<Object>();
	}
	
	/**
	 * Create new ArrayList
	 */
	@Override
	public Object create(int length, Iterator<Object> values) {
		LinkedList<Object> result = new LinkedList<Object>(); 
		while (values.hasNext()) result.add(values.next());		
		return result;
	}
	
    public Object create(Collection<Object> collection)
    throws BindingException {
    	return new LinkedList<Object>( collection );
    }	

	public Object create(Object[] values) {
		LinkedList<Object> array = new LinkedList<Object>();
		for (int i=0; i<values.length; i++)
			array.add(values[i]);
		return array;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", java.util.List expected");
		List<Object> list = (List<Object>) array;
		return list.get(index);
	}
	
	@SuppressWarnings("unchecked")
    @Override
	public void getAll(Object array, Object[] result) throws BindingException {
		List<Object> list = (List<Object>) array;
		int index = 0;
		for (Object o : list) {
			result[index++] = o;
		}
	}
	
	@SuppressWarnings("unchecked")
    @Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		List<Object> list = (List<Object>) array;
		list.set(index, value);
	}

	@SuppressWarnings("unchecked")
	@Override
	public int size(Object array) throws BindingException {		
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", java.util.List expected");		
		List<Object> list = (List<Object>) array;
		return list.size();
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof LinkedList<?>;
	}	
	
    @SuppressWarnings("unchecked")
    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
		List<Object> list = (List<Object>) array;
		list.add(index, element);
    }	
	
	@SuppressWarnings("unchecked")
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
		LinkedList<Object> list = (LinkedList<Object>) array;
		if (index<0 || index>=list.size()) throw new IndexOutOfBoundsException();
		if (index==0) {
			list.removeFirst();
			return;
		}
		if (index==list.size()) {
			list.removeLast();
			return;
		}
		Iterator<Object> iter = list.iterator();
		for (int i=0; i<index; i++)
			iter.next();
		for (int i=0; i<count; i++) {
			iter.next();
			iter.remove();
		}
	}
	
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	LinkedList<?> list = (LinkedList<?>) value;
    	Iterator<?> iter = list.iterator();
    	while (iter.hasNext()) {
    		Object element = iter.next();
    		result = 31*result + componentBinding.deepHashValue(element, hashedObjects);
    	}
    	return result;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
		// Compare Lengths
		int l1 = size(o1);
		int l2 = size(o2);
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		Binding c = getComponentBinding();
		Iterator<Object> i1 = ((LinkedList<Object>) o1).iterator();
		Iterator<Object> i2 = ((LinkedList<Object>) o2).iterator();
		while(i1.hasNext()) {
			Object e1 = i1.next();
			Object e2 = i2.next();			
			dif = c.deepCompare(e1, e2, compareHistory);
			if (dif!=0) return dif;
		}
		return 0;
    }    
    
	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		@SuppressWarnings("unchecked")
		List<Object> list = (List<Object>) array;
		int oldSize = list.size();
		if (oldSize==newSize) return;
		
		if (oldSize>newSize) {
			for (int i=oldSize-1; i<=newSize; i--)
				list.remove(i);
			return;
		} 
		
		int c = newSize - oldSize;
		for (int i=0; i<c; i++) {
			list.add( componentBinding.createDefault() );
		}
	}		
	
	@Override
	public boolean isImmutable() {
		return false;
	}

	@Override
	public boolean isResizable() {
		return true;
	}
	
}

