/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.IdentityHashMap;
import java.util.Set;

import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.impl.LongBindingDefault;
import org.simantics.databoard.binding.mutable.MutableLongBinding;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.util.IdentityPair;

/**
 * This is a binding of a Long Type and a Java Object.
 *
 * @see LongType
 * @see LongBindingDefault for java.lang.Long Binding
 * @see MutableLongBinding for MutableLong Binding
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class LongBinding extends NumberBinding {
	
	public LongBinding(LongType type) {
		this.type = type;
	}
	
	public abstract Object create(long value) throws BindingException;
	public abstract Object create(Long value) throws BindingException;
	public abstract Object create(Number value) throws BindingException;
	public abstract Object create(String value) throws BindingException;
	public abstract Long getValue(Object o) throws BindingException;
	public abstract long getValue_(Object o) throws BindingException; 
	public abstract void setValue(Object obj, Number value) throws BindingException;
	public abstract void setValue(Object obj, long value) throws BindingException;
	public abstract boolean isInstance(Object obj);
	
    @Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }
    	
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	
    	long v1 = getValue_(o1);
    	long v2 = getValue_(o2);
    	return (v1<v2 ? -1 : (v1==v2 ? 0 : 1));
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	long v = getValue_(value); 
    	return (int)(v ^ (v >>> 32));
    }

	public Object createUnchecked(long value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	public Object createUnchecked(Long value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	
	/**
	 * Get Data type
	 * 
	 * @return data type
	 */
	@Override
	public LongType type() {
		return (LongType) type;
	}
	
}
