/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.IdentityHashMap;
import java.util.Set;

import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.util.IdentityPair;


/**
 * Abstract class to bind Float Type and a java class.
 *
 * @see FloatType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class FloatBinding extends NumberBinding {
		
	public FloatBinding(FloatType type) {
		this.type = type;
	}
	
	@Override
	public FloatType type() {
		return (FloatType) type;
	}
	
	public abstract Object create(Float value) throws BindingException;
	public abstract Object create(float value) throws BindingException;
	public abstract Object create(Number value) throws BindingException;
	public abstract Object create(String value) throws BindingException;
	public abstract boolean isInstance(Object obj);
	public abstract Float getValue(Object o) throws BindingException; 
	public abstract float getValue_(Object o) throws BindingException;
	public abstract void setValue(Object obj, Number value) throws BindingException;
	public abstract void setValue(Object obj, float value) throws BindingException;
	
    @Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }
    	
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	
    	float v1 = getValue_(o1);
    	float v2 = getValue_(o2);
    	return Float.compare(v1, v2);    	
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	float v = getValue_(value);
    	return Float.floatToIntBits(v);
    }

	public Object createUnchecked(float value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	public Object createUnchecked(Float value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
    
}
