/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.java;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.Accessor;
import org.simantics.databoard.accessor.IntegerAccessor;
import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.accessor.error.ReferenceException;
import org.simantics.databoard.accessor.event.Event;
import org.simantics.databoard.accessor.event.ValueAssigned;
import org.simantics.databoard.accessor.impl.AccessorParams;
import org.simantics.databoard.accessor.impl.ListenerEntry;
import org.simantics.databoard.accessor.interestset.IntegerInterestSet;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.IntegerType;

public class JavaInteger extends JavaObject implements IntegerAccessor {
	
	public JavaInteger(Accessor parent, IntegerBinding binding, Object object, AccessorParams params) {
		super(parent, binding, object, params);
	}
	
	public IntegerBinding getBinding() {
		return (IntegerBinding) binding;
	}

	@Override
	public IntegerType type() {
		return getBinding().type();
	}

	@Override
	public int getValue() throws AccessorException {
		readLock();
		try {
			return getBinding().getValue_(object);
		} catch (BindingException e) {
			throw new AccessorException(e);
		} finally {
			readUnlock();
		}
	}
	
	@Override
	public void setValue(int newValue) throws AccessorException {
		if (newValue == getValue()) return;
		try {
			if (binding.isImmutable() && parent!=null && parent instanceof JavaArray) {
				JavaObject jo = (JavaObject) parent;
				ArrayBinding ab = (ArrayBinding) jo.binding;
				Object nv = getBinding().create(newValue);
				ab.set(jo.object, (Integer)keyInParent, nv);
				this.object = nv;
			} else if (binding.isImmutable() && parent!=null && parent instanceof JavaRecord) {
				JavaObject jo = (JavaObject) parent;
				RecordBinding rb = (RecordBinding) jo.binding;
				Object nv = getBinding().create(newValue);
				rb.setComponent(jo.object, (Integer)keyInParent, nv);
				this.object = nv;
			} else if (binding.isImmutable() && parent!=null && parent instanceof JavaVariant) {
				JavaObject jo = (JavaObject) parent;
				VariantBinding vb = (VariantBinding) jo.binding;
				Object nv = getBinding().create(newValue);
				vb.setContent(jo.object, getBinding(), nv );
				this.object = nv;
			} else {			
				// Set value
				getBinding().setValue(object, newValue);
			}
			
			// Notify
			ListenerEntry le = listeners;
			while (le!=null) {
				IntegerInterestSet is = le.getInterestSet();
				if (is.inNotifications()) {
					Event e = new ValueAssigned( Bindings.INTEGER, is.inValues() ? newValue : null );
					emitEvent(le, e);
				}
				le = le.next;
			}
			
		} catch (BindingException e) {
			throw new AccessorException(e);
		}
	}

	@Override
	public void setValue(Binding binding, Object newValue) throws AccessorException {
		try {
			int nv = ((IntegerBinding)binding).getValue_(newValue);
			setValue(nv);
		} catch (BindingException e) {
			throw new AccessorException(e);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T extends Accessor> T getComponent(ChildReference reference)
	throws AccessorConstructionException {
		if (reference==null) return (T) this;

		throw new ReferenceException(reference.getClass()+" is not a subreference of BooleanType");	
	}
	
	@Override
	Event applyLocal(Event e, boolean makeRollback) throws AccessorException {
		Event rollback = makeRollback ? new ValueAssigned( Bindings.INTEGER, getValue() ) : null; 		
		if (e instanceof ValueAssigned) {
			ValueAssigned va = (ValueAssigned) e;
			if (va.newValue == null) throw new AccessorException("Integer value expected, got null");			
			setValue(va.newValue.getBinding(), va.newValue.getValue());
			return rollback;
		} else {
			throw new AccessorException("Cannot apply "+e.getClass().getName()+" to Integer");
		}
	}
	
}

