/*******************************************************************************
 * Copyright (c) 2014, 2017 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #7313
 *******************************************************************************/
package org.simantics.charts.ui;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.charts.Activator;
import org.simantics.charts.internal.VariableUtils;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.adapter.DropActionFactory;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.VariableReference;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.PropertyVariables;
import org.simantics.modeling.PropertyVariablesImpl;
import org.simantics.modeling.utils.VariableReferences;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class SubscriptionDropActionFactory implements DropActionFactory {

    @Override
    public Runnable create(ReadGraph g, Object target, Object source, int operation) throws DatabaseException {
        //System.out.println("SUBSCRIPTION DROP: " + source + " -> " + target);

        final Resource subscription = ISelectionUtils.getSinglePossibleKey(target, SelectionHints.KEY_MAIN, Resource.class);
        if (subscription == null)
            return null;
        Resource targetModel = g.syncRequest(new PossibleModel(subscription));

        if(source instanceof RVI) {
            RVI rvi = (RVI)source;
            List<VariableReference> refs = Collections.singletonList(new VariableReference(rvi, VariableUtils.getDatatype(g, targetModel, rvi), null));
            return addSubscriptions(g, subscription, refs, Collections.<Resource>emptySet());
        }

        List<PropertyVariables> vars = ISelectionUtils.getPossibleKeys(source, SelectionHints.KEY_MAIN, PropertyVariables.class);
        if (!vars.isEmpty()) {
            // FIXME: this is a hack for indexed value support
            vars = PropertyVariablesImpl.resolve(g, vars);
            List<VariableReference> references = g.syncRequest(VariableReferences.toReferences(targetModel, vars));
            if (!references.isEmpty())
                return addSubscriptions(g, subscription, references, Collections.<Resource>emptySet());
        } else {
            List<Resource> srcs = ISelectionUtils.getPossibleKeys(source, SelectionHints.KEY_MAIN, Resource.class);
            ModelingResources MOD = ModelingResources.getInstance(g);
            Set<Resource> movedItems = new HashSet<>();
            for (Resource src : srcs) {
                if (g.isInstanceOf(src, MOD.Subscription_Item)) {
                    Resource model = g.syncRequest(new PossibleModel(src));
                    if (ObjectUtils.objectEquals(targetModel, model))
                        movedItems.add(src);
                }
            }
            if (!movedItems.isEmpty())
                return addSubscriptions(g, subscription, Collections.emptyList(), movedItems);
        }

        if (source instanceof String) {
            return handleStringDrop(g, subscription, targetModel, (String) source);
        }

        return null;
    }

    private Runnable handleStringDrop(ReadGraph graph, Resource subscription, Resource targetModel, String source) throws DatabaseException {
        try {
            List<VariableReference> refs = VariableUtils.getVariableReferencesFromString(graph, targetModel, source);
            return refs != null && !refs.isEmpty() ? addSubscriptions(graph, subscription, refs, Collections.emptySet()) : null;
        } catch (DatabaseException e) {
            Activator.getDefault().getLog().log(new Status(IStatus.WARNING, Activator.PLUGIN_ID, getClass().getSimpleName() + ": Unrecognized String input: " + source));
            return null;
        }
    }

	private Runnable addSubscriptions(ReadGraph graph, Resource subscription, List<VariableReference> references,
            Set<Resource> movedSubscriptionItems) throws DatabaseException {
        AddVariableToChartAction action = new AddVariableToChartAction(null, subscription, references).init(graph);
        return () -> {
            action.run();
            if(!movedSubscriptionItems.isEmpty()) {
                Simantics.getSession().asyncRequest(new WriteRequest() {
                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        Layer0 L0 = Layer0.getInstance(graph);
                        for (Resource item : movedSubscriptionItems) {
                            graph.deny(item, L0.PartOf);
                            graph.claim(subscription, L0.ConsistsOf, item);
                        }
                    }
                }, e -> {
                    if (e != null)
                        ErrorLogger.defaultLogError(e);
                });
            }
        };
    }

}
