/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.PasteHandlerAdapter;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.util.ClipboardUtils;
import org.simantics.db.layer0.util.PasteEventHandler;
import org.simantics.db.layer0.util.SimanticsClipboard;
import org.simantics.db.layer0.util.SimanticsClipboard.Representation;
import org.simantics.db.layer0.util.SimanticsKeys;
import org.simantics.utils.ui.dialogs.ShowError;

/**
 * @author Tuukka Lehtonen
 */
public class ChartGroupPasteHandler extends PasteHandlerAdapter {

    private Resource chartGroup;

    public ChartGroupPasteHandler(Resource chartGroup) {
        this.chartGroup = chartGroup;
    }

    @Override
    public Collection<Resource> pasteFromClipboard(WriteGraph graph, SimanticsClipboard clipboard, PasteEventHandler handler) throws DatabaseException {
        final List<Set<Representation>> copied = new ArrayList<>();
        final List<Resource> cut = new ArrayList<>();
        for (Set<Representation> object : clipboard.getContents()) {
            Collection<Resource> rs = ClipboardUtils.accept(object, SimanticsKeys.KEY_COPY_RESOURCES);
            if (rs != null) {
                copied.add(object);
            } else {
                rs = ClipboardUtils.accept(object, SimanticsKeys.KEY_CUT_RESOURCES);
                if (rs != null)
                    cut.addAll(rs);
            }
        }

        if (copied.isEmpty() && cut.isEmpty())
            return Collections.emptyList();

        return graph.syncRequest(new WriteResultRequest<Collection<Resource>>() {
            @Override
            public Collection<Resource> perform(WriteGraph graph) throws DatabaseException {

                ChartResource CHART = ChartResource.getInstance(graph);

                Resource expectedModel = graph.syncRequest( new PossibleModel(chartGroup) );
                if (!copied.isEmpty()) {
                    Set<Set<Representation>> sources = new HashSet<>();
                    for (Set<Representation> r : copied) {
                        Collection<Resource> srcs = ClipboardUtils.accept(r, SimanticsKeys.KEY_COPY_RESOURCES);
                        for(Resource src : srcs) {
                            if (graph.isInstanceOf(src, CHART.Chart)) {
                                Resource model = graph.syncRequest(new PossibleModel(src));
                                if (ObjectUtils.objectEquals(expectedModel, model)) {
                                    sources.add(r);
                                }
                            }
                        }
                    }
                    return ChartGroupDropActionFactory.copyChartsRequest(chartGroup, sources).perform(graph);
                } else if (!cut.isEmpty()) {
                    Set<Resource> sources = new HashSet<>();
                    for (Resource r : cut) {
                        if (graph.isInstanceOf(r, CHART.Chart)) {
                            Resource model = graph.syncRequest(new PossibleModel(r));
                            if (ObjectUtils.objectEquals(expectedModel, model))
                                sources.add(r);
                        }
                    }
                    String error = ChartGroupDropActionFactory.validateDrop(graph, chartGroup, sources);
                    if (error != null) {
                        ShowError.showError("Cut Failed", error, (Throwable) null);
                        return Collections.emptyList();
                    }
                    return ChartGroupDropActionFactory.moveChartsRequest(chartGroup, sources).perform(graph);
                }
                
                return Collections.emptyList();
                
            }
        });
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object getAdapter(Class adapter) {
        if (Resource.class == adapter)
            return chartGroup;
        return null;
    }

}
