/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.charts.query.AddChartItem;
import org.simantics.charts.query.ChartAndSubscriptionItemData;
import org.simantics.charts.query.ChartItemDescriptor;
import org.simantics.charts.query.NextChartItemIndexQuery;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.type.BooleanType;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableReference;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.subscription.NewSubscription;
import org.simantics.modeling.subscription.NewSubscriptionItem;
import org.simantics.modeling.subscription.SubscriptionsQuery;
import org.simantics.modeling.subscription.SubscriptionsQuery.SubscriptionsResult;
import org.simantics.trend.configuration.TrendItem.Renderer;
import org.simantics.utils.ui.dialogs.ShowError;

public class AddVariableToChartAction implements Runnable {

	Resource chart;
	Resource subscription;
	Resource model;
	
	String chartName;
	String subscriptionName;
	SubscriptionsResult subscriptions;
	List<VariableReference> refs;
	List<String> variableReferences; // String references
	
	public AddVariableToChartAction( Resource chart, Resource subscription, List<VariableReference> refs ) {
		this.refs = refs;
		this.chart = chart;
		this.subscription = subscription;
	}
	
	/**
	 * Reads necessary data from session.
	 * 
	 * @throws DatabaseException
	 */
	public AddVariableToChartAction init() throws DatabaseException 
	{
    	Simantics.getSession().sync( new ReadRequest() {
			@Override
			public void run(ReadGraph g) throws DatabaseException {
				init(g);
			}
		} );
    	return this;
	}

	/**
	 * Reads necessary data from session.
	 * 
	 * @throws DatabaseException
	 */
	public AddVariableToChartAction init( ReadGraph g ) throws DatabaseException 
	{
    	Layer0 L0 = Layer0.getInstance(g);
    	if(chart != null) {
    	    chartName = g.getPossibleRelatedValue(chart, L0.HasName, Bindings.STRING);
    	    model = g.syncRequest(new PossibleModel(chart));
    	}
    	else
    	    model = g.syncRequest(new PossibleModel(subscription));
        if (model == null) throw new DatabaseException("Model was not found");

		Variable configuration = Variables.getConfigurationContext(g, model);
        
		subscriptions = g.sync( new SubscriptionsQuery(model) );
		if(subscription != null)
		    subscriptionName = g.getPossibleRelatedValue(subscription, L0.HasLabel, Bindings.STRING);
		variableReferences = new ArrayList<String>();
		
		for (VariableReference var : refs) {
			if ( var.label != null ) {
				variableReferences.add( var.label );
			} else {
				variableReferences.add( var.getVariableId().toPossibleString(g, configuration) );
			}
		}

		return this;
	}
	
	/**
	 * Opens dialog, if clicked OK, writes to graph async.
	 * 
	 * Remember to run this in SWT Thread.
	 */
	public void runInSwt() {
				
	    final Display display = Display.getCurrent();
    	final Shell shell = display.getActiveShell();

        Simantics.getSession().markUndoPoint();

		for (int i=0; i<refs.size(); i++) {
			VariableReference var = refs.get(i);
			String varStrRef = variableReferences.get(i);

    		try {
    	    	ChartAndSubscriptionItemData data = new ChartAndSubscriptionItemData();
    	    	data.index = chart != null ? Simantics.sync(new NextChartItemIndexQuery(chart)) : -1;
    	    	data.mutableCollectionSettings = true;
    	    	data.chartName = chartName;
    	    	data.hasSubscriptionItem = true;
    	    	data.subscriptions = subscriptions.toNames();
    	    	data.subscription = subscriptionName;
    	    	data.binaryMode = var.getDatatype() instanceof BooleanType;
    	    	data.variableReference = varStrRef;    	    	
				ChartAndSubscriptionItemDialog d = new ChartAndSubscriptionItemDialog(shell, data, chart != null); 
				
		    	if ( d.open() == Dialog.OK ) {
		    		
		    		// (New) Subscription
		    		int j = subscriptions.names.indexOf( data.subscription );
		    		Resource subscription;
		    		if (j<0) {
		    			// Create new subscription
		    			NewSubscription ns = new NewSubscription(model, data.subscription);
						Simantics.getSession().sync( ns );
		    			subscription = ns.subscription;
		    		} else {
		    			subscription = subscriptions.resources.get(j);
		    		}	    		
		    		if ( subscription == null ) continue;
		    		
		    		
		    		// New Subscription Item
		    		NewSubscriptionItem ns = new NewSubscriptionItem(
		    				subscription,
		    				data.interval,
		    				data.deadband,
		    				data.gain,
		    				data.bias,
		    				data.unit,
		    				data.label,
		    				var.getItemId(),
		    				var.getVariableId(),
		    				var.getDatatype()
		    				);
		    		
		    		Simantics.getSession().sync( ns );
			    	if (ns.subscriptionItem == null) continue;
		    			    		
			    	// New Chart Item
			    	if(chart != null) {
    			    	ChartItemDescriptor cid = new ChartItemDescriptor();
    			    	cid.drawMode = data.drawmode;
    			    	cid.renderer = var.getDatatype() instanceof BooleanType ? Renderer.Binary : Renderer.Analog;
    			    	cid.scale = data.scale;
    			    	cid.min = data.min;
    			    	cid.max = data.max;
    			    	cid.subscriptionItem = ns.subscriptionItem;
    			    	cid.color = data.color;
    			    	cid.strokeWidth = data.strokeWidth;
    			    	AddChartItem ci = new AddChartItem(	chart, Collections.singletonList( cid )	);
    			    	Simantics.getSession().sync( ci );
			    	}
		    	}
			} catch (DatabaseException e) {
				ShowError.showError(e.getClass().getName(), e.getLocalizedMessage(), e);
			}
			
		}
				
	}
	
	/**
	 * Executes run in SWT Thread
	 */
	public void run() {
	    Display display = PlatformUI.getWorkbench().getDisplay();
		if ( display.getThread() == Thread.currentThread() ) {
			runInSwt();
		} else {
			display.asyncExec( new Runnable() {
				@Override
				public void run() {
					runInSwt();
				}} );
		}
	}
	
}

