/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui.gfx;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.util.Objects;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;

public class TextImageDescriptor extends ImageDescriptor {

	static final PaletteData _RGB = new PaletteData(0x00ff0000, 0x0000ff00, 0x000000ff);

	public String text;
	public int width, height;
	public String font;
	public int fontSize;
	public int style;
	public int rgb;

	private transient int hash;

	public TextImageDescriptor(String text, int width, int height, 
			String font, int fontSize, 
			int style, int rgb) {
		this.text = text;
		this.width = width;
		this.height = height;
		this.font = font;
		this.fontSize = fontSize;
		this.style = style;
		this.rgb = rgb;
				
		hash = text.hashCode() + 3*width + 5*height + 7*font.hashCode() + 11*fontSize + 13*style + 17*rgb;
	}

	@Override
	public ImageData getImageData(int zoom) {
		int w = width;
		int h = height;
		int fs = fontSize;
		if (zoom > 100) {
			float s = zoom / 100.0f;
			w = Math.round(width * s);
			h = Math.round(height * s);
			fs = Math.round(fontSize * s);
		}
		return getImageData(w, h, fs);
	}

	private ImageData getImageData(int width, int height, int fontSize) {
		ImageData id = new ImageData(width, height, 24, _RGB);

		BufferedImage bi = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		Graphics2D g = (Graphics2D) bi.getGraphics();
		g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
		Font f = new Font(font, style, fontSize);
		g.setFont( f );
		Color c = new Color(rgb);
		g.setColor( c );
		FontMetrics fm = g.getFontMetrics(f);
		Rectangle2D rect = fm.getStringBounds(text, g);
		g.drawString(text, (float) ((width-rect.getWidth())/2), (float) ( (height/2) + (height-rect.getHeight())/2) );
		g.dispose();

		Raster alpha = bi.getAlphaRaster();
		for (int x=0; x<width; x++) {
			for (int y=0; y<height; y++) {
				int a = alpha.getSample(x, y, 0);
				int rgb = bi.getRGB(x, y);
				id.setAlpha(x, y, a);
				id.setPixel(x, y, rgb);
			}
		}

		return id;
	}

	@Override
	public int hashCode() {
		return hash;
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof TextImageDescriptor==false) return false;
		TextImageDescriptor o = (TextImageDescriptor) obj;
		if ( !Objects.equals(text, o.text) ) return false;
		if ( !Objects.equals(font, o.font) ) return false;
		return width==o.width && height==o.height && fontSize==o.fontSize && style==o.style && rgb==o.rgb;
	}

}
