/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui.gfx;

//import org.eclipse.ui.IMemento;

/**
 * PixelFormat
 * @author Toni Kalajainen
 */
public final class PixelFormat {

    public static PixelFormat RGB24 = new PixelFormat(0xff, 0xff00, 0xff0000, 0);
    
    /** red mask */
    private final int redMask;
    /** green mask */
    private final int greenMask;
    /** blue mask */
    private final int blueMask;
    /** alpha mask */
    private final int alphaMask;
    
    private final int redHighBit;
    private final int greenHighBit;
    private final int blueHighBit;
    private final int alphaHighBit;

    private final int redLowBit;
    private final int greenLowBit;
    private final int blueLowBit;
    private final int alphaLowBit;
    
    // Highest bit
    private final int bitDepth;
    
    public PixelFormat(int redMask, int greenMask, int blueMask, int alphaMask) {
        this.redMask = redMask;
        this.greenMask = greenMask;
        this.blueMask = blueMask;
        this.alphaMask = alphaMask;

        this.redHighBit = findHighestBit(redMask);
        this.greenHighBit = findHighestBit(greenMask);
        this.blueHighBit = findHighestBit(blueMask);
        this.alphaHighBit = findHighestBit(alphaMask);

        this.redLowBit = findLowestBit(redMask);
        this.greenLowBit = findLowestBit(greenMask);
        this.blueLowBit = findLowestBit(blueMask);
        this.alphaLowBit = findLowestBit(alphaMask);
        
        this.bitDepth = findHighestBit(redMask|greenMask|blueMask|alphaMask)+1;        
    }
        
    public boolean hasAlpha() {
        return alphaMask != 0;
    }
    
    /**
     * Calculates mask's bit shift value
     *
     * @param mask color mask
     */
    static int calculateShiftFromMask(int mask) {
        return 7-findHighestBit(mask);
    }
    
    static int findHighestBit(int mask) {
        // Scan bits starting from left. Find first bit
        for (int i=31; i>=0; i--)
            // Check if bit i is 1
            if ((mask & (1<<i)) > 0) return i;
        return 0;
    }

    static int findLowestBit(int mask) {
        // Scan bits starting from left. Find first bit
        for (int i=0; i<32; i++)
            // Check if bit i is 1
            if ((mask & (1<<i)) > 0) return i;
        return 0;
    }

    public int getAlphaMask() {
        return alphaMask;
    }

    public int getAlphaShift() {
        return 7-alphaHighBit;
    }

    public int getAlphaDepth() {
        return alphaHighBit-alphaLowBit+1;
    }
    
    public int getAlphaHighBit() {
        return alphaHighBit;
    }
    
    public int getAlphaLowBit() {
        return alphaLowBit;
    }
    
    
    public int getBlueMask() {
        return blueMask;
    }

    public int getBlueShift() {
        return 7- blueHighBit;
    }

    public int getBlueDepth() {
        return blueHighBit-blueLowBit+1;
    }
    
    public int getBlueHighBit() {
        return blueHighBit;
    }
    
    public int getBlueLowBit() {
        return blueLowBit;
    }

    
    public int getGreenMask() {
        return greenMask;
    }

    public int getGreenShift() {
        return 7-greenHighBit;
    }

    public int getGreenDepth() {
        return greenHighBit-greenLowBit+1;
    }
    
    public int getGreenHighBit() {
        return greenHighBit;
    }
    
    public int getGreenLowBit() {
        return greenLowBit;
    }

    public int getRedMask() {
        return redMask;
    }
    
    public int getRedShift() {
        return 7-redHighBit;
    }

    public int getRedDepth() {
        return redHighBit-redLowBit+1;
    }
    
    public int getRedHighBit() {
        return redHighBit;
    }
    
    public int getRedLowBit() {
        return redLowBit;
    }
    
    public int getBitDepth() {
        return bitDepth; 
    }
    
    public int getNumberOfColorComponents() {
        int result = 0;
        if (redMask>0) result++;
        if (greenMask>0) result++;
        if (blueMask>0) result++;
        if (alphaMask>0) result++;
        return result;
    }
    
    
    /**
     * Return pixel size in bytes
     * @return pixel size in bytes
     */
    public int getPixelSize() {
        return (bitDepth+7)/8; 
    }
    
    @Override
    public int hashCode() {
        return (redLowBit)|(redHighBit<<4)|(greenLowBit<<8)|(greenHighBit<<12)|(blueLowBit<<16)|(blueHighBit<<20)|(alphaLowBit<<24)|(alphaHighBit<<28); 
    }
    
    @Override
    public boolean equals(Object o2) {
        if (!o2.getClass().equals(this.getClass())) return false;
        PixelFormat pf2 = (PixelFormat) o2;
        return (redMask == pf2.redMask) && (greenMask == pf2.greenMask) && (blueMask == pf2.blueMask) && (alphaMask == pf2.alphaMask);
    }    
    
    @Override
    public String toString() {
        return bitDepth+" bits";
    }
    
//    public void saveState(IMemento memento)
//    {   
//        memento.putInteger("redMask", redMask);
//        memento.putInteger("greenMask", greenMask);
//        memento.putInteger("blueMask", blueMask);
//        memento.putInteger("alphaMask", alphaMask);
//    }
//    
//    public static PixelFormat restoreState(IMemento memento)
//    {
//        return new PixelFormat(
//                memento.getInteger("redMask"),
//                memento.getInteger("greenMask"),
//                memento.getInteger("blueMask"),
//                memento.getInteger("alphaMask")
//                );
//    }
}
