/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * 27.12.2006
 */
package org.simantics.utils.ui.gfx;

import java.util.Collection;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.simantics.utils.datastructures.Array;

/**
 * CompositionImageDescriptor is image description that composes
 * multiple images into one image.
 * <p> 
 * 
 * @author Toni Kalajainen
 */
@SuppressWarnings("deprecation")
public class CompositionImageDescriptor extends ImageDescriptor {

    public static final PaletteData DEFAULT_PALETTEDATA = 
        new PaletteData(0x00ff0000, 0x0000ff00, 0x000000ff);
    
    public static ImageDescriptor compose(ImageDescriptor ... imageDescriptors) {
        if (imageDescriptors.length == 1)
            return imageDescriptors[0];
        return new CompositionImageDescriptor(imageDescriptors);
    }
    
    public static ImageDescriptor compose(Collection<ImageDescriptor> imageDescriptors) {
        int size = imageDescriptors.size();
        if (size == 1)
            return imageDescriptors.iterator().next();
        return new CompositionImageDescriptor(imageDescriptors.toArray(new ImageDescriptor[size]));
    }    
    
    ImageCache cache = ImageCache.getInstance();
    final Array<ImageDescriptor> ids;
    Point size;

    public CompositionImageDescriptor(ImageDescriptor[] ids) {
        this.ids = new Array<ImageDescriptor>(ids); 
        // Make sure that the argument is valid
        assert (this.ids.size() > 0);
        for (ImageDescriptor id : ids)
            assert (id != null);
    }
    
    public CompositionImageDescriptor(Array<ImageDescriptor> ids) {
        this.ids = ids;
        // Make sure that the argument is valid
        assert (this.ids.size() > 0);
        for (ImageDescriptor id : ids.toArray())
            assert (id != null);
    }
    
    @Override
    public ImageData getImageData() {
        ImageDescriptor [] _ids = ids.toArray();
        if (_ids.length==1)
            return cache.getImage(_ids[0]).getImageData();
            
        Point s = getSize();        
        PaletteData palette = DEFAULT_PALETTEDATA;
        ImageData id = new ImageData(s.x, s.y, 24, palette);
        id.setAlpha(0,0,0);
        for (int i=0; i<_ids.length; i++)
        {
            ImageData layer = ImageCache.getInstance().getImage(_ids[i]).getImageData();
            int width = Math.min(s.x, layer.width);
            int height = Math.min(s.y, layer.height);
            PaletteData layerPaletteData = layer.palette;
            if (layer.getTransparencyType()==SWT.TRANSPARENCY_MASK ||
                layer.getTransparencyType()==SWT.TRANSPARENCY_PIXEL)
            {
                ImageData mask = layer.getTransparencyMask();
                for (int y=0; y<height; y++)
                    for (int x=0; x<width; x++)
                        if (mask.getPixel(x, y)==1) {
                            RGB rgb = layerPaletteData.getRGB(layer.getPixel(x, y));
                            id.setPixel(x, y, palette.getPixel(rgb));
                            id.setAlpha(x, y, 255);
                        }
            } else {
                for (int y=0; y<height; y++)
                    for (int x=0; x<width; x++)
                    {                                                
                        int layerAlpha = layer.getAlpha(x, y);
                        int origAlpha = id.getAlpha(x,y);
                        RGB layerRGB = layerPaletteData.getRGB(layer.getPixel(x, y));
                        RGB origRGB = palette.getRGB(id.getPixel(x, y));
                        int newR = ( ( origRGB.red * (255-layerAlpha) ) +
                                ( layerRGB.red * (layerAlpha) ) ) / 255;
                        int newG = ( ( origRGB.green * (255-layerAlpha) ) +
                                ( layerRGB.green * (layerAlpha) ) ) / 255;
                        int newB = ( ( origRGB.blue * (255-layerAlpha) ) +
                                ( layerRGB.blue * (layerAlpha) ) ) / 255;                        
                        int newAlpha = origAlpha + ((255-origAlpha)*layerAlpha)/255;
                        id.setPixel(x, y, palette.getPixel(new RGB(newR, newG, newB)));
                        id.setAlpha(x, y, newAlpha);
                    }                
            }
                
        }
        //ImageData orig = ImageCache.getInstance().getImage(desc).getImageData();
        
        return id;
    }
    
    private Point _countSize()
    {
        int width = 0;
        int height = 0;
        for (ImageDescriptor id : ids.toArray())
        {
            Image i = cache.getImage(id);                
            int w = i.getImageData().width;
            if (w>width) width = w;
            int h = i.getImageData().height;
            if (h>height) height = h;
        }
        return new Point(width, height);
    }
    
    protected Point getSize() {
        if (size==null) size = _countSize(); 
        return size;
    }  
    
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof CompositionImageDescriptor))
            return false;
        CompositionImageDescriptor other = (CompositionImageDescriptor) obj;
        return other.ids.equals(ids);
    }
    
    @Override
    public int hashCode() {
        return ids.hashCode();
    }    

}
