/*
 * Decompiled with CFR 0.152.
 */
package org.simantics.utils.datastructures.file;

import gnu.trove.map.TObjectLongMap;
import gnu.trove.map.hash.TObjectLongHashMap;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.LongConsumer;
import org.simantics.databoard.util.BijectionMap;
import org.simantics.utils.FileUtils;
import org.simantics.utils.datastructures.MapSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectorySizeTracker
implements Runnable,
Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DirectorySizeTracker.class);
    private static final int TRACE_NONE = 0;
    private static final int TRACE_DIRS = 1;
    private static final int TRACE_EVENTS = 2;
    private static final int TRACE_FILES = 3;
    private static final WatchEvent.Kind<?>[] ALL_EVENTS = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY};
    private static AtomicInteger threadCounter = new AtomicInteger(1);
    private LongConsumer consumer;
    private WatchEvent.Kind<?>[] events;
    private WatchService watcher;
    private Object lock = new Object();
    private BijectionMap<WatchKey, Path> trackedDirs = new BijectionMap();
    private MapSet<Path, Path> subdirs = new MapSet.Hash<Path, Path>();
    private MapSet<Path, Path> files = new MapSet.Hash<Path, Path>();
    private TObjectLongMap<Path> entrySizes = new TObjectLongHashMap(1024, 0.5f, -1L);
    private TObjectLongMap<Path> dirSizes = new TObjectLongHashMap(512, 0.5f, -1L);
    private long totalSize = 0L;
    private int traceLevel = 0;
    private boolean running = true;
    private Thread thread;
    private FileVisitor<Path> REGISTER = new SimpleFileVisitor<Path>(){
        private Path currentDir;

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            this.currentDir = dir;
            if (DirectorySizeTracker.this.traceLevel > 2) {
                LOGGER.info("Set current dir to " + String.valueOf(this.currentDir));
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            DirectorySizeTracker.this.registerFile(this.currentDir, file, attrs);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            LOGGER.warn("Failed to visit file " + String.valueOf(file), (Throwable)exc);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            Path parent = dir.getParent();
            if (DirectorySizeTracker.this.trackDir(dir)) {
                DirectorySizeTracker.this.subdirs.add(parent, dir);
            }
            this.currentDir = parent;
            return FileVisitResult.CONTINUE;
        }
    };

    public static DirectorySizeTracker startTracker(LongConsumer sizeChangeListener) throws IOException {
        DirectorySizeTracker watcher = new DirectorySizeTracker(sizeChangeListener, ALL_EVENTS);
        watcher.thread = new Thread((Runnable)watcher, DirectorySizeTracker.class.getSimpleName() + "-" + threadCounter.get());
        watcher.thread.start();
        return watcher;
    }

    private DirectorySizeTracker(LongConsumer sizeChangeListener, WatchEvent.Kind<?>[] events) throws IOException {
        this.consumer = sizeChangeListener;
        this.events = events;
        this.watcher = FileSystems.getDefault().newWatchService();
    }

    public void close(boolean joinThread) throws InterruptedException {
        this.running = false;
        this.thread.interrupt();
        if (joinThread) {
            this.thread.join();
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.close(true);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void track(Path dir) throws IOException {
        Object object = this.lock;
        synchronized (object) {
            if (this.trackedDirs.containsRight((Object)dir)) {
                return;
            }
            if (this.traceLevel > 0) {
                LOGGER.info("Starting to track entire directory " + String.valueOf(dir) + " with a total of " + this.entrySizes.size() + " files and " + this.trackedDirs.size() + " dirs with a total size of " + this.totalSize);
            }
            Files.walkFileTree(dir, this.REGISTER);
            if (this.traceLevel > 0) {
                LOGGER.info("Now tracking entire directory " + String.valueOf(dir) + " with a total of " + this.entrySizes.size() + " files and " + this.trackedDirs.size() + " dirs with a total size of " + this.totalSize);
            }
        }
    }

    private boolean trackDir(Path dir) throws IOException {
        if (this.trackedDirs.containsRight((Object)dir)) {
            return false;
        }
        WatchKey key = dir.register(this.watcher, this.events);
        if (this.traceLevel > 1) {
            Path prev = (Path)this.trackedDirs.getRight((Object)key);
            if (prev == null) {
                LOGGER.info("Tracking new directory {}\n", (Object)dir);
            } else if (!dir.equals(prev)) {
                LOGGER.info("Tracked directory update: {} -> {}\n", (Object)prev, (Object)dir);
            }
        }
        this.trackedDirs.map((Object)key, (Object)dir);
        return true;
    }

    private boolean registerFile(Path dir, Path file, BasicFileAttributes attrs) {
        if (this.files.add(dir, file)) {
            long entrySize = attrs.size();
            this.entrySizes.put((Object)file, entrySize);
            this.dirSizes.adjustOrPutValue((Object)dir, entrySize, entrySize);
            this.totalSize += entrySize;
            if (this.traceLevel > 2) {
                LOGGER.info("Registered file " + String.valueOf(file) + " size " + entrySize + " for total size " + this.totalSize);
            }
            return true;
        }
        long size = attrs.size();
        long oldSize = this.entrySizes.put((Object)file, size);
        long sizeDelta = oldSize >= 0L ? size - oldSize : size;
        this.totalSize += sizeDelta;
        this.dirSizes.adjustOrPutValue((Object)dir, sizeDelta, sizeDelta);
        if (this.traceLevel > 2) {
            LOGGER.info("Modified " + String.valueOf(file) + " size from " + oldSize + " to " + size + " with delta " + sizeDelta + ", dir size = " + this.dirSizes.get((Object)dir) + ", total size = " + this.totalSize);
        }
        return false;
    }

    private boolean unregisterFile(Path dir, Path file) {
        long fileSize = this.entrySizes.remove((Object)file);
        if (fileSize >= 0L) {
            this.totalSize -= fileSize;
            if (this.files.remove(dir, file)) {
                this.dirSizes.adjustValue((Object)dir, -fileSize);
            }
            if (this.traceLevel > 2) {
                LOGGER.info("Unregistered file " + String.valueOf(file) + " of size " + fileSize + ", dirSize = " + this.dirSizes.get((Object)dir) + ", totalSize = " + this.totalSize);
            }
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void untrack(Path dir) {
        Object object = this.lock;
        synchronized (object) {
            if (!this.trackedDirs.containsRight((Object)dir)) {
                return;
            }
            if (this.traceLevel > 0) {
                LOGGER.info("Starting to untrack entire directory " + String.valueOf(dir) + " with total tracked size " + this.totalSize);
            }
            this.untrackTree(dir);
            this.subdirs.remove(dir.getParent(), dir);
            if (this.traceLevel > 0) {
                LOGGER.info("Done untracking entire directory " + String.valueOf(dir) + " with total tracked size " + this.totalSize);
            }
        }
    }

    private void untrackTree(Path dir) {
        Set<Path> subdirs = this.subdirs.removeValues(dir);
        for (Path subdir : subdirs) {
            this.untrackTree(subdir);
        }
        this.untrackDir(dir);
    }

    private void untrackDir(Path dir) {
        WatchKey key;
        if (this.traceLevel > 1) {
            LOGGER.info("Untrack directory " + String.valueOf(dir) + " with total tracked size " + this.totalSize);
        }
        if ((key = (WatchKey)this.trackedDirs.removeWithRight((Object)dir)) != null) {
            key.cancel();
        }
        Set<Path> registeredFiles = this.files.removeValues(dir);
        for (Path file : registeredFiles) {
            this.unregisterFile(dir, file);
        }
        this.dirSizes.remove((Object)dir);
    }

    static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return event;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processEvents() {
        while (this.running) {
            WatchKey key;
            try {
                key = this.watcher.take();
            }
            catch (InterruptedException interruptedException) {
                return;
            }
            Path dir = (Path)this.trackedDirs.getRight((Object)key);
            if (dir == null) {
                LOGGER.error("WatchKey not registered: " + String.valueOf(key));
                for (WatchEvent<?> event : key.pollEvents()) {
                    WatchEvent.Kind<?> kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                    WatchEvent evt = DirectorySizeTracker.cast(event);
                    Path name = (Path)evt.context();
                    LOGGER.error(String.format("MISSED EVENT: %s: %s", event.kind().name(), name));
                }
                boolean valid = key.reset();
                LOGGER.error("RESET KEY RETURNED: " + valid);
                continue;
            }
            Object object = this.lock;
            synchronized (object) {
                for (WatchEvent<?> event : key.pollEvents()) {
                    BasicFileAttributes attrs;
                    WatchEvent.Kind<?> kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                    WatchEvent evt = DirectorySizeTracker.cast(event);
                    Path name = (Path)evt.context();
                    Path child = dir.resolve(name);
                    if (this.traceLevel > 1) {
                        LOGGER.info(String.format("%s: %s", event.kind().name(), child));
                    }
                    if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                        try {
                            attrs = Files.readAttributes(child, BasicFileAttributes.class, new LinkOption[0]);
                            if (attrs.isDirectory()) {
                                this.track(child);
                                continue;
                            }
                            if (!attrs.isRegularFile()) continue;
                            this.registerFile(dir, child, attrs);
                        }
                        catch (NoSuchFileException noSuchFileException) {
                        }
                        catch (AccessDeniedException accessDeniedException) {
                        }
                        catch (IOException ioe) {
                            LOGGER.error("Failed to read attribute for path " + String.valueOf(child), (Throwable)ioe);
                        }
                        continue;
                    }
                    if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
                        try {
                            attrs = Files.readAttributes(child, BasicFileAttributes.class, new LinkOption[0]);
                            if (!attrs.isRegularFile()) continue;
                            this.registerFile(dir, child, attrs);
                        }
                        catch (NoSuchFileException noSuchFileException) {
                            if (this.entrySizes.containsKey((Object)child)) continue;
                            this.untrack(child);
                        }
                        catch (AccessDeniedException accessDeniedException) {
                        }
                        catch (IOException ioe) {
                            LOGGER.error("Failed to read attribute for path " + String.valueOf(child), (Throwable)ioe);
                        }
                        continue;
                    }
                    if (kind != StandardWatchEventKinds.ENTRY_DELETE || this.unregisterFile(dir, child)) continue;
                    this.untrack(child);
                }
                boolean valid = key.reset();
                if (!valid) {
                    if (this.traceLevel > 0) {
                        LOGGER.info("WatchKey for dir " + String.valueOf(dir) + " is no longer valid. Untracking it.");
                    }
                    this.untrack(dir);
                    if (this.trackedDirs.isEmpty()) {
                        break;
                    }
                }
            }
            if (this.traceLevel > 1) {
                LOGGER.info("STATUS: Tracking a total of " + this.entrySizes.size() + " files and " + this.trackedDirs.size() + " dirs with a total size of " + this.totalSize);
            }
            if (this.consumer == null) continue;
            this.consumer.accept(this.totalSize);
        }
    }

    @Override
    public void run() {
        try {
            while (this.running) {
                this.processEvents();
            }
        }
        finally {
            this.cancelWatchKeys();
            FileUtils.uncheckedClose((Closeable)this.watcher);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cancelWatchKeys() {
        Object object = this.lock;
        synchronized (object) {
            for (WatchKey key : this.trackedDirs.getLeftSet()) {
                key.cancel();
                key.pollEvents();
            }
        }
    }

    public long getTotalSize() {
        return this.totalSize;
    }
}

