/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures.persistent;

public abstract class ImmutableStack<T> {
	
	private ImmutableStack() {		
	}
	
	private static class SingleStackNode<T> extends ImmutableStack<T> {
		ImmutableStack<T> parent;
		T value;
		
		public SingleStackNode(ImmutableStack<T> parent, T value) {		
			this.parent = parent;
			this.value = value;
		}

		@Override
		public T get(int i) {
			return i==0 ? value : parent.get(i-1);
		}	
	}
	
	private static class MultiStackNode<T> extends ImmutableStack<T> {
		ImmutableStack<T> parent;
		T[] values;
		
		public MultiStackNode(ImmutableStack<T> parent, T[] values) {		
			this.parent = parent;
			this.values = values;
		}

		@Override
		public T get(int i) {
			return i<values.length ? values[i] : parent.get(i-values.length);
		}	
	}
	
	private static class EmptyStack<T> extends ImmutableStack<T> {

		@Override
		public T get(int i) {
			throw new IllegalArgumentException("No such element in stack.");
		}
		
	}
	
	@SuppressWarnings({ "rawtypes" })
	static final EmptyStack EMPTY = new EmptyStack();
	
	public ImmutableStack<T> push(T value) {
		return new SingleStackNode<T>(this, value);
	}
	
	public ImmutableStack<T> push(T[] values) {
		if(values.length > 1)
			return new MultiStackNode<T>(this, values);
		else if(values.length == 1)
			return new SingleStackNode<T>(this, values[0]);
		else
			return this;
	}
	
	public abstract T get(int i); 
	
	@SuppressWarnings("unchecked")
	public static <T> ImmutableStack<T> empty() {
		return (ImmutableStack<T>)EMPTY;
	}
	
	public static <T> ImmutableStack<T> of(T value) {
		return new SingleStackNode<T>(null, value);
	}
	
	@SuppressWarnings("unchecked")
	public static <T> ImmutableStack<T> of(T[] values) {
		if(values.length > 1)
			return new MultiStackNode<T>((ImmutableStack<T>)empty(), values);
		else if(values.length == 1)
			return new SingleStackNode<T>((ImmutableStack<T>)empty(), values[0]);
		else
			return empty();
	}
}
